import type {
    ReceiptMovementAddressed,
    ReceiptMovementAddressedBinding,
    StoringList,
} from '@sage/wh-input-api-partial';
import {
    SerialNumberManagement,
    SerialNumberManagementError,
} from '@sage/wh-input/lib/client-functions/serial-number-management';
import type { Location, SerialGroup } from '@sage/wh-master-data-api';
import { receipt } from '@sage/wh-master-data/build/lib/menu-items/receipt';
import { dialogConfirmation, dialogMessage } from '@sage/wh-master-data/lib/client-functions/dialogs';
import {
    AuthorizedOperation,
    getAuthorizedOperator,
} from '@sage/wh-master-data/lib/client-functions/get-authorized-operator';
import { getCurrentSiteDepositor } from '@sage/wh-master-data/lib/client-functions/get-selected-site-depositor';
import { MAX_STORE_CODE_LENGTH } from '@sage/wh-master-data/lib/shared-functions/constants';
import type { GraphApi } from '@sage/wh-pages-api';
import type { Product } from '@sage/wh-product-data-api';
import { getConsumptionUnit } from '@sage/wh-product-data/lib/client-functions/get-consumption-unit';
import { getProductConfiguration } from '@sage/wh-product-data/lib/client-functions/get-product-configuration';
import { ProductPalletizationPlan } from '@sage/wh-product-data/lib/client-functions/product-palletization-plan';
import { ConsumptionUnit, ContainerUnit, ProductConfiguration } from '@sage/wh-product-data/lib/interfaces';
import type { SerialNumber, SerialNumberBinding } from '@sage/wh-stock-data-api';
import { setApplicativePageCrudActions } from '@sage/wh-system/lib/client-functions/applicative-crud-actions';
import { ExtractEdges, extractEdges, Filter } from '@sage/xtrem-client';
import { SystemError } from '@sage/xtrem-shared';
import * as ui from '@sage/xtrem-ui';
import type { PartialCollectionValueWithIds } from '@sage/xtrem-ui/build/lib/component/types';

enum SelectionSection {
    disableAll = 'disableAll',
    main = 'main',
    validateMovement = 'validateMovement',
    declareSerialNumbers = 'declareSerialNumbers',
    readdressing = 'readdressing',
}

interface SectionParameters {
    isCreate?: boolean;
    isUpdate?: boolean;
    isDelete?: boolean;
    isEdit?: boolean;
    isSaved?: boolean;
    isReinitialize?: boolean;
}

// Same as import template used by the mutation
interface ImportStoringList {
    code: string;
    operatorCode: string;
    addressedReceiptMovements: ImportReceiptMovementAddressed[];
}

interface ImportReceiptMovementAddressed {
    code: string;
    store: string;
    location: string;
    controlCode: string;
    homogeneousContainer: string;
    homogeneousQuantity: number;
    serialNumbers: ImportSerialNumber[];
}
interface ImportSerialNumber {
    code: string;
    double: number;
    occurrence: number;
    serialGroup?: string;
    free1?: string;
    free2?: string;
    free3?: string;
    free4?: string;
    free5?: string;
    actionImport?: string;
}

const _serialNumberManagementError: Record<SerialNumberManagementError, string> = {
    [SerialNumberManagementError.quantityIsMandatory]: ui.localize(
        '@sage/wh-pages/pages__mobile_putaway__error__quantity_is_mandatory',
        'Quantity is mandatory',
    ),
    [SerialNumberManagementError.serialNumberAlreadySelected]: ui.localize(
        '@sage/wh-pages/pages__mobile_putaway__error__serial_number_already_selected',
        'Serial number is already selected',
    ),
    [SerialNumberManagementError.tooManySerialNumbers]: ui.localize(
        '@sage/wh-pages/pages__mobile_putaway__error__too_many_serial_numbers',
        'Too many serial numbers',
    ),
    [SerialNumberManagementError.noSerialNumberFound]: ui.localize(
        '@sage/wh-pages/pages__mobile_putaway__error__no_serial_number_found',
        'No serial number found',
    ),
    [SerialNumberManagementError.noSerialNumberForReturn]: ui.localize(
        '@sage/wh-pages/pages__mobile_putaway__error__no_serial_number_for_return',
        'No serial number for return',
    ),
    [SerialNumberManagementError.tooManySerialNumbersToReturn]: ui.localize(
        '@sage/wh-pages/pages__mobile_putaway__error__too_many_serial_numbers_to_return',
        'Too many serial numbers to return',
    ),
    [SerialNumberManagementError.serialGroupRequired]: ui.localize(
        '@sage/wh-pages/pages__mobile_putaway__error__serial_group_required',
        'Serial group is required',
    ),
    // This error is used to indicate that a new serial number needs to be added.
    // It does not correspond to an actual error message to be displayed to the user.
    [SerialNumberManagementError.addNewSerialNumber]: '',
};

@ui.decorators.page<MobilePutaway>({
    title: 'Putaway',
    isTitleHidden: true,
    mode: 'default',
    menuItem: receipt,
    priority: 200,
    node: '@sage/wh-input/StoringList',
    createAction: undefined,
    authorizationCode: 'CWSSTL',
    access: { node: '@sage/wh-input/StoringList' },
    skipDirtyCheck: true,
    headerCard() {
        switch (this._getActiveSection()) {
            case SelectionSection.main:
                return undefined;

            case SelectionSection.validateMovement:
                return {
                    title: this.storingListHeader,
                };

            case SelectionSection.declareSerialNumbers:
                return {
                    title: this.inputMovementHeader,
                    titleRight: this.productHeader,
                    line2: this.locationHeader,
                    line2Right: this.putawayQuantityHeader,
                };

            case SelectionSection.readdressing:
                break;

            default:
                return undefined;
        }
        return undefined;
    },
    headerDropDownActions() {
        return [
            // this.$standardSaveAction,
            // this.$standardCancelAction,
            // this.$standardOpenRecordHistoryAction,
        ];
    },
    async onLoad() {
        if (!(await this._initialize())) {
            this.disablePage();
            // Close the page
            this.$.finish();
        }
    },
    async onDirtyStateUpdated(isDirty: boolean) {
        setApplicativePageCrudActions({
            page: this,
            isDirty,
            save: undefined, // this.$standardSaveAction,
            cancel: undefined, // this.$standardCancelAction,
            remove: undefined, // this.$standardDeleteAction,
        });
    },
    businessActions() {
        return this._getBusinessActions();
    },
    navigationPanel: undefined,
})
export class MobilePutaway extends ui.Page<GraphApi> {
    /** internal properties */
    private _siteCodeSelected?: string;

    private _depositorCodeSelected?: string;

    private _inputMovementSelected?: PartialCollectionValueWithIds<ReceiptMovementAddressedBinding>;

    private _currentMovementNumber = 0;

    private _maxMovementNumber = 0;

    private _productConfiguration?: ProductConfiguration;

    private _productPalletizationPlan!: ProductPalletizationPlan;

    private _serialNumberManagement!: SerialNumberManagement;

    private _containerConsumptionOrUpcUnit?: ContainerUnit;

    private _controlCode?: string;

    private _saveStoreLocation: ExtractEdges<any> | null = null;

    @ui.decorators.pageAction<MobilePutaway>({
        title: 'Cancel',
        shortcut: ['f4'],
        buttonType: 'secondary',
        isHidden: true,
        async onClick() {
            if (
                this._getActiveSection() === SelectionSection.readdressing ||
                (await this._dialogConfirmDelete(
                    ui.localize(
                        '@sage/wh-pages/dialog__confirm_cancel_action__putaway_cancelButton',
                        'You will cancel the entry for this level. This action cannot be undone. Do you want to continue?',
                    ),
                ))
            ) {
                if (this._getActiveSection() === SelectionSection.declareSerialNumbers) {
                    this.serialNumbers.value = [];
                    await this._setActiveSection(SelectionSection.validateMovement, <SectionParameters>{
                        isEdit: true,
                    });
                } else if (this._getActiveSection() === SelectionSection.readdressing) {
                    await this._setActiveSection(SelectionSection.validateMovement, <SectionParameters>{
                        isReinitialize: true,
                        isEdit: true,
                    });
                } else if (this._getActiveSection() === SelectionSection.validateMovement) {
                    await this._setActiveSection(SelectionSection.main, <SectionParameters>{
                        isReinitialize: true,
                    });
                } else {
                    await this._setActiveSection(SelectionSection.main);
                }
            }
        },
    })
    cancelButton!: ui.PageAction;

    @ui.decorators.pageAction<MobilePutaway>({
        title: 'Readdressing',
        buttonType: 'secondary',
        isHidden: true,
        async onClick() {
            if (this._getActiveSection() === SelectionSection.validateMovement) {
                await this._setActiveSection(SelectionSection.readdressing);
            }
        },
    })
    readdressingButton!: ui.PageAction;

    @ui.decorators.pageAction<MobilePutaway>({
        title: 'Next',
        shortcut: ['f3'],
        buttonType: 'primary',
        isHidden: true,
        async onClick() {
            if (this._getActiveSection() === SelectionSection.validateMovement) {
                if (!this.inputMovements.value.length) {
                    await dialogMessage(
                        this,
                        'error',
                        ui.localize('@sage/wh-pages/dialog-error-title', 'Error'),
                        ui.localize(
                            '@sage/wh-pages/pages__mobile_putaway__notification__no_receipt_error',
                            'Select at least one input movement.',
                        ),
                    );
                    return;
                }
                await this._setSectionNextMovement();
                return;
            }

            // Validate page
            const _errors = await this.$.page.validate();
            if (!_errors.length) {
                switch (this._getActiveSection()) {
                    case SelectionSection.main:
                        if (!this._selectFirstMovement()) {
                            await dialogMessage(
                                this,
                                'error',
                                ui.localize('@sage/wh-pages/dialog-error-title', 'Error'),
                                ui.localize(
                                    '@sage/wh-pages/pages__mobile_putaway__notification__no_receipt_error',
                                    'Select at least one input movement.',
                                ),
                            );
                            return;
                        }

                        await this._setActiveSection(SelectionSection.validateMovement);
                        break;

                    case SelectionSection.readdressing:
                        if (!this.newStoreLocation.value) {
                            await dialogMessage(
                                this,
                                'error',
                                ui.localize('@sage/wh-pages/dialog-error-title', 'Error'),
                                ui.localize(
                                    '@sage/wh-pages/pages__mobile_putaway__notification__no_location_error',
                                    'Select a new store location.',
                                ),
                            );
                            return;
                        }

                        await this._setActiveSection(SelectionSection.validateMovement, <SectionParameters>{
                            isEdit: true,
                        });

                        break;

                    default:
                        break;
                }
            }
        },
    })
    nextButton!: ui.PageAction;

    @ui.decorators.pageAction<MobilePutaway>({
        title(_value, _rowValue) {
            if (this._getActiveSection() === SelectionSection.readdressing) {
                return ui.localize('@sage/wh-pages/pages__mobile_putaway__button__title__readdressing', 'Readdressing');
            }
            if (
                this._getActiveSection() === SelectionSection.validateMovement &&
                this._productConfiguration?.isLocalizedSerialNumberAllowed
            ) {
                return ui.localize(
                    '@sage/wh-pages/pages__mobile_putaway__button__title__serial_numbers',
                    'Serial Numbers',
                );
            }
            return ui.localize('@sage/wh-pages/pages__mobile_putaway__button__title__putaway', 'Putaway');
        },
        buttonType: 'primary',
        shortcut: ['f2'],
        isDisabled: true,
        async onError(e) {
            await this._onError_createButton(e, true);
        },
        async onClick() {
            // Validate page
            const _errors = await this.$.page.validate();
            if (!_errors.length && this._siteCodeSelected) {
                const _section = this._getActiveSection();

                // If localized serial number, go to serial number declaration instead of creating the putaway directly
                if (
                    _section === SelectionSection.validateMovement &&
                    this._productConfiguration?.isLocalizedSerialNumberAllowed
                ) {
                    await this._setActiveSection(SelectionSection.declareSerialNumbers);
                    return;
                }

                let _result;

                this.createButton.isDisabled = true;
                this.nextButton.isDisabled = true;
                this.cancelButton.isDisabled = true;

                // Check if operator is authorized
                const _operatorCode = await getAuthorizedOperator(this, AuthorizedOperation.receipt);

                if (_operatorCode === undefined) {
                    this.$.showToast(
                        ui.localize(
                            '@sage/wh-pages/notification-error-operator-not-authorized',
                            'Operator not authorized.',
                        ),
                        { type: 'error', timeout: 5000 },
                    );
                    this.createButton.isDisabled = false;
                    this.nextButton.isDisabled = this.nextButton.isHidden;
                    this.cancelButton.isDisabled = this.cancelButton.isHidden;
                    return;
                }

                if (_operatorCode instanceof Error) {
                    _result = _operatorCode;
                } else {
                    this.$.loader.isHidden = false;
                    _result = await this._callCreationAPI(_operatorCode);
                    this.$.loader.isHidden = true;
                }

                if (_result instanceof Error) {
                    await this._onError_createButton(_result);
                } else {
                    await this.$.sound.success();

                    this.$.showToast(
                        ui.localize(
                            '@sage/wh-pages/notification-success-input-movement-stored',
                            'Input movement {{ inputMovement }} stored.',
                            {
                                inputMovement: this._inputMovementSelected?.code ?? '',
                            },
                        ),
                        { type: 'success', timeout: 5000 },
                    );

                    await this._setSectionNextMovement();
                }
            }
        },
    })
    createButton!: ui.PageAction;

    // validateButton!: ui.PageAction;

    /**
     * 🔴 Error handling for create button
     * @param e
     * @param noCancelButton if true, no cancel button is displayed
     */
    private async _onError_createButton(e: Error, noCancelButton?: boolean): Promise<void> {
        const _section = this._getActiveSection();
        if (!(await this._dialogError(e, noCancelButton))) {
            await this._setActiveSection(SelectionSection.main, <SectionParameters>{ isReinitialize: true });
            return;
        }

        // Return to the calling section
        await this._setActiveSection(_section, <SectionParameters>{ isEdit: true });
    }

    /**
     * 🔴 Error handling and ask the user if they should start again
     *
     * @param e  error
     * @param noCancelButton if true, no cancel button is displayed
     * @returns return true to restart the operation
     */
    private async _dialogError(e: Error, noCancelButton?: boolean): Promise<boolean> {
        this.$.loader.isHidden = true;
        const options: ui.dialogs.DialogOptions = {
            acceptButton: {
                text: ui.localize('@sage/wh-pages/button-goback', 'Go back'),
            },
            ...(!noCancelButton && {
                cancelButton: {
                    text: ui.localize('@sage/wh-pages/button-cancel', 'Cancel'),
                },
            }),
            size: 'small',
            mdContent: true,
        };

        let message = '';

        const _errors = (<any>e)?.errors;
        const _firstError =
            Array.isArray(_errors) && _errors.length > 0 ? _errors[0]?.extensions?.diagnoses : undefined;
        const _diagnoses = _firstError ?? _errors;

        if (_diagnoses?.length || e?.message) {
            const _messages = <string[]>[];
            _diagnoses
                ?.filter((d: { severity: number; message: any }) => (d?.severity ?? 3) > 2 && d.message)
                ?.forEach((d: { message: any }) => {
                    const _message = d.message.split('\n');
                    _messages.push(..._message);
                });
            const _result = _messages.length ? _messages : e.message.split('\n');
            message = `**${ui.localize(
                '@sage/wh-pages/dialog-error-an-error-has-occurred',
                'An error has occurred',
            )}**\n\n`;
            if (_result.length === 1) {
                message += `${_result[0]}`;
            } else {
                message += _result.map(item => `* ${item}`).join('\n');
            }
        } else {
            message = `${ui.localize(
                '@sage/wh-pages/pages_creation_error_a-connection-or-webservice-error-has-occurred',
                'A connection or webservice error error has occurred. Contact your administrator.',
            )}`;
        }
        await this.$.sound.error();

        return !noCancelButton
            ? dialogMessage(this, 'error', ui.localize('@sage/wh-pages/dialog-error-title', 'Error'), message, options)
            : dialogConfirmation(
                  this,
                  'error',
                  ui.localize('@sage/wh-pages/dialog-error-title', 'Error'),
                  message,
                  options,
              );
    }

    /*
     *
     *  Sections 🔵
     *
     */

    @ui.decorators.section<MobilePutaway>({
        isTitleHidden: true,
        isHidden: false,
    })
    mainSection!: ui.containers.Section;

    @ui.decorators.section<MobilePutaway>({
        isTitleHidden: true,
        isHidden: true,
    })
    validateMovementSection!: ui.containers.Section;

    @ui.decorators.section<MobilePutaway>({
        isTitleHidden: true,
        isHidden: true,
    })
    declareSerialNumbersSection!: ui.containers.Section;

    @ui.decorators.section<MobilePutaway>({
        isTitleHidden: true,
        isHidden: true,
    })
    readdressingSection!: ui.containers.Section;

    /*
     *
     *  Blocks
     *
     */

    @ui.decorators.block<MobilePutaway>({
        parent() {
            return this.mainSection;
        },
    })
    mainBlock!: ui.containers.Block;

    @ui.decorators.block<MobilePutaway>({
        parent() {
            return this.validateMovementSection;
        },
    })
    validateMovementBlock!: ui.containers.Block;

    @ui.decorators.block<MobilePutaway>({
        parent() {
            return this.declareSerialNumbersSection;
        },
    })
    declareSerialNumbersBlock!: ui.containers.Block;

    @ui.decorators.block<MobilePutaway>({
        parent() {
            return this.readdressingSection;
        },
    })
    readdressingBlock!: ui.containers.Block;

    /*
     *
     *  Page properties
     *
     */
    // -----------------------------

    /*
     *
     *  Technical Fields
     *
     */

    /**
     * Case of movement validation section
     */

    // Header section

    @ui.decorators.textField<MobilePutaway>({
        title: 'Storing no.',
        prefix: ui.localize('@sage/wh-pages/storing-list-prefix', 'Storing no.'),
        isTransient: true,
        isFullWidth: true,
    })
    storingListHeader!: ui.fields.Text;

    /**
     * Case of serial numbers section
     */

    // Header section

    @ui.decorators.textField<MobilePutaway>({
        title: 'Input mvt no.',
        // prefix: ui.localize('@sage/wh-pages/input-movement-prefix', 'Input mvt no.'),
        isTransient: true,
        isFullWidth: true,
    })
    inputMovementHeader!: ui.fields.Text;

    @ui.decorators.textField<MobilePutaway>({
        title: 'Product',
        // prefix: ui.localize('@sage/wh-pages/Product-prefix', 'Product'),
        isTransient: true,
        isFullWidth: true,
    })
    productHeader!: ui.fields.Text;

    @ui.decorators.textField<MobilePutaway>({
        title: 'Location',
        // prefix: ui.localize('@sage/wh-pages/store-location-prefix', 'Location'),
        isTransient: true,
        isFullWidth: true,
    })
    locationHeader!: ui.fields.Text;

    @ui.decorators.numericField<MobilePutaway>({
        title: 'Putaway quantity',
        // prefix: ui.localize('@sage/wh-pages/putaway-quantity-prefix', 'Putaway quantity'),
        postfix(_value, _rowValue) {
            return this._productPalletizationPlan?.productConsumptionUnitCode ?? '';
        },
        scale(_value, _rowValue) {
            return this._productPalletizationPlan?.productConsumptionUnitPrecision ?? 0;
        },
        isTransient: true,
        isFullWidth: true,
    })
    putawayQuantityHeader!: ui.fields.Numeric;

    /*
     *
     *  Fields
     *
     */

    // -----------------------------------------------
    // Block: 🟢 Main block
    // -----------------------------------------------

    @ui.decorators.referenceField<MobilePutaway, StoringList>({
        parent() {
            return this.mainBlock;
        },
        title: 'Storing no.',
        placeholder: 'Scan or select...',
        node: '@sage/wh-input/StoringList',
        valueField: 'code',
        isTransient: true,
        isFullWidth: true,
        isHelperTextHidden: true,
        isAutoSelectEnabled: true,
        canFilter: false,
        filter() {
            const _filter: Filter<StoringList> = {
                site: { code: this._siteCodeSelected },
                depositor: { code: this._depositorCodeSelected },
                status: { _nin: ['beingCreated', 'validated'] },
            };

            return _filter;
        },
        async onChange() {
            if (this.storingList.value?.code) {
                this.receiptMovementAddressed.isDisabled = true;
                this.receiptMovementAddressed.value = null;
                await this._getInputMovement(this.storingList.value.code);
            } else {
                this.receiptMovementAddressed.isDisabled = false;
                await this._getInputMovement();
            }
        },
        columns: [
            ui.nestedFields.text({
                bind: '_id',
                title: 'ID',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'code',
                title: 'Storing no.',
                isReadOnly: true,
            }),
            ui.nestedFields.select({
                bind: 'status',
                title: 'Status',
                isReadOnly: true,
                optionType: '@sage/wh-product-data/StoringListPurchaseOrderStatus',
            }),
            ui.nestedFields.date({
                bind: 'date',
                title: 'Date',
                isReadOnly: true,
            }),
        ],
    })
    storingList!: ui.fields.Reference;

    @ui.decorators.referenceField<MobilePutaway, ReceiptMovementAddressed>({
        parent() {
            return this.mainBlock;
        },
        title: 'Input movement no.',
        placeholder: 'Scan or select...',
        node: '@sage/wh-input/ReceiptMovementAddressed',
        valueField: 'code',
        isTransient: true,
        isFullWidth: true,
        isHelperTextHidden: true,
        isAutoSelectEnabled: true,
        canFilter: false,
        filter() {
            const _filter: Filter<ReceiptMovementAddressed> = {
                site: { code: this._siteCodeSelected },
                depositor: { code: this._depositorCodeSelected },
                storingList: { code: { _ne: null } },
                ...(this.storingList.value?.code && { storingList: { code: this.storingList.value.code } }),
                ...(this.inputMovements.value.length && {
                    code: { _nin: this.inputMovements.value.map(movement => movement.code) },
                }),
            };

            return _filter;
        },
        async onChange() {
            if (this.receiptMovementAddressed.value?.code) {
                this.storingList.isDisabled = true;
                this.storingList.value = null;
                await this._getInputMovement(undefined, this.receiptMovementAddressed.value.code);
                this.receiptMovementAddressed.value = null;
                this.receiptMovementAddressed.focus();
            } else {
                this.storingList.isDisabled = !this.inputMovements.value.length;
            }
        },
        columns: [
            ui.nestedFields.text({
                bind: '_id',
                title: 'ID',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.reference({
                bind: 'storingList',
                title: 'Storing no.',
                isReadOnly: true,
                node: '@sage/wh-input/StoringList',
                valueField: 'code',
            }),
            ui.nestedFields.reference({
                bind: 'product',
                title: 'Product',
                isReadOnly: true,
                node: '@sage/wh-product-data/Product',
                valueField: 'code',
            }),
            ui.nestedFields.reference({
                bind: 'location',
                title: 'Location',
                node: '@sage/wh-master-data/Location',
                isReadOnly: true,
                valueField: 'displayCode',
            }),
            ui.nestedFields.text({
                bind: 'code',
                title: 'Movement no.',
                isReadOnly: true,
            }),
        ],
    })
    receiptMovementAddressed!: ui.fields.Reference;

    @ui.decorators.tableField<MobilePutaway, ReceiptMovementAddressedBinding>({
        parent() {
            return this.mainBlock;
        },
        title(_value, _rowValue) {
            return ui.localize(
                '@sage/wh-pages/number-of-input-movements',
                '{{ numberOfMovements }} Input movement(s)',
                {
                    numberOfMovements: this.inputMovements.value.length,
                },
            );
        },
        node: '@sage/wh-input/ReceiptMovementAddressed',
        canSelect: false,
        canUserHideColumns: false,
        canFilter: false,
        isTransient: true,
        displayMode: ui.fields.TableDisplayMode.comfortable,
        mobileCard: undefined,
        orderBy: {
            storingList: { code: 1 },
            storingListLineNumber: 1,
            homogeneousContainer: { container: { code: 1 } },
            homogeneousQuantity: 1,
            code: 1,
        },
        columns: [
            ui.nestedFields.text({
                bind: '_id',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'code',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                bind: 'product',
                node: '@sage/wh-product-data/Product',
                isReadOnly: true,
                valueField: 'code',
            }),
            ui.nestedFields.reference({
                bind: 'location',
                node: '@sage/wh-master-data/Location',
                isReadOnly: true,
                valueField: 'displayCode',
            }),
            ui.nestedFields.numeric({
                bind: 'quantityInConsumptionUnit',
                isReadOnly: true,
                postfix(_value, rowValue) {
                    return (<ConsumptionUnit>(<any>rowValue?.product)?.consumptionUnit)?.code ?? '';
                },
                scale(_value, rowValue) {
                    return rowValue?.product?.stockUnit?.numberOfDecimals ?? 0;
                },
            }),
            ui.nestedFields.reference({
                bind: 'storingList',
                node: '@sage/wh-input/StoringList',
                isReadOnly: true,
                valueField: 'code',
            }),
            ui.nestedFields.reference({
                bind: 'location',
                node: '@sage/wh-master-data/Location',
                isReadOnly: true,
                isHidden: true,
                valueField: 'storeLocationCode',
            }),
            ui.nestedFields.reference({
                bind: 'location',
                node: '@sage/wh-master-data/Location',
                isReadOnly: true,
                isHidden: true,
                valueField: 'storeLocationKey',
            }),
            ui.nestedFields.numeric({
                bind: 'storingListLineNumber',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.reference({
                bind: 'homogeneousContainer',
                node: '@sage/wh-product-data/ProductContainer',
                isReadOnly: true,
                isHidden: true,
                valueField: 'code',
            }),
            ui.nestedFields.numeric({
                bind: 'homogeneousQuantity',
                isReadOnly: true,
                isHidden: true,
            }),
        ],
        dropdownActions: [
            {
                icon: 'bin',
                title: 'Delete',
                isDestructive: true,
                async onClick(recordId: string) {
                    if (
                        await this._dialogConfirmDelete(
                            ui.localize(
                                '@sage/wh-pages/dialog__confirm_delete_action__putwaway__movement',
                                'You are going to delete this movement. This action cannot be undone. Do you want to continue?',
                            ),
                        )
                    ) {
                        if (this.inputMovements.value.length === 1) {
                            await this._setActiveSection(SelectionSection.main, <SectionParameters>{
                                isReinitialize: true,
                            });
                            return;
                        }
                        this.inputMovements.removeRecord(recordId);
                        this._updateButtonsState();
                    }
                },
            },
        ],
    })
    inputMovements!: ui.fields.Table<ReceiptMovementAddressedBinding>;

    // -----------------------------------------------
    // Block: 🟢 Validate movement block
    // -----------------------------------------------

    @ui.decorators.referenceField<MobilePutaway, ReceiptMovementAddressed>({
        parent() {
            return this.validateMovementBlock;
        },
        title: 'Input movement no.',
        node: '@sage/wh-input/ReceiptMovementAddressed',
        valueField: 'code',
        isTransient: true,
        isReadOnly: true,
        isDisabled: true,
        isHelperTextHidden: true,
        canFilter: false,
    })
    inputMovement!: ui.fields.Reference;

    @ui.decorators.referenceField<MobilePutaway, Product>({
        parent() {
            return this.validateMovementBlock;
        },
        title: 'Product',
        node: '@sage/wh-product-data/Product',
        bind: 'product',
        valueField: 'code',
        isTransient: true,
        isReadOnly: true,
        isDisabled: true,
        isHelperTextHidden: true,
        canFilter: false,
    })
    product!: ui.fields.Reference;

    @ui.decorators.textField<MobilePutaway>({
        parent() {
            return this.validateMovementBlock;
        },
        title: 'Location',
        isTransient: true,
        isReadOnly: true,
        isDisabled: true,
    })
    storeLocationCode!: ui.fields.Text;

    @ui.decorators.textField<MobilePutaway>({
        parent() {
            return this.validateMovementBlock;
        },
        title: 'Barcode location',
        placeholder: 'Scan or select...',
        isTransient: true,
        isHelperTextHidden: true,
        async onChange() {
            if (this.barcodeStoreLocation.value) {
                const _barcodeStoreLocation = this.barcodeStoreLocation.value?.trim();
                if (_barcodeStoreLocation) {
                    const _controlCode = this._controlCode
                        ? _barcodeStoreLocation
                        : _barcodeStoreLocation.toUpperCase();
                    this.barcodeStoreLocation.value = _controlCode;
                    this._isValidLocationForProduct(_controlCode);
                }
            }
            this._updateButtonsState();
        },
        async validation(_value) {
            if (this.barcodeStoreLocation.value && this._getActiveSection() === SelectionSection.validateMovement) {
                return this._checkLocationForProduct(this.barcodeStoreLocation.value);
            }
            return undefined;
        },
    })
    barcodeStoreLocation!: ui.fields.Text;

    /**
     * Checks if the given barcode store location is valid for the product.
     * @param barcodeStoreLocation The barcode store location to check or undefined for readdressing field.
     * @returns True if the location is valid, false otherwise.
     */
    private _isValidLocationForProduct(barcodeStoreLocation?: string): boolean {
        const _controlCode = barcodeStoreLocation?.trim();
        const _storeCode = this._inputMovementSelected?.store?.code;
        const _locationCode = this._inputMovementSelected?.location?.code;

        if (!_storeCode || !_locationCode || !_controlCode) {
            return false;
        }

        const _errorMessage = this._checkLocationForProduct(_controlCode);
        if (_errorMessage) {
            this.$.showToast(_errorMessage, { type: 'error', timeout: 5000 });
            return false;
        }

        return true;
    }

    /**
     * Checks if the given barcode store location is valid for the product.
     * The check will verify the original location unless a new one is specified,
     * in which case it must be different.
     * All checks are applied when the barcode is provided because it is a control field, not a readdressing field.
     * @param barcodeStoreLocation The barcode store location to check or undefined for readdressing field.
     * @returns An error message if the location is invalid, or undefined if it is valid.
     */
    private _checkLocationForProduct(barcodeStoreLocation?: string): string | undefined {
        let _controlCode = barcodeStoreLocation?.trim();
        const _storeCode = this._inputMovementSelected?.store?.code;
        const _locationCode = _storeCode ? this._inputMovementSelected?.location?.code : undefined;

        if (!_storeCode || !_locationCode || !_controlCode) {
            return undefined;
        }

        const _isNotDefault = this.newStoreLocation.value !== null;
        const _newStoreCode = _isNotDefault ? this.newStoreLocation.value?.store?.code : _storeCode;
        const _newLocationCode = _isNotDefault ? this.newStoreLocation.value?.code : _locationCode;
        const _isSameLocation = _isNotDefault && _newStoreCode === _storeCode && _newLocationCode === _locationCode;

        if (!_controlCode && _newStoreCode && _newLocationCode) {
            // Build the store location code from the new store and location codes (readdressing case in progress)
            const _newStoreCode2 = _newStoreCode.trim();
            const _newLocationCode2 = _newLocationCode.trim();
            if (_storeCode && _locationCode) {
                _controlCode = `${_newStoreCode2?.padEnd(MAX_STORE_CODE_LENGTH)?.substring(0, MAX_STORE_CODE_LENGTH)}${_newLocationCode2}`;
            }
        }

        if (_isSameLocation) {
            return ui.localize(
                '@sage/wh-pages/notification-error-same-location',
                'Starting location = destination location.',
            );
        }

        const _barcodeStoreLocation = _controlCode.toUpperCase();
        const _currentStoreCode = _barcodeStoreLocation.substring(0, MAX_STORE_CODE_LENGTH)?.trimEnd();
        const _currentLocationCode = _barcodeStoreLocation.substring(MAX_STORE_CODE_LENGTH);

        if (this._controlCode && !_isNotDefault) {
            if (this._controlCode !== _controlCode) {
                return ui.localize(
                    '@sage/wh-pages/notification-error-barcode-incorrect-control-code',
                    'Incorrect control code.',
                );
            }
        } else if (_newStoreCode !== _currentStoreCode && _newLocationCode !== _currentLocationCode) {
            return ui.localize(
                '@sage/wh-pages/notification-error-barcode-incorrect-store-and-location',
                'Barcode incorrect for the store and the address {{ storeCode }} {{ locationCode }}.',
                {
                    storeCode: _newStoreCode,
                    locationCode: _newLocationCode,
                },
            );
        } else if (_newStoreCode !== _currentStoreCode) {
            return ui.localize(
                '@sage/wh-pages/notification-error-barcode-incorrect-store',
                'Barcode incorrect for the store {{ storeCode }}.',
                {
                    storeCode: _newStoreCode,
                },
            );
        } else if (_newLocationCode !== _currentLocationCode) {
            return ui.localize(
                '@sage/wh-pages/notification-error-barcode-incorrect-location',
                'Barcode incorrect for the address {{ locationCode }}.',
                {
                    locationCode: _newLocationCode,
                },
            );
        }

        return undefined;
    }

    @ui.decorators.dropdownListField<MobilePutaway>({
        parent() {
            return this.validateMovementBlock;
        },
        title: 'Container',
        placeholder: 'Scan or select...',
        isTransient: true,
        isReadOnly: true,
        isDisabled: true,
        options(_value, _rowValue) {
            return this._productPalletizationPlan.getProductContainerUnitOptions();
        },
    })
    container!: ui.fields.DropdownList;

    @ui.decorators.numericField<MobilePutaway>({
        parent() {
            return this.validateMovementBlock;
        },
        title: 'Quantity',
        isTransient: true,
        isReadOnly: true,
        isDisabled: true,
        isHelperTextHidden: true,
    })
    numberOfContainers!: ui.fields.Numeric;

    @ui.decorators.dropdownListField<MobilePutaway>({
        parent() {
            return this.validateMovementBlock;
        },
        title: 'Homogeneous container',
        placeholder: 'Scan or select...',
        isMandatory: true,
        isTransient: true,
        options(_value, _rowValue) {
            return this._productPalletizationPlan.getHomogeneousProductContainerUnitOptions();
        },
        async validation(value: string) {
            if (
                value &&
                this.container.value &&
                !this._productPalletizationPlan.validateContainerUnit(this.container.value, value, true)
            ) {
                return ui.localize(
                    '@sage/wh-pages/validate-error-container-the-container-is-missing-from-the-palletization-plan',
                    'The container is missing from the palletization plan: {{ inputContainerCode }}.',
                    { inputContainerCode: this.container.value },
                );
            }
            return undefined;
        },
        async onChange() {
            this._updateHomogeneousQuantityInConsumptionUnit();
            this._updateButtonsState();
        },
    })
    homogeneousContainer!: ui.fields.DropdownList;

    @ui.decorators.numericField<MobilePutaway>({
        parent() {
            return this.validateMovementBlock;
        },
        title: 'Quantity',
        placeholder: 'Enter...',
        isTransient: true,
        isHelperTextHidden: true,
        isMandatory: true,
        min: 0,
        async validation(value: number) {
            const _quantityInConsumptionUnit = Number(this._inputMovementSelected?.quantityInConsumptionUnit ?? 0);

            if (_quantityInConsumptionUnit) {
                const regex = /[1-9]\d*/; // reg ex for any positive numbers (integers or decimals) excluding 0
                if ((value.toString().match(regex)?.length ?? 0) === 0) {
                    return ui.localize('@sage/wh-pages/validate-error-invalid-value', 'Invalid value');
                }

                const _homogeneousQuantityInConsumptionUnit = this._getHomogeneousQuantityInConsumptionUnit();

                if (_homogeneousQuantityInConsumptionUnit > _quantityInConsumptionUnit) {
                    return ui.localize(
                        '@sage/wh-pages/dialog-error-entered-quantity-greater-than-initial-quantity-of-the-movement',
                        'Entered quantity {{ enteredQuantity }} > initial quantity {{ initialQuantity }} of the movement.',
                        {
                            enteredQuantity: _homogeneousQuantityInConsumptionUnit,
                            initialQuantity: _quantityInConsumptionUnit,
                        },
                    );
                }

                const _remainder = this._productPalletizationPlan.getHomogeneousProductContainerUnitMultiple(
                    _homogeneousQuantityInConsumptionUnit,
                    this.container.value ?? '',
                    this.homogeneousContainer.value ?? '',
                    true,
                );

                if (_remainder > 0) {
                    return ui.localize(
                        '@sage/wh-pages/dialog-error-entered-quantity-is-not-a-multiple-of-the-homogeneous-quantity',
                        'Entered quantity is not a multiple of the homogeneous quantity ({{ enteredQuantity }} {{ consumptionUnitCode }}/STK).',
                        {
                            consumptionUnitCode: this._productPalletizationPlan.productConsumptionUnitCode,
                            enteredQuantity: _remainder,
                        },
                    );
                }
            }

            return undefined;
        },
        scale(_value, _rowValue?) {
            return this._productPalletizationPlan.getProductContainerUnitPrecision(this.homogeneousContainer.value);
        },
        async onChange() {
            this._updateHomogeneousQuantityInConsumptionUnit();
            this._updateButtonsState();
        },
    })
    homogeneousQuantity!: ui.fields.Numeric;

    /**
     * Updates the homogeneous quantity in the consumption unit.
     * @returns True if the update was successful, false otherwise.
     */
    private _updateHomogeneousQuantityInConsumptionUnit(): boolean {
        const _quantityInConsumptionUnit = Number(this._inputMovementSelected?.quantityInConsumptionUnit ?? 0);

        if (_quantityInConsumptionUnit) {
            const _homogeneousQuantityInConsumptionUnit = this._getHomogeneousQuantityInConsumptionUnit();
            if (_homogeneousQuantityInConsumptionUnit > _quantityInConsumptionUnit) {
                this._serialNumberManagement.homogeneousQuantityInConsumptionUnit =
                    _homogeneousQuantityInConsumptionUnit;
            }
        }
        return false;
    }

    // -----------------------------------------------
    // Block: 🟢 Serial numbers declaration block
    // -----------------------------------------------

    @ui.decorators.referenceField<MobilePutaway, SerialGroup>({
        parent() {
            return this.declareSerialNumbersBlock;
        },
        title: 'Serial group',
        placeholder: 'Scan or select...',
        node: '@sage/wh-master-data/SerialGroup',
        valueField: 'code',
        isTransient: true,
        isFullWidth: true,
        isHelperTextHidden: true,
        isAutoSelectEnabled: true,
        canFilter: false,
        filter() {
            return this._serialNumberManagement?.serialGroupFilter;
        },
        async onChange() {
            this._displaySerialNumberManagementError(await this._serialNumberManagement?.onChange_serialGroup(this));
            this._updateButtonsState();
        },
        columns: [
            ui.nestedFields.text({
                bind: '_id',
                title: 'ID',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'code',
                title: 'Storing no.',
                isReadOnly: true,
            }),
            ui.nestedFields.reference({
                bind: 'parentGroup',
                title: 'Parent group',
                isReadOnly: true,
                node: '@sage/wh-master-data/SerialGroup',
                valueField: 'code',
            }),
        ],
    })
    serialGroup!: ui.fields.Reference;

    @ui.decorators.filterSelectField<MobilePutaway, SerialNumber>({
        parent() {
            return this.declareSerialNumbersBlock;
        },
        title: 'Serial number',
        placeholder: 'Scan or select...',
        node: '@sage/wh-stock-data/SerialNumber',
        valueField: 'code',
        isTransient: true,
        isFullWidth: true,
        isHelperTextHidden: true,
        canFilter: false,
        isNewEnabled: true, // Recalculated dynamically in serial number management
        maxLength: 30,
        validation: /^[\w/\- ]*$/, // alphanumeric + / + - + space
        filter() {
            return this._serialNumberManagement?.serialNumberFilter;
        },
        async onChange() {
            this._displaySerialNumberManagementError(await this._serialNumberManagement?.onChange_serialNumber(this));
            this._updateButtonsState();
        },
        columns: [
            ui.nestedFields.text({
                bind: '_id',
                title: 'ID',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'displayCode',
                title: 'Serial no.',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'code',
                title: 'Serial no.',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'dummy' as any,
                isTransient: true,
                isReadOnly: true,
            }),

            ui.nestedFields.numeric({
                bind: 'double',
                title: 'Double',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'occurrence',
                title: 'Occurrence',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.reference({
                bind: 'serialGroup',
                title: 'Serial group',
                isReadOnly: true,
                node: '@sage/wh-master-data/SerialGroup',
                valueField: 'code',
            }),
            ui.nestedFields.reference({
                bind: 'serialParentGroup',
                title: 'Serial parent group',
                isReadOnly: true,
                node: '@sage/wh-master-data/SerialGroup',
                valueField: 'code',
            }),
            ui.nestedFields.text({
                bind: 'keyCodeForDuplicates',
                isReadOnly: true,
                isHidden: true,
            }),
        ],
    })
    serialNumber!: ui.fields.FilterSelect;

    @ui.decorators.tableField<MobilePutaway, SerialNumberBinding>({
        parent() {
            return this.declareSerialNumbersBlock;
        },
        title(_value, _rowValue) {
            return ui.localize(
                '@sage/wh-pages/number-of-serial-numbers',
                '{{ numberOfSerialNumbers }} Serial number(s)',
                {
                    numberOfSerialNumbers: this.serialNumbers.value.length,
                },
            );
        },
        node: '@sage/wh-stock-data/SerialNumber',
        canSelect: false,
        canUserHideColumns: false,
        canFilter: false,
        isTransient: true,
        displayMode: ui.fields.TableDisplayMode.comfortable,
        mobileCard: undefined,
        orderBy: { code: 1, double: 1, occurrence: 1 },
        columns: [
            ui.nestedFields.text({
                bind: '_id',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'displayCode',
                isReadOnly: true,
            }),
            ui.nestedFields.text({
                bind: 'code',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'double',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.numeric({
                bind: 'occurrence',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'free1',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'free2',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'free3',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'free4',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'free5',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'keyCodeForDuplicates',
                isReadOnly: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                bind: 'actionImport',
                isReadOnly: true,
                isHidden: true,
            }),
        ],
        dropdownActions: [
            {
                icon: 'bin',
                title: 'Delete',
                isDestructive: true,
                isHidden(_recordId, rowItem) {
                    return !this._isSerialNumberDeletionAllowed(rowItem);
                },
                isDisabled(_recordId, rowItem) {
                    return !this._isSerialNumberDeletionAllowed(rowItem);
                },
                async onClick(recordId: string) {
                    const _record = this.serialNumbers.getRecordValue(recordId);
                    if (
                        _record &&
                        this._isSerialNumberDeletionAllowed(_record) &&
                        (await this._dialogConfirmDelete(
                            ui.localize(
                                '@sage/wh-pages/dialog__confirm_delete_action__putwaway__serial_number',
                                'You are going to delete this serial number. This action cannot be undone. Do you want to continue?',
                            ),
                        ))
                    ) {
                        this.serialNumbers.removeRecord(recordId);
                        this._serialNumberManagement.resetSerialNumberNewEnabledState();
                        this._updateButtonsState();
                    }
                },
            },
        ],
    })
    serialNumbers!: ui.fields.Table<SerialNumberBinding>;

    /**
     *  Indicates whether the deletion of a serial number is allowed.
     * @param record  The row item associated with the record.
     * @returns  True if the deletion is allowed, false otherwise.
     */
    private _isSerialNumberDeletionAllowed(record: any): boolean {
        // You can authorize the deletion of a serial number when:
        // - you are requesting its creation
        // - this number does not belong to the movement and the total quantity available exceeds the requested quantity.;
        const _lineQuantityRemaining = Math.max(
            Number(this._inputMovementSelected?.directInputLine?.numberOfConsumptionUnitRemaining ?? 0),
            0,
        );
        const _movementQuantity = Math.min(
            Number(this._inputMovementSelected?.quantityInConsumptionUnit),
            this._getHomogeneousQuantityInConsumptionUnit(),
        );
        return (
            record?.actionImport === 'C' ||
            (record?.inputMovement?.code === undefined && _lineQuantityRemaining > _movementQuantity)
        );
    }

    // -----------------------------------------------
    // Block: 🟢 Readdressing block
    // -----------------------------------------------

    @ui.decorators.textField<MobilePutaway>({
        parent() {
            return this.readdressingBlock;
        },
        title: 'Current location',
        isTransient: true,
        isDisabled: true,
        isReadOnly: true,
        isHelperTextHidden: true,
    })
    currentStoreLocation!: ui.fields.Text;

    @ui.decorators.referenceField<MobilePutaway, Location>({
        parent() {
            return this.readdressingBlock;
        },
        title: 'New location',
        node: '@sage/wh-master-data/Location',
        bind: 'location',
        valueField: 'storeLocationCode',
        placeholder: 'Scan or select...',
        isTransient: true,
        isAutoSelectEnabled: true,
        shouldSuggestionsIncludeColumns: true,
        isHelperTextHidden: true,
        canFilter: false,
        orderBy: { storeLocationCode: 1 },
        filter() {
            const _filter: Filter<any> = {
                site: { code: this._siteCodeSelected },
                depositor: { code: { _in: [this._depositorCodeSelected, undefined] } },
                product: { code: { _in: [this.product.value?.code, undefined] } },
            };
            return _filter;
        },
        columns: [
            ui.nestedFields.text({
                title: 'Location',
                bind: 'storeLocationCode',
                canFilter: true,
            }),
            ui.nestedFields.reference({
                title: 'Product',
                bind: 'product',
                node: '@sage/wh-product-data/Product',
                canFilter: true,
                valueField: 'code',
            }),
            ui.nestedFields.select({
                title: 'Location type',
                bind: 'locationType',
                optionType: '@sage/wh-master-data/LocationType',
                canFilter: true,
            }),
            ui.nestedFields.text({
                title: 'Location',
                bind: 'storeLocationKey',
                canFilter: true,
                isHidden: true,
            }),
            ui.nestedFields.text({
                title: 'Location',
                bind: 'code',
                canFilter: true,
                isHidden: true,
            }),
            ui.nestedFields.reference({
                title: 'Store',
                bind: 'store',
                node: '@sage/wh-master-data/Store',
                canFilter: true,
                valueField: 'code',
            }),
        ],
        async onChange() {
            if (this.newStoreLocation.value) {
                this._isValidLocationForProduct();
            }
            this._updateButtonsState();
        },
        async validation(_value) {
            if (this.newStoreLocation.value && this._getActiveSection() === SelectionSection.readdressing) {
                return this._checkLocationForProduct();
            }
            return undefined;
        },
    })
    newStoreLocation!: ui.fields.Reference;

    /**
     * Initialize the business actions for the page
     * @returns List of business actions
     */
    private _getBusinessActions(): ui.PageAction<MobilePutaway>[] {
        switch (this._getActiveSection()) {
            case SelectionSection.main:
                return [this.cancelButton, this.nextButton];

            case SelectionSection.validateMovement:
                // TODO: ⚫🟣 Implement the future readdressing functionality once the basic mutation has been updated.
                // return [this.createButton, this.nextButton, this.readdressingButton, this.cancelButton];
                return [this.nextButton, this.createButton];

            case SelectionSection.declareSerialNumbers:
                return [this.cancelButton, this.createButton];

            case SelectionSection.readdressing:
                return [this.cancelButton, this.nextButton];

            default:
                /**
                 * The order in which the buttons are displayed is defined by the list,
                 * knowing that the first will be on the right, the second on the left...
                 */
                return [
                    // this.$standardSaveAction,
                    // this.$standardCancelAction,
                    // this.$standardNewAction,
                    // this.$standardDeleteAction,
                    this.cancelButton,
                    this.nextButton,
                    // this.createButton,
                ];
        }
    }

    /**
     * Initialize the page
     * @returns True if the page is correctly initialized, false otherwise
     */
    private async _initialize(): Promise<boolean> {
        if (!(await this._initSiteDepositor())) {
            return false;
        }

        this._productPalletizationPlan = new ProductPalletizationPlan(
            this.container,
            this.numberOfContainers,
            this.homogeneousContainer,
            this.homogeneousQuantity,
        );

        this._serialNumberManagement = new SerialNumberManagement(
            this.serialGroup,
            this.serialNumber,
            this.serialNumbers,
        );

        /**
         * Even if the general initialization reloads the site and the depositor,
         * it is positioned before to detect a possible problem.
         */
        if (!this._serialNumberManagement.initializeSiteDepositor(this)) {
            return false;
        }

        await this._setActiveSection(SelectionSection.main);

        return true;
    }

    /**
     *  Initialize the site and depositor from the user function profile
     * @returns True if the site and depositor are correctly initialized, false otherwise
     */
    private async _initSiteDepositor(): Promise<boolean> {
        const siteDepositor = await getCurrentSiteDepositor(
            this,
            ui.localize('@sage/wh-pages/dialog-error-title', 'Error'),
            ui.localize(
                '@sage/wh-pages/dialog-error-location-inquiry-set-site-depositor',
                'Define a default site and depositor on the user function profile.',
            ),
        );
        if (siteDepositor && siteDepositor?.site && siteDepositor?.depositor) {
            this._siteCodeSelected = siteDepositor?.site;
            this._depositorCodeSelected = siteDepositor?.depositor;
            return true;
        }
        return false;
    }

    /**
     * Disable the page
     * Used when there is a blocking problem
     */
    private async disablePage() {
        await this._setActiveSection(SelectionSection.disableAll);
        this.mainSection.isDisabled = true;
        this.validateMovementSection.isDisabled = true;
        this.declareSerialNumbersSection.isDisabled = true;
        this.readdressingSection.isDisabled = true;
        this.mainBlock.isDisabled = true;
        this.validateMovementBlock.isDisabled = true;
        this.declareSerialNumbersBlock.isDisabled = true;
        this.readdressingBlock.isDisabled = true;
    }

    /**
     * Set the active section
     * The section will only be made visible at the end, so as not to have the various
     * changes made to the section.
     * @param section target section
     * @param parameters optional parameters
     * @returns activated section
     */
    private async _setActiveSection(
        section: SelectionSection,
        parameters?: SectionParameters,
    ): Promise<SelectionSection> {
        const _previousSection = this._getActiveSection();
        let _section = section;
        let _numberOfSelectedMovements = this.inputMovements.value.length;
        let _numberOfSerialNumber = this.serialNumbers.value.length;

        /**
         * Evaluate the right section to display
         */
        switch (section) {
            case SelectionSection.disableAll:
                _numberOfSelectedMovements = 0;
                _numberOfSerialNumber = 0;
                this._reinitializePendingReceiptMovements();
                await this.$.router.emptyPage(true);
                break;

            case SelectionSection.main:
                _section = section;
                _numberOfSerialNumber = 0;
                this.serialNumbers.value = [];
                if (parameters?.isReinitialize) {
                    _numberOfSelectedMovements = 0;
                    this._reinitializePendingReceiptMovements();
                    await this.$.router.emptyPage(true);
                }
                break;

            case SelectionSection.validateMovement:
                if (_previousSection === SelectionSection.main) {
                    this._currentMovementNumber = 1;
                    this._maxMovementNumber = _numberOfSelectedMovements;
                } else if (_previousSection === SelectionSection.readdressing && parameters?.isReinitialize) {
                    this.newStoreLocation.value = this._saveStoreLocation;
                }
                this._saveStoreLocation = null;
                break;

            case SelectionSection.declareSerialNumbers:
                break;

            case SelectionSection.readdressing:
                this._saveStoreLocation = this.newStoreLocation.value;
                break;

            default:
                _section = SelectionSection.main;
                break;
        }

        /**
         *  Set the header
         */

        this._setSubTitleForSection(_section);

        switch (_section) {
            case SelectionSection.disableAll:
                break;
            case SelectionSection.main:
                await this._initEnterMainBlockHeader();
                await this._initEnterMainBlock(false);
                break;
            case SelectionSection.validateMovement:
                await this._initEnterValidateMovementBlockHeader(_previousSection);
                await this._initEnterValidateMovementBlock(_previousSection, parameters);
                break;
            case SelectionSection.declareSerialNumbers:
                await this._initEnterDeclareSerialNumbersHeader();
                await this._initEnterDeclareSerialNumbersBlock(_previousSection, parameters);
                break;

            case SelectionSection.readdressing:
                await this._initEnterReaddressingHeader();
                await this._initEnterReaddressingBlock(_previousSection);
                break;
            default:
                break;
        }

        /**
         * Change buttons visibility
         * Only useful buttons will be active in the section displayed (business actions)
         */

        this.nextButton.isHidden = ![
            SelectionSection.main,
            SelectionSection.validateMovement,
            SelectionSection.readdressing,
        ].includes(_section);
        this.nextButton.isDisabled = this.nextButton.isHidden;

        this.cancelButton.isHidden = ![
            SelectionSection.declareSerialNumbers,
            SelectionSection.validateMovement,
            SelectionSection.readdressing,
        ].includes(_section);
        this.cancelButton.isDisabled = this.cancelButton.isHidden;

        this.createButton.isHidden = ![
            SelectionSection.validateMovement,
            SelectionSection.declareSerialNumbers,
        ].includes(_section);

        this.createButton.isDisabled = true;

        // TODO: ⚫🟣 Implement the future readdressing functionality once the basic mutation has been updated.
        // this.readdressingButton.isHidden = _section !== SelectionSection.validateMovement;
        this.readdressingButton.isHidden = true;
        this.readdressingButton.isDisabled = this.readdressingButton.isHidden;

        /**
         * Change display section
         */
        this.mainSection.isHidden = _section !== SelectionSection.main;
        this.validateMovementSection.isHidden = _section !== SelectionSection.validateMovement;
        this.declareSerialNumbersSection.isHidden = _section !== SelectionSection.declareSerialNumbers;
        this.readdressingSection.isHidden = _section !== SelectionSection.readdressing;

        /**
         * Apply changes based on new section, and update the page after all changes
         */

        this._updateButtonsState(_section);

        await this.$.commitValueAndPropertyChanges();

        // Closing section
        switch (_section) {
            case SelectionSection.disableAll:
                break;
            case SelectionSection.main:
                if (this.inputMovements.value.length && this.storingList.isDisabled) {
                    this.receiptMovementAddressed.focus();
                } else {
                    this.storingList.focus();
                }
                break;

            case SelectionSection.validateMovement:
                this.barcodeStoreLocation.focus();
                break;

            case SelectionSection.declareSerialNumbers:
                this.serialNumber.focus();
                break;

            case SelectionSection.readdressing:
                this.newStoreLocation.focus();
                break;

            default:
                break;
        }

        // The page is revalidated to cancel any errors that existed before the section reload,
        // except for those that remain.
        if (![SelectionSection.main, SelectionSection.disableAll].includes(_section)) {
            await this.$.page.validate();
        }

        return _section;
    }

    /**
     * Get the active section
     * @returns active section
     */
    private _getActiveSection(): SelectionSection {
        if (!this.mainSection.isHidden) {
            return SelectionSection.main;
        }
        if (!this.validateMovementSection.isHidden) {
            return SelectionSection.validateMovement;
        }
        if (!this.declareSerialNumbersSection.isHidden) {
            return SelectionSection.declareSerialNumbers;
        }
        if (!this.readdressingSection.isHidden) {
            return SelectionSection.readdressing;
        }
        return SelectionSection.disableAll;
    }

    /**
     * 🔸Update receipt button disabled state :
     *
     * Acceptance of the sequence of movements phase is:
     * - Either we're not entering movements.
     * - Either there are no more quantities to receive.
     * - or there are saved movements, which have been completely deleted because the user no longer wants the movements entered.
     *
     */
    private _updateButtonsState(_activeSection = this._getActiveSection()) {
        switch (_activeSection) {
            case SelectionSection.main:
                this.nextButton.isDisabled = !this.inputMovements.value.length;
                break;

            case SelectionSection.validateMovement:
                this.createButton.isDisabled = !this.barcodeStoreLocation.value || !this.homogeneousQuantity.value;
                this.nextButton.isDisabled = false;
                break;

            case SelectionSection.declareSerialNumbers:
                this.createButton.isDisabled =
                    !this.serialNumbers.value.length ||
                    this.serialNumbers.value.length !== this._getHomogeneousQuantityInConsumptionUnit();
                this.cancelButton.isDisabled = false;
                break;

            case SelectionSection.readdressing:
                this.nextButton.isDisabled = !this.newStoreLocation.value;
                break;

            default:
                this.nextButton.isDisabled = true;
                break;
        }
    }

    /**
     * Reinitialize pending movements with optional fields selectors.
     * @param _includeSelector If true, resets relevant state (default: false)
     */
    private _reinitializePendingReceiptMovements(_includeSelector: boolean = false): void {
        // Implement logic to reset or reinitialize working line and movements as needed.
        // This is a placeholder to avoid TypeScript errors.
        // If no logic is needed, leave empty.
        this._currentMovementNumber = 0;
        this._maxMovementNumber = 0;
        this._inputMovementSelected = undefined;
        this.inputMovements.value = [];
        this.serialNumbers.value = [];
        this.newStoreLocation.value = null;
    }

    /**
     *  Get the error message associated with a serial number management error code.
     * @param errorCode
     * @returns
     */
    private _displaySerialNumberManagementError(
        errorCode: SerialNumberManagementError | undefined,
    ): string | undefined {
        const _message = errorCode ? _serialNumberManagementError[errorCode] : undefined;

        // This error is used to indicate that a new serial number needs to be added.
        // It does not correspond to an actual error message to be displayed to the user.
        if (!_message || errorCode === SerialNumberManagementError.addNewSerialNumber) {
            return undefined;
        }

        this.$.showToast(_message, { type: 'error', timeout: 5000 });
        return _message;
    }

    /**
     * Set subtitle for active section
     * @returns true when subtitle has changed
     */
    private _setSubTitleForSection(_activeSection = this._getActiveSection()): boolean {
        let subTitle: string;
        switch (_activeSection) {
            case SelectionSection.main:
                subTitle = '';
                break;

            case SelectionSection.validateMovement:
                subTitle = ui.localize(
                    '@sage/wh-pages/pages__mobile_putaway__subtitle__validate_the_input_movement',
                    'Validate the input movement',
                );
                break;

            case SelectionSection.readdressing:
                subTitle = ui.localize('@sage/wh-pages/pages__mobile_putaway__subtitle__readdressing', 'Readdressing');
                break;

            case SelectionSection.declareSerialNumbers:
                subTitle = ui.localize(
                    '@sage/wh-pages/pages__mobile_putaway__subtitle__declare_serial_numbers',
                    'Declare serial numbers',
                );
                break;

            default:
                subTitle = '';
                break;
        }
        this.$.page.subtitle = subTitle;
        return subTitle !== '';
    }

    /**
     * Confirm delete dialog
     * @param optional message
     * @returns true if the user confirms the deletion
     */
    private async _dialogConfirmDelete(
        message: string = ui.localize(
            '@sage/wh-pages/dialog__confirm_delete_action__message',
            'You are going to make a deletion. This action cannot be undone. Do you want to continue?',
        ),
    ): Promise<boolean> {
        const options: ui.dialogs.DialogOptions = {
            acceptButton: {
                text: ui.localize('@sage/wh-pages/button-accept-continue', 'Continue'),
            },
            cancelButton: {
                text: ui.localize('@sage/wh-pages/button-cancel', 'Cancel'),
            },
            fullScreen: true,
        };
        return dialogConfirmation(
            this,
            'warn',
            ui.localize('@sage/wh-pages/dialog__delete_action_title', 'Confirm delete'),
            message,
            options,
        );
    }

    /** Initialize primary page */
    private async _initFieldsDestination(): Promise<boolean> {
        return !!this._siteCodeSelected && !!this._depositorCodeSelected && (await this._initEnterMainBlockHeader());
    }

    /**
     * ⚪ Main block management
     */

    /**
     * Reinitialize fields for main block filter
     * @returns true when done
     */
    // eslint-disable-next-line class-methods-use-this
    private async _initEnterMainBlockHeader(): Promise<boolean> {
        return true;
    }

    /**
     *  Update main section information
     *  @param refresh
     *  @returns true when exists receipt lines
     */
    // eslint-disable-next-line class-methods-use-this
    private async _initEnterMainBlock(_refresh: boolean = true): Promise<boolean> {
        return true;
    }

    /**
     * ⚪ Validate movement block management
     */

    /**
     * Initialize validate movement block
     *
     * @param _previousSection  previous section
     * @returns
     */
    private async _initEnterValidateMovementBlockHeader(_previousSection?: SelectionSection): Promise<boolean> {
        this.storingListHeader.value = this._inputMovementSelected?.storingList?.code ?? null;

        this.validateMovementBlock.title = ui.localize(
            '@sage/wh-pages/page__mobile_putaway__current_movement',
            '{{ currentMovementNumber }} / {{ maxMovementNumber }} mvt(s)',
            {
                currentMovementNumber: this._currentMovementNumber,
                maxMovementNumber: this._maxMovementNumber,
            },
        );

        return !!this.storingListHeader.value;
    }

    /**
     * Set the validate movement header
     */
    private async _initEnterValidateMovementBlock(
        _previousSection?: SelectionSection,
        _parameters?: SectionParameters,
    ): Promise<boolean> {
        // Partial configuration loaded
        this._productConfiguration = <ProductConfiguration>this._inputMovementSelected?.product;

        // If a control code is required on the movement, it will be entered instead of the location barcode.
        this._controlCode = this._getControlCodeFromMovement();

        // reading the palletization plan before assigning fields
        const _product = this._inputMovementSelected?.product;
        const _containerCode = this._inputMovementSelected?.container?.container?.code;
        let _isConsumptionUnit = false;

        if (
            !_parameters?.isEdit ||
            (_previousSection &&
                ![
                    SelectionSection.validateMovement,
                    SelectionSection.declareSerialNumbers,
                    SelectionSection.readdressing,
                ].includes(_previousSection))
        ) {
            this.newStoreLocation.value = null;

            if (await this._productPalletizationPlan.initialize(this, _product?.code)) {
                // We only need the standard plan
                const _containerUnitAndOptions = this._productPalletizationPlan.getProductContainerUnitDependingBarCode(
                    'supplierBc',
                    _containerCode,
                );

                this._containerConsumptionOrUpcUnit = _containerUnitAndOptions?.containerUnit;
                this.container.options = _containerUnitAndOptions?.containerOptions ?? [];
                this.container.value = _containerCode ?? null;
                _isConsumptionUnit = _containerCode === _containerUnitAndOptions?.containerUnit?.code;
                this.homogeneousContainer.isDisabled = _isConsumptionUnit;
            } else {
                this._productPalletizationPlan.reinitialize();
                this.homogeneousContainer.isDisabled = true;
                this.container.options = [];
                this.container.value = null;
            }

            // Fill in the fields in the section
            this.inputMovement.value = this._inputMovementSelected ?? null;
            this.product.value = _product ?? null;
            this.storeLocationCode.value = this._getStoreLocationFromMovement();
            this.barcodeStoreLocation.value = null;
            this.numberOfContainers.value = Number(this._inputMovementSelected?.numberOfContainers ?? 0);
            this.homogeneousContainer.value =
                this._inputMovementSelected?.homogeneousContainer?.container?.code ?? null;
            this.homogeneousQuantity.value = Number(
                (_isConsumptionUnit
                    ? this._inputMovementSelected?.numberOfContainers
                    : this._inputMovementSelected?.homogeneousQuantity) ?? 0,
            );
        } else if (_previousSection === SelectionSection.readdressing) {
            this.storeLocationCode.value = this._getStoreLocationFromMovement();
        }

        return !!this._inputMovementSelected;
    }

    /**
     * Get the store and location codes of the selected movement or the new location if readdressed
     * @returns the store and location codes of the selected movement
     */
    private _getStoreLocationFromMovement(): string {
        return !this.newStoreLocation.value
            ? `${this._inputMovementSelected?.store?.code ?? ''} ${this._inputMovementSelected?.location?.code ?? ''}`
            : `${this.newStoreLocation.value.store?.code ?? ''} ${this.newStoreLocation?.value?.code ?? ''}`;
    }

    /**
     * Get the control code of the selected movement if mandatory
     * @returns the control code of the current location selected movement or undefined
     */
    private _getControlCodeFromMovement(): string | undefined {
        return this._inputMovementSelected &&
            this._productConfiguration?.isKeyInControlCode &&
            this._inputMovementSelected.location?.controlCode &&
            this._inputMovementSelected.location.store?.isKeyInControlCode &&
            this._inputMovementSelected.location.isKeyInControlCode
            ? this._inputMovementSelected.location.controlCode
            : undefined;
    }

    /**
     * Get the homogeneous quantity in consumption unit
     * @returns The homogeneous quantity in consumption unit
     */
    private _getHomogeneousQuantityInConsumptionUnit(): number {
        return this._productPalletizationPlan.convertContainerQuantityInConsumptionUnit(
            this.homogeneousQuantity.value ?? 0,
            this.homogeneousContainer.value,
        );
    }

    /**
     * ⚪ localized serial numbers block management
     */

    /**
     * Set the declare serial numbers header
     */
    private async _initEnterDeclareSerialNumbersHeader(): Promise<boolean> {
        this.inputMovementHeader.value = this.inputMovement.value?.code ?? null;
        this.productHeader.value = this.product.value?.code ?? null;
        this.locationHeader.value = this._getStoreLocationFromMovement();
        this.putawayQuantityHeader.value = this._getHomogeneousQuantityInConsumptionUnit();

        // this.declareSerialNumbersBlock.title = ui.localize(
        //    '@sage/wh-pages/page__mobile_putaway__current_movement',
        //    '{{ currentMovementNumber }} / {{ maxMovementNumber }} mvt(s)',
        //    {
        //        currentMovementNumber: this._currentMovementNumber,
        //        maxMovementNumber: this._maxMovementNumber,
        //    },
        // );
        return true;
    }

    /**
     * Add serial numbers localized in grid ?
     * @returns true when done
     */
    private async _initEnterDeclareSerialNumbersBlock(
        _previousSection?: SelectionSection,
        _parameters?: SectionParameters,
    ): Promise<boolean> {
        this._productConfiguration = <ProductConfiguration>this._inputMovementSelected?.product;
        if (!_parameters?.isEdit || !this._productConfiguration) {
            if (
                !this._productConfiguration ||
                !(await this._serialNumberManagement.initialize(
                    this,
                    this._inputMovementSelected?.code,
                    this._getHomogeneousQuantityInConsumptionUnit(),
                ))
            ) {
                this._setActiveSection(SelectionSection.validateMovement, { isReinitialize: true });
                this.$.showToast(
                    ui.localize(
                        '@sage/wh-pages/notification-error-unable-to-initialize-the-localized-serial-number-manager',
                        'Unable to initialize the localized serial number manager.',
                    ),
                    { type: 'error', timeout: 5000 },
                );
                return false;
            }

            return true;
        }
        return !!this._inputMovementSelected;
    }

    /**
     * ⚪ readdressing block management
     */

    /**
     * Set the readdressing movement header
     */
    // eslint-disable-next-line class-methods-use-this
    private async _initEnterReaddressingHeader(): Promise<boolean> {
        // Reserved
        return false;
    }

    /**
     * Update address in current movement
     * @returns true when done
     */

    private async _initEnterReaddressingBlock(_previousSection?: SelectionSection): Promise<boolean> {
        this.currentStoreLocation.value = this._getStoreLocationFromMovement();
        this.newStoreLocation.value = null;
        return true;
    }

    /**
     * Positions itself on the 1st movement and returns its ID otherwise undefined.
     *
     * @returns movement ID or undefined
     */
    private _selectFirstMovement(): string | undefined {
        const _firstMovement = this.inputMovements.value[0];
        this._inputMovementSelected = _firstMovement;
        return _firstMovement?._id;
    }

    /**
     * Deletes the first existing movement then selects the next one if it exists.
     * If no movement is left, the selection will be cleared and a message will be displayed.
     *
     * @returns ID of the next selected move or undefined
     */
    private async _selectNextMovement(): Promise<string | undefined> {
        if (!this._removeFirstMovement()) {
            return undefined;
        }

        const _response = this._selectFirstMovement();

        if (!_response) {
            await dialogMessage(
                this,
                'info',
                ui.localize('@sage/wh-pages/dialog-information-title', 'Information'),
                ui.localize('@sage/wh-pages/dialog-info-end-of-processing', 'End of processing.'),
            );
            return undefined;
        }

        return _response;
    }

    /**
     * Sets the next movement as current and activates the movement validation section.
     * @returns
     */
    private async _setSectionNextMovement(): Promise<boolean> {
        if (await this._selectNextMovement()) {
            this._currentMovementNumber += 1;
            await this._setActiveSection(SelectionSection.validateMovement);
            return true;
        }

        await this._setActiveSection(SelectionSection.main, <SectionParameters>{
            isReinitialize: true,
        });
        return false;
    }

    /**
     * Deletes the first existing movement
     *
     * @returns True if the 1st record could be deleted
     */
    private _removeFirstMovement(): boolean {
        const _firstId = this._selectFirstMovement();
        if (_firstId) {
            const _firstMovement = this.inputMovements.getRecordValue(_firstId);
            if (_firstMovement) {
                this.inputMovements.removeRecord(_firstId);
                return true;
            }
        }
        return false;
    }

    /**
     * Replaces the storing list, adds a movement input or clears the table.
     *
     * @param storingListCode  storing list number to replace OR
     * @param receiptMovementAddressedCode input movement to add
     * @returns count of movements in the table
     */
    private async _getInputMovement(storingListCode?: string, receiptMovementAddressedCode?: string): Promise<number> {
        // The temporary table is emptied if a new storing list is to be taken or if no parameter has been provided to purge the table.
        if (storingListCode || !receiptMovementAddressedCode) {
            this.inputMovements.value = [];
        }

        let _receiptMovementAddressed: ExtractEdges<ReceiptMovementAddressed>[] = [];

        if (storingListCode) {
            _receiptMovementAddressed = await this._getReceiptMovementAddressedByStoringListCode(storingListCode);
        } else if (receiptMovementAddressedCode) {
            _receiptMovementAddressed =
                await this._getReceiptMovementAddressedByMovementCode(receiptMovementAddressedCode);
        }

        if (storingListCode) {
            this.inputMovements.value = _receiptMovementAddressed;
        } else if (receiptMovementAddressedCode && _receiptMovementAddressed.length) {
            this.inputMovements.value = [...this.inputMovements.value, ..._receiptMovementAddressed].sort(
                this._movementComparator,
            );
        }

        this._updateButtonsState();

        return this.inputMovements.value?.length ?? 0;
    }

    /**
     *
     * @param a first movement
     * @param b second movement
     * @returns
     */
    // eslint-disable-next-line class-methods-use-this
    private _movementComparator(
        a: PartialCollectionValueWithIds<ReceiptMovementAddressedBinding>,
        b: PartialCollectionValueWithIds<ReceiptMovementAddressedBinding>,
    ): number {
        const aStore = a.storingList?.code;
        const bStore = b.storingList?.code;

        // Missing storingList case -> first
        if (!aStore && bStore) return -1;
        if (!bStore && aStore) return 1;

        if (aStore && bStore) {
            if (aStore < bStore) return -1;
            if (aStore > bStore) return 1;
        }
        // Movement code already present
        const aCode = a.code!;
        const bCode = b.code!;
        if (aCode < bCode) return -1;
        if (aCode > bCode) return 1;
        return 0;
    }

    /**
     * 🔶 Get receipt movements addressed by storing list code
     * @param storingListCode
     * @returns returns the list of available movements
     */
    private async _getReceiptMovementAddressedByStoringListCode(
        storingListCode: string,
    ): Promise<ExtractEdges<ReceiptMovementAddressed>[]> {
        const _filter: Filter<ReceiptMovementAddressed> = {
            storingList: { code: storingListCode },
        };
        return this._getReceiptMovementAddressedInternal(_filter);
    }

    /**
     * 🔶 Get receipt movement addressed by movement code
     * @param receiptMovementAddressedCode
     * @returns returns the available move unless it is already in the cache
     */
    private async _getReceiptMovementAddressedByMovementCode(
        receiptMovementAddressedCode: string,
        allowDuplicates?: boolean,
    ): Promise<ExtractEdges<ReceiptMovementAddressed>[]> {
        // There can only be one copy of the movement in the cache
        if (
            !allowDuplicates &&
            this.inputMovements.value.some(movementCode => movementCode.code === receiptMovementAddressedCode)
        ) {
            return [];
        }

        const _filter: Filter<ReceiptMovementAddressed> = {
            code: receiptMovementAddressedCode,
        };

        return this._getReceiptMovementAddressedInternal(_filter);
    }

    /**
     * 🔶 This function is used to get administrative lines by custom functions
     * @param _localFilter
     * @returns
     */
    private async _getReceiptMovementAddressedInternal(
        _localFilter: Filter<ReceiptMovementAddressed> = {},
    ): Promise<ExtractEdges<ReceiptMovementAddressed>[]> {
        try {
            const _response = extractEdges<any>(
                await this.$.graph
                    .node('@sage/wh-input/ReceiptMovementAddressed')
                    .query(
                        ui.queryUtils.edgesSelector<ReceiptMovementAddressed>(
                            {
                                _id: true,
                                site: { code: true, _id: true },
                                depositor: { code: true, _id: true },
                                storingList: { code: true, _id: true },
                                storingListLineNumber: true,
                                code: true,
                                directInput: {
                                    _id: true,
                                    code: true,
                                },
                                directInputLine: {
                                    _id: true,
                                    lineNumber: true,
                                    linePointerNumber: true,
                                    receiptLineNumber: true,
                                    isCustomerReturn: true,
                                    numberOfConsumptionUnit: true,
                                    numberOfConsumptionUnitStored: true,
                                    numberOfConsumptionUnitRemaining: true,
                                },
                                product: {
                                    _id: true,
                                    code: true,
                                    localizedDescription: true,
                                },
                                store: { code: true, _id: true, isKeyInControlCode: true },
                                location: {
                                    code: true,
                                    displayCode: true,
                                    storeLocationCode: true,
                                    storeLocationKey: true,
                                    _id: true,
                                    isKeyInControlCode: true,
                                    controlCode: true,
                                    store: { code: true, _id: true, isKeyInControlCode: true },
                                },
                                isExclusiveStore: true,
                                quantityInConsumptionUnit: true,
                                numberOfContainers: true,
                                container: { container: { code: true, _id: true }, _id: true },
                                homogeneousQuantity: true,
                                homogeneousContainer: { container: { code: true, _id: true }, _id: true },
                            },
                            {
                                filter: {
                                    site: { code: this._siteCodeSelected },
                                    depositor: { code: this._depositorCodeSelected },
                                    ..._localFilter,
                                },
                                orderBy: {
                                    site: { code: 1 },
                                    depositor: { code: 1 },
                                    storingList: { code: 1 },
                                    storingListLineNumber: 1,
                                    status: 1,
                                    homogeneousContainer: { container: { code: 1 } },
                                    homogeneousQuantity: 1,
                                    code: 1,
                                },
                            },
                        ),
                    )
                    .execute(),
            );

            // Force consumption unit in product
            if (_response.length) {
                await this._addProductsConsumptionUnitsToReceiptMovements(_response);
            }

            return _response;
        } catch (error) {
            ui.console.error('Error reading receipt movements addressed:\n', JSON.stringify(error));
            return [];
        }
    }

    /**
     * 🔶 Add products consumption units to receipt movements
     * @param _receiptMovementAddressed The receipt movements addressed
     */
    private async _addProductsConsumptionUnitsToReceiptMovements(
        _receiptMovementAddressed: ExtractEdges<ReceiptMovementAddressedBinding>[],
    ): Promise<void> {
        if (_receiptMovementAddressed.length) {
            interface ProductData {
                consumptionUnit: ConsumptionUnit | null;
                productConfiguration: ProductConfiguration | null;
            }
            // Using a product dictionary to avoid multiple calls for the same product
            const _productData: Record<string, ProductData> = {};

            // Get unique product codes
            const uniqueProductCodes = [
                ...new Set(_receiptMovementAddressed.map(_movement => _movement.product?.code).filter(Boolean)),
            ];

            // Load consumption units for unique products
            await Promise.all(
                uniqueProductCodes.map(async _productCode => {
                    // Determines the product configuration
                    _productData[_productCode] = <ProductData>{
                        productConfiguration: await getProductConfiguration(this, _productCode),
                        // As well as the consumption unit and its precision
                        consumptionUnit: await getConsumptionUnit(this, _productCode),
                    };
                }),
            );

            // Apply consumption units to lines
            _receiptMovementAddressed.forEach(_movement => {
                const _productCode = _movement.product?.code;
                if (_productCode && _productData[_productCode].consumptionUnit) {
                    const _consumptionUnit = _productData[_productCode].consumptionUnit;
                    const _productConfiguration = _productData[_productCode].productConfiguration;
                    if (_consumptionUnit && _productConfiguration) {
                        // force configuration and consumption unit in product
                        _movement.product = {
                            ..._movement.product,
                            ...{
                                isKeyInControlCode: _productConfiguration.isKeyInControlCode,
                                isLocalizedSerialNumberAllowed: _productConfiguration.isLocalizedSerialNumberAllowed,
                                isKeyInLotNumber: _productConfiguration.isKeyInLotNumber,
                                numberOfDecimals: _productConfiguration.numberOfDecimals,
                            },
                            ...{ consumptionUnit: _consumptionUnit },
                        };
                    }
                }
            });
        }
    }

    /**
     * 🟡 Prepare the import payload
     * @param operatorCode The operator code
     * @returns {ImportStoringList} The import payload
     */

    private prepareDataMutation(operatorCode: string = ''): ImportStoringList {
        const _storeCode = this.newStoreLocation.value?.store?.code;
        const _locationCode = _storeCode ? this.newStoreLocation.value?.code : undefined;
        const _serialNumbers = this._productConfiguration?.isLocalizedSerialNumberAllowed
            ? this.serialNumbers.value.map<ImportSerialNumber>(serialNumber => ({
                  code: serialNumber.code ?? '',
                  double: serialNumber.double ?? 0,
                  occurrence: serialNumber.occurrence ?? 0,
                  ...(serialNumber.free1 && { free1: serialNumber.free1 }),
                  ...(serialNumber.free2 && { free2: serialNumber.free2 }),
                  ...(serialNumber.free3 && { free3: serialNumber.free3 }),
                  ...(serialNumber.free4 && { free4: serialNumber.free4 }),
                  ...(serialNumber.free5 && { free5: serialNumber.free5 }),
                  ...(serialNumber.actionImport && { actionImport: serialNumber.actionImport }),
              }))
            : [];
        const _receiptMovementAddressed = <ImportReceiptMovementAddressed>{
            code: this.inputMovement.value?.code,
            ...(_storeCode && { store: _storeCode }),
            ...(_locationCode && { location: _locationCode }),
            ...(this._controlCode &&
                this.barcodeStoreLocation.value && { controlCode: this.barcodeStoreLocation.value }),
            homogeneousContainer: this.homogeneousContainer.value,
            homogeneousQuantity: this.homogeneousQuantity.value ?? 0,
            ...(this._productConfiguration?.isLocalizedSerialNumberAllowed &&
                _serialNumbers.length && {
                    serialNumbers: _serialNumbers,
                }),
        };

        return <ImportStoringList>{
            code: this.inputMovement.value?.storingList?.code,
            operatorCode,
            ...(_receiptMovementAddressed && { addressedReceiptMovements: [_receiptMovementAddressed] }),
        };
    }

    /**
     * 🟡 Call the creation API
     * @param operatorCode The operator code
     * @returns {any} The result of the API call
     */
    private async _callCreationAPI(operatorCode: string = ''): Promise<any> {
        const _args = <any>this.prepareDataMutation(operatorCode);

        let _result: any;
        try {
            _result = (
                await this.$.graph
                    .node('@sage/wh-input/StoringList')
                    .mutations.putaway(
                        {
                            code: true,
                        },
                        {
                            parameter: _args,
                        },
                    )
                    .execute()
            )?.code;
            if (!_result) {
                throw new SystemError(
                    ui.localize(
                        '@sage/wh-pages/pages__mobile_putaway__notification__movement-not-validated',
                        'Movement not validated.',
                    ),
                );
            }
        } catch (error) {
            return error;
        }
        return _result;
    }
}
