"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateScalarValue = validateScalarValue;
const xtrem_date_time_1 = require("@sage/xtrem-date-time");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const lodash_1 = require("lodash");
function validateBoolean(value, localize) {
    if (value !== 'true' && value !== 'false') {
        return localize('@sage/xtrem-ui-components/invalid-boolean', 'Invalid boolean');
    }
    return undefined;
}
function validateEnum(enumValues, localize) {
    if (!(0, lodash_1.every)(enumValues?.map(item => enumValues.includes(item)))) {
        return localize('@sage/xtrem-ui-components/invalid-value', 'Invalid value');
    }
    return undefined;
}
function validateDate(value, localize) {
    const invalidRange = localize('@sage/xtrem-ui-components/invalid-range', 'Invalid range');
    if (value && Array.isArray(value)) {
        const errors = value.map(v => validateScalarValue({ type: xtrem_shared_1.GraphQLTypes.Date, value: v, localize }) ?? '');
        if (errors.some(Boolean)) {
            return errors.join(xtrem_shared_1.RANGE_DIVIDER);
        }
        const [start, end] = value;
        if (start && end && Date.parse(end.rawValue) < Date.parse(start.rawValue)) {
            return `${xtrem_shared_1.RANGE_DIVIDER}${invalidRange}`;
        }
        return undefined;
    }
    if (typeof value === 'object' && (value.formattedValue.length !== 10 || !(0, xtrem_date_time_1.isValidIsoDate)(value.rawValue))) {
        return localize('@sage/xtrem-ui-components/invalid-date', 'Invalid date');
    }
    return undefined;
}
function validateNumeric(value, type, localize) {
    const invalidRange = localize('@sage/xtrem-ui-components/invalid-range', 'Invalid range');
    const invalidInteger = localize('@sage/xtrem-ui-components/invalid-integer', 'Invalid integer');
    const invalidNumber = localize('@sage/xtrem-ui-components/invalid-number', 'Invalid number');
    // IMPORTANT: Validation of ranges must be done prior to the validation of integer and float
    //            values. Otherwise the validation of ranges failes.
    if (value && value.includes(xtrem_shared_1.RANGE_DIVIDER)) {
        const values = String(value).split(xtrem_shared_1.RANGE_DIVIDER);
        const errors = values.map(v => validateScalarValue({ type, value: v, localize }) ?? '');
        if (errors.some(Boolean)) {
            return errors.join(xtrem_shared_1.RANGE_DIVIDER);
        }
        const [start, end] = values.map(Number);
        if (start && end && end < start) {
            return `${xtrem_shared_1.RANGE_DIVIDER}${invalidRange}`;
        }
        return undefined;
    }
    if ((0, lodash_1.includes)([xtrem_shared_1.GraphQLTypes.Int, xtrem_shared_1.GraphQLTypes.IntReference], type) &&
        (value === undefined || value === null || !value.match(/^[\d]+$/gu))) {
        return invalidInteger;
    }
    if ((0, lodash_1.includes)([xtrem_shared_1.GraphQLTypes.Decimal, xtrem_shared_1.GraphQLTypes.Float], type) &&
        (value === undefined || value === null || Number.isNaN(Number.parseFloat(value)))) {
        return invalidNumber;
    }
    return undefined;
}
function validateScalarValue({ type, value, enumValues, localize, }) {
    switch (type) {
        case xtrem_shared_1.GraphQLTypes.Boolean:
            return validateBoolean(value, localize);
        case xtrem_shared_1.GraphQLTypes.Enum:
            return validateEnum(enumValues, localize);
        case xtrem_shared_1.GraphQLTypes.Date:
        case xtrem_shared_1.GraphQLTypes.DateTime:
            return validateDate(value, localize);
        case xtrem_shared_1.GraphQLTypes.Decimal:
        case xtrem_shared_1.GraphQLTypes.Float:
        case xtrem_shared_1.GraphQLTypes.Int:
        case xtrem_shared_1.GraphQLTypes.IntReference:
            return validateNumeric(value, type, localize);
        case 'InputStream':
        case xtrem_shared_1.GraphQLTypes.Json:
        case xtrem_shared_1.GraphQLTypes.ExternalReference:
            // Unsupported
            return undefined;
        default:
            return undefined;
    }
}
//# sourceMappingURL=table-validation-utils.js.map