"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.isPropertyFilterable = exports.mapFilterToEditorRow = exports.isCompleteRow = exports.SplitBox = void 0;
exports.filterTableReducer = filterTableReducer;
exports.getPropertyParentNode = getPropertyParentNode;
const xtrem_shared_1 = require("@sage/xtrem-shared");
const box_1 = __importDefault(require("carbon-react/esm/components/box"));
const lodash_1 = require("lodash");
const styled_components_1 = __importDefault(require("styled-components"));
const table_validation_utils_1 = require("../table-validation-utils");
const utils_1 = require("../utils");
exports.SplitBox = (0, styled_components_1.default)(box_1.default) `
    & > div {
        flex: 1;
    }
`;
const isCompleteRow = (row, validations, parameterMode) => row.property != null &&
    row.filterType != null &&
    // In parameter creation mode, the empty filter value is actually expected given the value is a parameter
    (row.filterValue != null ||
        (!!row.parameter && parameterMode === 'creation') ||
        row.filterType === 'empty' ||
        row.filterType === 'notEmpty') &&
    row.path != null &&
    row.labelPath != null &&
    !!Object.values(validations[row._id] ?? {}).every(v => v == null);
exports.isCompleteRow = isCompleteRow;
exports.mapFilterToEditorRow = (0, lodash_1.memoize)(({ value, selectedProperties = {}, }) => {
    return value.map(({ filterType, filterValue, data: { type }, id, labelPath, parameter, key, labelKey, label, title: _title, canBeExpanded: _canBeExpanded, canBeSelected: _canBeSelected, ...rest }, index) => {
        return {
            // spread needed in case columns are extended
            ...rest,
            _id: String(index + 1),
            property: selectedProperties?.[id],
            // legacy "filterType" for enums was "matches"
            filterType: type === 'Enum' && filterType === 'matches' ? 'set' : filterType,
            // legacy "filterValue" for enums could also not be an array
            filterValue: type === 'Enum' && !Array.isArray(filterValue) ? [filterValue] : filterValue,
            path: id,
            labelPath,
            key,
            labelKey,
            parameter,
            label,
        };
    });
});
function filterTableReducer() {
    return (0, lodash_1.memoize)((localize) => {
        return (state, action) => {
            switch (action.type) {
                case 'DATA_RESET':
                    return handleDataReset(state, action);
                case 'ROW_ADDED':
                    return handleRowAdded(state);
                case 'ROW_REMOVED':
                    return handleRowRemoved(state, action);
                case 'ROW_DRAGGED':
                    return handleRowDragged(state, action);
                case 'COLUMNS_CHANGED':
                    return handleColumnsChanged(state, action);
                case 'CELL_CHANGED':
                    return handleCellChanged(state, action, localize);
                default:
                    return state;
            }
        };
    });
}
function getPropertyParentNode({ labelPath, nodeNames, node, }) {
    const split = labelPath?.split('.');
    return split && split.length > 1 ? split.slice(-2, -1)[0] : (nodeNames[node ?? ''] ?? '');
}
function handleRowAdded(state) {
    const newRow = state.columns.reduce((acc, curr) => {
        acc[curr.id] = undefined;
        return acc;
    }, { _id: String(state.counter + 1) });
    return {
        ...state,
        counter: state.counter + 1,
        data: [...state.data, newRow],
    };
}
function handleRowRemoved(state, action) {
    return {
        ...state,
        data: state.data.filter(element => element._id !== action.row._id),
    };
}
function handleDataReset(state, { value, selectedProperties }) {
    const incompleteRows = state.data.filter(row => !(0, exports.isCompleteRow)(row, state.validations, state.parameterMode));
    const completeRows = (0, exports.mapFilterToEditorRow)({
        value,
        selectedProperties,
    });
    const completeRowsLength = completeRows.length;
    const data = [
        ...completeRows,
        ...incompleteRows.map((incompleteRow, index) => ({
            ...incompleteRow,
            _id: String(index + completeRowsLength + 1),
        })),
    ];
    return {
        ...state,
        counter: data.length,
        data,
    };
}
function handleColumnsChanged(state, { columns }) {
    return {
        ...state,
        columns,
    };
}
function handleRowDragged(state, action) {
    const orderedRows = (0, lodash_1.sortBy)(state.data, item => action.ids.indexOf(item._id));
    return {
        ...state,
        data: orderedRows,
    };
}
function getActualValue({ columnId, value, }) {
    switch (columnId) {
        case '_id':
        case 'key':
        case 'labelKey':
        case 'path':
        case 'labelPath':
        case 'parameter':
        case 'filterValue':
            return (0, lodash_1.isNil)(value) ? undefined : value;
        case 'property':
        case 'filterType':
            return value || undefined;
        default:
            return value;
    }
}
function handleCellChanged(state, action, localize) {
    const { changes: { rowId, columnId, value, rowData }, selectedProperties, } = action;
    const validationMessage = rowData.property?.data?.type && (columnId === 'filterValue' || rowData.filterValue) && !rowData.parameter
        ? (0, table_validation_utils_1.validateScalarValue)({
            type: rowData.property.data.type,
            value: columnId === 'filterValue' ? value : rowData.filterValue,
            enumValues: rowData.property.data.enumValues,
            localize,
        })
        : undefined;
    const data = state.data.map(c => {
        const curr = { ...c };
        if (curr._id !== rowId || (0, lodash_1.isEqual)(curr[columnId], value)) {
            return curr;
        }
        // reset filter value when setting filter type to type range
        // or filter type was range and is set to something else
        if (columnId === 'filterType' && (value === xtrem_shared_1.RANGE || curr.filterType === xtrem_shared_1.RANGE)) {
            curr.filterValue = undefined;
        }
        if (!(columnId === 'filterValue' &&
            value === undefined &&
            curr.parameter === true &&
            state.parameterMode === 'creation')) {
            // set actual value
            (0, lodash_1.set)(curr, columnId, getActualValue({ columnId, value }));
        }
        // set type and path for given property
        if (curr.property) {
            const prop = selectedProperties?.[curr.property.id];
            curr.path = prop?.id;
            curr.labelPath = prop?.labelPath;
        }
        // set filter type in case it cannot be chosen
        if (curr.property?.data?.type && [xtrem_shared_1.GraphQLTypes.Boolean].includes(curr.property.data.type)) {
            curr.filterType = xtrem_shared_1.MATCHES;
        }
        const columnType = curr.property?.data.type;
        // reset filter type & value if filter type is not compatible
        if (columnId === 'property' &&
            columnType &&
            // always reset value for enums & booleans
            (columnType === xtrem_shared_1.GraphQLTypes.Boolean ||
                columnType === xtrem_shared_1.GraphQLTypes.Enum ||
                !curr.filterType ||
                !xtrem_shared_1.filterGraphqlMapping[columnType]?.includes(curr.filterType) ||
                (curr.filterType === 'timeFrame' &&
                    columnType !== xtrem_shared_1.GraphQLTypes.Date &&
                    columnType !== xtrem_shared_1.GraphQLTypes.DateTime))) {
            curr.filterType = undefined;
            curr.filterValue = undefined;
        }
        // set default filter type
        if (columnId === 'property' &&
            curr.property &&
            !curr.filterType &&
            curr.property.data.type === xtrem_shared_1.GraphQLTypes.Enum) {
            curr.filterType = xtrem_shared_1.SET;
        }
        // set filter value for parameter filter in creation mode
        if (columnId === 'parameter' && curr.property && state.parameterMode === 'creation') {
            curr.filterValue = value ? curr.property.id : undefined;
        }
        return curr;
    });
    const changedRow = data.find(d => d._id === rowData._id);
    const validationResets = changedRow
        ? (0, xtrem_shared_1.objectKeys)(changedRow).reduce((acc, key) => {
            if (changedRow?.[key] === undefined) {
                (0, lodash_1.set)(acc, key, undefined);
            }
            return acc;
        }, {})
        : {};
    const validations = {
        ...state.validations,
        [rowData._id]: {
            ...state.validations?.[rowData._id],
            [columnId]: validationMessage,
            ...validationResets,
        },
    };
    return {
        ...state,
        data,
        validations,
    };
}
const isPropertyFilterable = (p) => {
    if (!p.data.canFilter) {
        return false;
    }
    const isTypeFilterable = p.data.type && (0, lodash_1.includes)(utils_1.filterableGraphqlTypes, p.data.type);
    const isKindFilterable = p.data.kind &&
        (0, lodash_1.includes)(utils_1.filterableGraphqlTypes.map(type => type.toLowerCase()), p.data.kind.toLowerCase());
    return isTypeFilterable || isKindFilterable;
};
exports.isPropertyFilterable = isPropertyFilterable;
//# sourceMappingURL=filter-table-utils.js.map