"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getMainPresentation = exports.presentationTranslations = void 0;
exports.tableWidgetReducer = tableWidgetReducer;
exports.enforceContiguousOrder = enforceContiguousOrder;
exports.areAllPropertiesNonSortable = areAllPropertiesNonSortable;
exports.checkValidGroupOrder = checkValidGroupOrder;
exports.isContentRowValid = isContentRowValid;
const xtrem_shared_1 = require("@sage/xtrem-shared");
const lodash_1 = require("lodash");
exports.presentationTranslations = (0, lodash_1.memoize)((localize) => ({
    Checkbox: localize('@sage/xtrem-ui-components/Checkbox', 'Checkbox'),
    Date: localize('@sage/xtrem-ui-components/Date', 'Date'),
    Label: localize('@sage/xtrem-ui-components/Label', 'Label'),
    Numeric: localize('@sage/xtrem-ui-components/Numeric', 'Numeric'),
    Progress: localize('@sage/xtrem-ui-components/Progress', 'Progress'),
    Text: localize('@sage/xtrem-ui-components/Text', 'Text'),
    Aggregate: localize('@sage/xtrem-ui-components/Aggregate', 'Aggregate'),
    Button: localize('@sage/xtrem-ui-components/Button', 'Button'),
    Calendar: localize('@sage/xtrem-ui-components/Calendar', 'Calendar'),
    Chart: localize('@sage/xtrem-ui-components/Chart', 'Chart'),
    Card: localize('@sage/xtrem-ui-components/Card', 'Card'),
    ContentTable: localize('@sage/xtrem-ui-components/ContentTable', 'Content table'),
    Count: localize('@sage/xtrem-ui-components/Count', 'Count'),
    Datetime: localize('@sage/xtrem-ui-components/Datetime', 'Datetime'),
    DatetimeRange: localize('@sage/xtrem-ui-components/DatetimeRange', 'Datetime range'),
    DetailList: localize('@sage/xtrem-ui-components/DetailList', 'Detail list'),
    DropdownList: localize('@sage/xtrem-ui-components/DropdownList', 'Dropdown list'),
    File: localize('@sage/xtrem-ui-components/File', 'File'),
    DynamicPod: localize('@sage/xtrem-ui-components/DynamicPod', 'Dynamic pod'),
    DynamicSelect: localize('@sage/xtrem-ui-components/DynamicSelect', 'Dynamic select'),
    FileDeposit: localize('@sage/xtrem-ui-components/FileDeposit', 'File deposit'),
    FilterEditor: localize('@sage/xtrem-ui-components/FilterEditor', 'Filter editor'),
    FilterSelect: localize('@sage/xtrem-ui-components/FilterSelect', 'Filter select'),
    FormDesigner: localize('@sage/xtrem-ui-components/FormDesigner', 'Form designer'),
    Icon: localize('@sage/xtrem-ui-components/Icon', 'Icon'),
    Image: localize('@sage/xtrem-ui-components/Image', 'Image'),
    Link: localize('@sage/xtrem-ui-components/Link', 'Link'),
    Message: localize('@sage/xtrem-ui-components/Message', 'Message'),
    MultiDropdown: localize('@sage/xtrem-ui-components/MultiDropdown', 'Multi dropdown'),
    MultiFileDeposit: localize('@sage/xtrem-ui-components/MultiFileDeposit', 'Multi file deposit'),
    MultiReference: localize('@sage/xtrem-ui-components/MultiReference', 'Multi reference'),
    NestedGrid: localize('@sage/xtrem-ui-components/NestedGrid', 'Nested grid'),
    NodeBrowserTree: localize('@sage/xtrem-ui-components/NodeBrowserTree', 'Node browser tree'),
    Plugin: localize('@sage/xtrem-ui-components/Plugin', 'Plugin'),
    Pod: localize('@sage/xtrem-ui-components/Pod', 'Pod'),
    Reference: localize('@sage/xtrem-ui-components/Reference', 'Reference'),
    RichText: localize('@sage/xtrem-ui-components/RichText', 'Rich text'),
    PodCollection: localize('@sage/xtrem-ui-components/PodCollection', 'Pod collection'),
    Preview: localize('@sage/xtrem-ui-components/Preview', 'Preview'),
    Radio: localize('@sage/xtrem-ui-components/Radio', 'Radio'),
    RelativeDate: localize('@sage/xtrem-ui-components/RelativeDate', 'Relative date'),
    Select: localize('@sage/xtrem-ui-components/Select', 'Select'),
    SelectionCard: localize('@sage/xtrem-ui-components/SelectionCard', 'Selection card'),
    Separator: localize('@sage/xtrem-ui-components/Separator', 'Separator'),
    StaticContent: localize('@sage/xtrem-ui-components/StaticContent', 'Static content'),
    Switch: localize('@sage/xtrem-ui-components/Switch', 'Switch'),
    Table: localize('@sage/xtrem-ui-components/Table', 'Table'),
    StepSequence: localize('@sage/xtrem-ui-components/StepSequence', 'Step sequence'),
    TableSummary: localize('@sage/xtrem-ui-components/TableSummary', 'Table summary'),
    Technical: localize('@sage/xtrem-ui-components/Technical', 'Technical'),
    TechnicalJson: localize('@sage/xtrem-ui-components/TechnicalJson', 'Technical JSON'),
    TextArea: localize('@sage/xtrem-ui-components/TextArea', 'Text area'),
    Time: localize('@sage/xtrem-ui-components/Time', 'Time'),
    ToggleButton: localize('@sage/xtrem-ui-components/ToggleButton', 'Toggle button'),
    Tree: localize('@sage/xtrem-ui-components/Tree', 'Tree'),
    VisualProcess: localize('@sage/xtrem-ui-components/VisualProcess', 'Visual process'),
    VitalPod: localize('@sage/xtrem-ui-components/VitalPod', 'Vital pod'),
    Workflow: localize('@sage/xtrem-ui-components/Workflow', 'Workflow'),
}));
exports.getMainPresentation = (0, lodash_1.memoize)(({ localize, type }) => (0, exports.presentationTranslations)(localize)[xtrem_shared_1.presentationGraphqlMapping[type][0]]);
function tableWidgetReducer() {
    return (0, lodash_1.memoize)((localize) => {
        return (state, action) => {
            switch (action.type) {
                case 'ORDER_CHANGED':
                    return handleOrderChange(state, action);
                case 'DATA_RESET':
                    return handleDataReset(state, action);
                case 'ROW_ADDED':
                    return handleRowAdded(state, action.data);
                case 'ROW_REMOVED':
                    return handleRowRemoved(state, action.row._id);
                case 'ROW_DRAGGED':
                    return handleRowDragged(state, action.ids);
                case 'CELL_CHANGED':
                    return handleCellChanged(state, action, action.selectedProperties, localize);
                case 'COLUMNS_CHANGED':
                    return handleColumnsChanged(state, action);
                default:
                    return state;
            }
        };
    });
}
function enforceContiguousOrder({ orderedData, key, }) {
    const changes = (0, xtrem_shared_1.arrayOrderContiguous)({
        orderedData,
        key,
    });
    return changes.reduce((acc, { rowId, columnId, value }) => {
        const element = acc.find(d => d._id === rowId);
        if (element !== undefined) {
            element[columnId] = value;
        }
        return acc;
    }, orderedData);
}
function areAllPropertiesNonSortable(data) {
    return data.every(row => {
        const { canSort } = row.property?.data ?? {};
        return typeof canSort === 'boolean' && !canSort;
    });
}
function checkValidGroupOrder({ data, enforceContiguous, }) {
    const groups = (0, lodash_1.groupBy)(data, enforceContiguous);
    const invalidFields = [];
    let isValid = true;
    Object.keys(groups).forEach(key => {
        const group = groups[key];
        const element = group[0];
        const name = element?.property?.data?.label;
        const canSort = element?.property?.data?.canSort;
        const isValidGroup = typeof canSort === 'undefined' || canSort;
        if (!isValidGroup) {
            isValid = false;
            if (name) {
                invalidFields.push(name);
            }
        }
    });
    return { isValid, invalidFields };
}
function handleOrderChange(state, { orderedIds, id, enforceContiguous }) {
    const orderedData = (0, lodash_1.sortBy)(state.data, d => (0, lodash_1.findIndex)(orderedIds, element => d[id] === element));
    const data = enforceContiguous !== undefined
        ? enforceContiguousOrder({
            orderedData,
            key: enforceContiguous,
        })
        : orderedData;
    return {
        ...state,
        data,
    };
}
function handleDataReset(state, { data }) {
    const dataLength = data.length;
    return {
        ...state,
        counter: dataLength,
        data: [
            ...data,
            ...state.data
                .filter(row => !isContentRowValid({ row, validations: state.validations }))
                .map((row, index) => ({ ...row, _id: String(index + dataLength + 1) })),
        ],
    };
}
function handleRowAdded(state, data) {
    const maxId = state.data.reduce((max, row) => {
        const rowId = Number(row._id);
        return rowId > max ? rowId : max;
    }, 0);
    const newRowId = String(maxId + 1);
    const newRow = {
        _id: newRowId,
        presentation: undefined,
        title: undefined,
        formatting: undefined,
        divisor: undefined,
        path: '',
        labelPath: undefined,
        property: undefined,
    };
    const newData = [...state.data, { ...newRow, ...data }];
    return {
        ...state,
        counter: newData.length,
        data: newData,
    };
}
function handleColumnsChanged(state, { columns }) {
    return {
        ...state,
        columns,
    };
}
function handleRowRemoved(state, rowId) {
    const newData = state.data.filter(element => element._id !== rowId);
    return {
        ...state,
        counter: newData.length,
        data: newData,
    };
}
function handleRowDragged(state, ids) {
    const orderedRows = (0, lodash_1.sortBy)(state.data, item => {
        return ids.indexOf(item._id);
    });
    return {
        ...state,
        data: orderedRows,
    };
}
function getActualValue({ columnId, value, }) {
    if (columnId === 'property') {
        return value || undefined;
    }
    return (0, lodash_1.isNil)(value) ? undefined : value;
}
function isContentRowValid({ row, validations, }) {
    return (row.property != null &&
        row.presentation != null &&
        row.property?.data?.type != null &&
        row.path != null &&
        row.labelPath != null &&
        row.title != null &&
        Object.values(validations[row._id] ?? {}).every(v => v == null));
}
function handleCellChanged(state, action, selectedProperties, localize) {
    const { rowId, columnId, value, rowData } = action.changes;
    const validationMessage = validateCellChange(rowData, columnId, value, localize);
    const data = state.data.map(row => {
        if (row._id === rowId && !(0, lodash_1.isEqual)((0, lodash_1.get)(row, columnId), value)) {
            // set actual value
            (0, lodash_1.set)(row, columnId, getActualValue({ columnId, value }));
            // set type and path for given property
            if (row.property) {
                const idOrPath = row.property.id ?? row.property.path;
                const prop = selectedProperties?.[idOrPath];
                row.path = idOrPath;
                row.labelPath = prop?.labelPath;
            }
            if (row.property && row.property.data && columnId === 'title') {
                row.property.data.label = value ?? '';
            }
            const columnType = row.property?.data.type;
            // set presentation type in case it cannot be chosen
            if (columnType && xtrem_shared_1.presentationGraphqlMapping[columnType]?.length === 1) {
                row.presentation = xtrem_shared_1.presentationGraphqlMapping[columnType][0];
            }
            if (columnId === 'property') {
                // reset presentation & formatting if property is not compatible
                row.presentation = undefined;
                row.formatting = undefined;
                // always set default title upon property change
                row.title = row.property?.label;
                // always reset divisor upon property change
                row.divisor =
                    row.property?.data?.type && (0, lodash_1.includes)(xtrem_shared_1.numericFields, row.property.data.type) ? '1' : undefined;
            }
            if (columnId === 'presentation') {
                row.formatting = undefined;
            }
            if (!row.presentation && columnType) {
                row.presentation = xtrem_shared_1.presentationGraphqlMapping[columnType][0];
            }
            if (columnId !== 'formatting' &&
                !row.formatting &&
                row.property?.data?.type &&
                (0, lodash_1.includes)(xtrem_shared_1.numericFields, row.property.data.type) &&
                row.presentation === xtrem_shared_1.FieldKey.Numeric) {
                row.formatting = '0';
            }
        }
        return row;
    });
    const changedRow = data.find(d => d._id === rowData._id);
    const validationResets = changedRow
        ? (0, xtrem_shared_1.objectKeys)(changedRow).reduce((acc, key) => {
            if (changedRow?.[key] === undefined) {
                (0, lodash_1.set)(acc, key, undefined);
            }
            return acc;
        }, {})
        : {};
    const validations = {
        ...state.validations,
        [rowData._id]: {
            ...state.validations?.[rowData._id],
            [columnId]: validationMessage,
            ...validationResets,
        },
    };
    return {
        ...state,
        data,
        validations,
    };
}
function validateCellChange(rowData, columnId, value, localize) {
    if (rowData.property?.data?.type &&
        (columnId === 'divisor' || columnId === 'formatting') &&
        (0, lodash_1.includes)(xtrem_shared_1.numericFields, rowData.property.data.type)) {
        const parsedNumber = Number.parseInt(String(value), 10);
        if (Number.isNaN(parsedNumber)) {
            return localize('@sage/xtrem-ui-components/invalid-number', 'Invalid number');
        }
        if (columnId === 'formatting' && (parsedNumber < 0 || parsedNumber > 4)) {
            return localize('@sage/xtrem-ui-components/must-be-between-zero-and-four', 'This value needs to be between 0 and 4.');
        }
        if (columnId === 'divisor' && parsedNumber <= 0) {
            return localize('@sage/xtrem-ui-components/must-be-greater-than-zero', 'This value needs to be greater than 0.');
        }
    }
    return undefined;
}
//# sourceMappingURL=content-table-utils.js.map