"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.mapConditionToEditorRow = exports.isCompleteRow = exports.SplitBox = void 0;
exports.conditionEditorTableReducer = conditionEditorTableReducer;
const box_1 = __importDefault(require("carbon-react/esm/components/box"));
const lodash_1 = require("lodash");
const styled_components_1 = __importDefault(require("styled-components"));
const table_validation_utils_1 = require("../table-validation-utils");
const utils_1 = require("../utils");
exports.SplitBox = (0, styled_components_1.default)(box_1.default) `
    & > div {
        flex: 1;
    }
`;
const isCompleteRow = (row, validations) => row.conjunction != null &&
    row.operator != null &&
    row.value1 != null &&
    row.valueType1 != null &&
    (row.operator === 'empty' || row.operator === 'notEmpty' || (row.value2 != null && row.valueType2 != null)) &&
    !!Object.values(validations[row._id] ?? {}).every(v => v == null);
exports.isCompleteRow = isCompleteRow;
exports.mapConditionToEditorRow = (0, lodash_1.memoize)(({ value, }) => {
    return value.map(({ conjunction, operator, valueType1, valueType2, value1, value2, ...rest }, index) => {
        return {
            // spread needed in case columns are extended
            ...rest,
            key: String(index + 1),
            _id: String(index + 1),
            property: undefined,
            conjunction,
            operator,
            valueType1,
            valueType2,
            value1,
            value2,
        };
    });
});
function conditionEditorTableReducer() {
    return (0, lodash_1.memoize)((localize, parameters) => {
        return (state, action) => {
            switch (action.type) {
                case 'DATA_RESET':
                    return handleDataReset(state, action);
                case 'ROW_ADDED':
                    return handleRowAdded(state);
                case 'ROW_REMOVED':
                    return handleRowRemoved(state, action);
                case 'ROW_DRAGGED':
                    return handleRowDragged(state, action);
                case 'CELL_CHANGED':
                    return handleCellChanged(state, action, localize, parameters);
                default:
                    return state;
            }
        };
    });
}
function handleRowAdded(state) {
    const newRow = state.columns.reduce((acc, curr) => {
        acc[curr.id] = undefined;
        return acc;
    }, { _id: String(state.counter + 1) });
    newRow.conjunction = 'and';
    newRow.valueType1 = 'property';
    newRow.operator = 'equals';
    return {
        ...state,
        counter: state.counter + 1,
        data: [...state.data, newRow],
    };
}
function handleRowRemoved(state, action) {
    return {
        ...state,
        data: state.data.filter(element => element._id !== action.row._id),
    };
}
function handleDataReset(state, { value }) {
    const incompleteRows = state.data.filter(row => !(0, exports.isCompleteRow)(row, state.validations));
    const completeRows = (0, exports.mapConditionToEditorRow)({
        value,
    });
    const completeRowsLength = completeRows.length;
    const data = [
        ...completeRows,
        ...incompleteRows.map((incompleteRow, index) => ({
            ...incompleteRow,
            _id: String(index + completeRowsLength + 1),
        })),
    ];
    return {
        ...state,
        counter: data.length,
        data,
    };
}
function handleRowDragged(state, action) {
    const orderedRows = (0, lodash_1.sortBy)(state.data, item => action.ids.indexOf(item._id));
    return {
        ...state,
        data: orderedRows,
    };
}
function getActualValue({ columnId, value, }) {
    switch (columnId) {
        case '_id':
        case 'key':
        case 'value1':
        case 'value2':
        case 'valueType1':
        case 'valueType2':
            return (0, lodash_1.isNil)(value) ? undefined : value;
        case 'conjunction':
            return value || 'and';
        default:
            return value;
    }
}
function handleCellChanged(state, action, localize, parameters) {
    const { changes: { rowId, columnId, value, rowData }, } = action;
    const isParameter = rowData.valueType1 === 'parameter';
    const isEnum = !isParameter && (0, utils_1.isEnumField)(rowData.value1?.data);
    // If the value is a parameter, the type is the type of the parameter
    const type = isParameter ? parameters?.find(p => p.name === rowData.value1)?.type : rowData.value1?.data?.type;
    const validationMessage = rowData.valueType2 === 'constant' && (columnId === 'value2' || rowData.value2)
        ? (0, table_validation_utils_1.validateScalarValue)({
            type,
            value: columnId === 'value2' ? value : rowData.value2,
            enumValues: isEnum ? rowData.value1.data.enumValues : [],
            localize,
        })
        : undefined;
    const data = state.data.map(curr => {
        if (curr._id !== rowId || (0, lodash_1.isEqual)(curr[columnId], value)) {
            return curr;
        }
        // If the property or value type changes, reset the value
        if (columnId === 'valueType1') {
            curr.value1 = null;
            curr.operator = 'equals';
            curr.valueType2 = 'property';
            curr.value2 = null;
        }
        if (columnId === 'value1') {
            curr.operator = 'equals';
            curr.valueType2 = 'property';
            curr.value2 = null;
        }
        if (columnId === 'valueType2') {
            curr.value2 = null;
        }
        if (columnId === 'operator' && (value === 'empty' || value === 'notEmpty')) {
            curr.valueType2 = 'constant';
        }
        if (!(columnId === 'value1' && value === undefined)) {
            // set actual value
            (0, lodash_1.set)(curr, columnId, getActualValue({ columnId, value }));
        }
        if (!(columnId === 'value2' && value === undefined)) {
            // set actual value
            (0, lodash_1.set)(curr, columnId, getActualValue({ columnId, value }));
        }
        return curr;
    });
    const changedRow = data.find(d => d._id === rowData._id);
    const validationResets = Object.keys(changedRow ?? {}).reduce((acc, key) => {
        if (changedRow?.[key] === undefined) {
            (0, lodash_1.set)(acc, key, undefined);
        }
        return acc;
    }, {});
    const validations = {
        ...state.validations,
        [rowData._id]: {
            ...state.validations?.[rowData._id],
            [columnId]: validationMessage,
            ...validationResets,
        },
    };
    return {
        ...state,
        data,
        validations,
    };
}
//# sourceMappingURL=condition-editor-table-utils.js.map