"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildGetValueNotCompliantError = buildGetValueNotCompliantError;
exports.buildIsClearedByResetNotCompliantError = buildIsClearedByResetNotCompliantError;
exports.buildComputeValueNotCompliantWarning = buildComputeValueNotCompliantWarning;
exports.getErrorWhenConverting = getErrorWhenConverting;
exports.getEnclosingPropertyName = getEnclosingPropertyName;
/**
 * @fileoverview Check property decorators
 * @author Sage
 */
const xtrem_ts_to_sql_1 = require("@sage/xtrem-ts-to-sql");
const utils_1 = require("@typescript-eslint/utils");
const { AST_NODE_TYPES } = utils_1.TSESTree;
function buildGetValueNotCompliantError(propertyName, error) {
    return `The function ${propertyName}.getValue() is too complex to be converted to SQL:  you won't be able to use it in filter and sort operations. Detailed message: ${error.message}`;
}
function buildIsClearedByResetNotCompliantError(propertyName) {
    return `The function ${propertyName}.isClearedByReset() is too complex to be converted to SQL:  you won't be able to use it in filter and sort operations`;
}
function buildComputeValueNotCompliantWarning(propertyName) {
    return `The function ${propertyName}.computeValue() can be parsed to SQL: rename it to getValue()`;
}
function getErrorWhenConverting(fn) {
    try {
        new xtrem_ts_to_sql_1.EsLintConverter({}).convertFunctionExpression(fn);
        return undefined;
    }
    catch (ex) {
        return ex;
    }
}
function checkComputeValueFunction(context, fnToTest) {
    // 'computeValue' functions should not be "ts-to-sql" compliant
    const propertyName = fnToTest.parent.parent.parent.parent.parent.key.name;
    const error = getErrorWhenConverting(fnToTest);
    if (error)
        return;
    context.report({
        node: fnToTest.parent, // fnToTest.parent: puts the warning on the function itself to allow auto-fixes.
        messageId: 'error',
        data: {
            error: buildComputeValueNotCompliantWarning(propertyName),
        },
        fix: (fixer) => {
            // fnToTest.parent.key is the name of the function
            return fixer.replaceText(fnToTest.parent.key, 'getValue');
        },
    });
}
function checkGetValueFunction(context, fnToTest) {
    // 'getValue' functions must be "ts-to-sql" compliant
    const error = getErrorWhenConverting(fnToTest);
    if (!error)
        return;
    const propertyName = getEnclosingPropertyName(fnToTest);
    context.report({
        // fnToTest.parent: puts the warning on the function itself to allow auto-fixes.
        // fnToTest.parent is better for auto-fixes but error.node is better for locating the error.
        node: error.node ?? fnToTest.parent,
        messageId: 'error',
        data: {
            error: buildGetValueNotCompliantError(propertyName, error),
        },
        fix: (fixer) => {
            // fnToTest.parent.key is the name of the function
            return fixer.replaceText(getEnclosingFunctionName(fnToTest), 'computeValue');
        },
    });
}
function checkIsClearedByResetFunction(context, fnToTest) {
    // 'isClearedByReset' functions must be "ts-to-sql" compliant
    const error = getErrorWhenConverting(fnToTest);
    if (!error)
        return;
    const propertyName = getEnclosingPropertyName(fnToTest);
    context.report({
        node: fnToTest.parent, // fnToTest.parent: puts the warning on the function itself to allow auto-fixes.
        messageId: 'error',
        data: {
            error: buildIsClearedByResetNotCompliantError(propertyName),
        },
    });
}
const rule = {
    defaultOptions: [],
    meta: {
        type: 'problem',
        docs: {
            url: '',
            description: 'Check property decorators',
            // recommended: 'strict',
        },
        fixable: 'code', // or "code" or "whitespace"
        schema: [
        // fill in your schema
        ],
        messages: {
            error: '{{error}}',
        },
    },
    create(context) {
        // Retrieve all the property decorators
        const decoratorsSelector = 'PropertyDefinition > Decorator > CallExpression > ObjectExpression';
        return {
            // Rule: computeValue() function can be parsed to SQL: rename it to getValue()"
            [`${decoratorsSelector} > Property[key.name="computeValue"][value.type='FunctionExpression'] > FunctionExpression`](getValueFunction) {
                checkComputeValueFunction(context, getValueFunction);
            },
            // Rule: getValue() functions must be "ts-to-sql compliant"
            [`${decoratorsSelector} > Property[key.name="getValue"][value.type='FunctionExpression'] > FunctionExpression`](getValueFunction) {
                checkGetValueFunction(context, getValueFunction);
            },
            // Rule: isClearedByResetFunction() functions must be "ts-to-sql compliant"
            [`${decoratorsSelector} > Property[key.name="isClearedByReset"][value.type='FunctionExpression'] > FunctionExpression`](isClearedByResetFunction) {
                checkIsClearedByResetFunction(context, isClearedByResetFunction);
            },
        };
    },
};
function getEnclosingPropertyName(fnToTest) {
    const propertyNode = fnToTest.parent?.parent?.parent?.parent?.parent;
    if (!(propertyNode &&
        (propertyNode.type === AST_NODE_TYPES.Property ||
            propertyNode.type === AST_NODE_TYPES.PropertyDefinition) &&
        propertyNode.key.type === AST_NODE_TYPES.Identifier)) {
        throw new Error('cannot find property name');
    }
    return propertyNode.key.name;
}
function getEnclosingFunctionName(fnToTest) {
    const propertyNode = fnToTest.parent;
    if (!(propertyNode &&
        (propertyNode.type === AST_NODE_TYPES.Property ||
            propertyNode.type === AST_NODE_TYPES.PropertyDefinition) &&
        propertyNode.key.type === AST_NODE_TYPES.Identifier)) {
        throw new Error('cannot find function name');
    }
    return propertyNode.key;
}
exports.default = rule;
//# sourceMappingURL=property-decorators-errors.js.map