"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * @fileoverview Adds Promise<...> and Reference<...> to the types of decorated properties.
 * @author Sage
 */
/* TEMPORARY RULE FOR NODE 16 CONVERSION */
const utils_1 = require("@typescript-eslint/utils");
const rule = {
    defaultOptions: [],
    meta: {
        type: 'suggestion',
        docs: {
            url: '',
            description: 'Prepare decorators for async/await',
            // recommended: 'strict',
        },
        fixable: 'code',
        schema: [],
        messages: {
            fixAsyncDecorator: 'Fix decorator for async/await',
            fixPropertyType: 'Wrap property type with {{wrapper}}(...)',
            fixPropertyReadonly: 'Fix missing readonly property modifier',
        },
    },
    create(context) {
        return {
            Decorator(decorator) {
                if (decorator.expression.type !== utils_1.AST_NODE_TYPES.CallExpression)
                    return;
                if (decorator.expression.callee.type !== utils_1.AST_NODE_TYPES.MemberExpression)
                    return;
                if (decorator.expression.callee.object.type !== utils_1.AST_NODE_TYPES.Identifier)
                    return;
                if (decorator.expression.callee.object.name !== 'decorators')
                    return;
                if (decorator.expression.callee.property.type !== utils_1.AST_NODE_TYPES.Identifier)
                    return;
                const decoratorName = decorator.expression.callee.property.name;
                if (decoratorName.endsWith('Property') || decoratorName.endsWith('PropertyOverride')) {
                    if (decorator.parent?.type !== utils_1.AST_NODE_TYPES.PropertyDefinition &&
                        // To investigate: why is this needed? This node type is not in the AST spec.
                        // see https://typescript-eslint.io/packages/typescript-estree/ast-spec/#ast_node_types
                        decorator.parent?.type !== 'ClassProperty')
                        throw new Error(`${decoratorName}: invalid decorator target: ${decorator.parent?.type}`);
                    const nodeProperty = decorator.parent;
                    if (!nodeProperty.typeAnnotation)
                        throw new Error(`${decoratorName}: type annotation missing`);
                    if (!nodeProperty.readonly) {
                        context.report({
                            node: nodeProperty,
                            messageId: 'fixPropertyReadonly',
                            fix: fixer => {
                                return fixer.insertTextBefore(nodeProperty.key, 'readonly ');
                            },
                        });
                    }
                    if (decoratorName.startsWith('collectionProperty'))
                        return;
                    const typeAnnotation = nodeProperty.typeAnnotation.typeAnnotation;
                    const expectedTypeWrapper = decoratorName.startsWith('referenceProperty') ? 'Reference' : 'Promise';
                    if (typeAnnotation.type === utils_1.AST_NODE_TYPES.TSTypeReference &&
                        typeAnnotation.typeName.type === utils_1.AST_NODE_TYPES.Identifier &&
                        /^(Promise|Reference)$/.test(typeAnnotation.typeName.name))
                        return;
                    context.report({
                        node: typeAnnotation,
                        messageId: 'fixPropertyType',
                        data: { wrapper: expectedTypeWrapper },
                        fix: (fixer) => {
                            const typeText = context.getSourceCode().getText(typeAnnotation);
                            return fixer.replaceText(typeAnnotation, `${expectedTypeWrapper}<${typeText}>`);
                        },
                    });
                }
            },
        };
    },
};
exports.default = rule;
//# sourceMappingURL=fix-async-decorator.js.map