"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebsocketAWSRepository = void 0;
const client_sqs_1 = require("@aws-sdk/client-sqs");
const client_dynamodb_1 = require("@aws-sdk/client-dynamodb");
const xtrem_deployment_1 = require("@sage/xtrem-deployment");
const websocketRepository_1 = require("../websocketRepository");
const DEFAULT_SOURCE_VALUE = "xtreem";
const MAX_MESSAGE_SIZE_IN_BYTES = 1024 * 128;
class WebsocketAWSRepository extends websocketRepository_1.WebsocketRepository {
    constructor(config) {
        super();
        this.config = config;
        this.tracer = this.config.tracer || console.log;
        this.SQS = (this.config.AWSOverrideForUnitTesting && this.config.AWSOverrideForUnitTesting.SQS) || new client_sqs_1.SQS(config.awsConfigOverride || {});
        this.DynamoDB = (this.config.AWSOverrideForUnitTesting && this.config.AWSOverrideForUnitTesting.DynamoDB) ?
            this.config.AWSOverrideForUnitTesting.DynamoDB() :
            new client_dynamodb_1.DynamoDB(config.awsConfigOverride || {});
        if (!this.config.source) {
            this.config.source = DEFAULT_SOURCE_VALUE;
        }
    }
    async isUserOfTenantConnected(connectedUserFilter, acrossAllApps = false) {
        const dbParams = {
            TableName: this.config.dynamoDBTableName,
            IndexName: "GSI1",
            KeyConditionExpression: `#tenantId = :TENANT  and #userId = :USER`,
            ExpressionAttributeNames: {
                "#tenantId": "tenantId",
                "#userId": "userId"
            },
            ExpressionAttributeValues: {
                ":TENANT": { S: connectedUserFilter.tenantId },
                ":USER": { S: connectedUserFilter.userId }
            }
        };
        const result = await this.DynamoDB.query(dbParams);
        if (result && result.Items && result.Items.length > 0) {
            return result.Items.some(item => {
                const id = (item.clusterId && item.clusterId.S) || "";
                const clusterAppIdentifier = (0, xtrem_deployment_1.getClusterAppIdentifier)(id);
                const isMatchingApp = acrossAllApps ? true : clusterAppIdentifier.app === this.config.app;
                // ensure cluster and app matches
                return clusterAppIdentifier.cluster === this.config.clusterId && isMatchingApp;
            });
        }
        else {
            // not connected
            return false;
        }
    }
    async broadcastMessage(sendTo, payload) {
        const messageContentStringified = JSON.stringify(payload);
        const messageSize = Buffer.byteLength(Buffer.from(messageContentStringified, "utf-8"));
        if (messageSize > MAX_MESSAGE_SIZE_IN_BYTES) {
            throw new Error("Cannot send the notification message, the payload size is too important (max payload is 128 KB)");
        }
        const sendMessageParams = {
            MessageBody: JSON.stringify(payload),
            MessageAttributes: this.generateSQSBroadcastMessageAttributes(sendTo),
            QueueUrl: this.config.websocketQueueUrl
        };
        try {
            await this.SQS.sendMessage(sendMessageParams);
        }
        catch (err) {
            const msg = { message: "", stack: "" };
            if (err instanceof Error) {
                msg.message = err.message;
                msg.stack = err.stack || "";
            }
            throw new Error(`Failed to broadcast message to SQS for tenant : ${sendTo.tenantId}. Err ${msg.stack !== "" ? msg.stack : msg.message}`);
        }
    }
    async sendUserNotification(sendTo, payload) {
        const messageContentStringified = JSON.stringify(payload);
        const messageSize = Buffer.byteLength(Buffer.from(messageContentStringified, "utf-8"));
        if (messageSize > MAX_MESSAGE_SIZE_IN_BYTES) {
            throw new Error("Cannot send the notification message, the payload size is too important (max payload is 128 KB)");
        }
        const sendMessageParams = {
            MessageBody: JSON.stringify(payload),
            MessageAttributes: this.generateSQSMessageAttributes(sendTo),
            QueueUrl: this.config.websocketQueueUrl
        };
        try {
            await this.SQS.sendMessage(sendMessageParams);
        }
        catch (err) {
            const msg = { message: "", stack: "" };
            if (err instanceof Error) {
                msg.message = err.message;
                msg.stack = err.stack || "";
            }
            throw new Error(`Failed to send message to SQS for user : ${JSON.stringify(sendTo)}. Err ${msg.stack !== "" ? msg.stack : msg.message}`);
        }
    }
    generateSQSMessageAttributes(sendTo) {
        return {
            "tenantId": {
                DataType: "String",
                StringValue: sendTo.tenantId
            },
            "userEmail": {
                DataType: "String",
                StringValue: sendTo.userId
            },
            "clusterId": {
                DataType: "String",
                StringValue: (0, xtrem_deployment_1.getClusterAppDynamoDBIdentifier)(this.config.clusterId, this.config.app)
            },
            "source": {
                DataType: "String",
                StringValue: this.config.source
            },
            "targetKind": {
                DataType: "String",
                StringValue: "WEB_USER"
            },
            "allUserSessions": {
                DataType: "String",
                StringValue: `${sendTo.sendToAllMatchingUserSessions}`
            },
            "allUserApp": {
                DataType: "String",
                StringValue: sendTo.sendToAllUserApps ? "true" : "false"
            }
        };
    }
    generateSQSBroadcastMessageAttributes(sendTo) {
        return {
            "tenantId": {
                DataType: "String",
                StringValue: sendTo.tenantId
            },
            "clusterId": {
                DataType: "String",
                StringValue: (0, xtrem_deployment_1.getClusterAppDynamoDBIdentifier)(this.config.clusterId, this.config.app)
            },
            "source": {
                DataType: "String",
                StringValue: this.config.source
            },
            "targetKind": {
                DataType: "String",
                StringValue: "ALL_WEB_USER"
            }
        };
    }
    generateSQSMessageThirdPartyAttributes(sendTo) {
        return {
            "tenantId": {
                DataType: "String",
                StringValue: sendTo.tenantId
            },
            "applicationId": {
                DataType: "String",
                StringValue: sendTo.applicationId
            },
            "clusterId": {
                DataType: "String",
                StringValue: (0, xtrem_deployment_1.getClusterAppDynamoDBIdentifier)(this.config.clusterId, this.config.app)
            },
            "source": {
                DataType: "String",
                StringValue: this.config.source
            },
            "targetKind": {
                DataType: "String",
                StringValue: "APPLICATION"
            },
            "allUserApp": {
                DataType: "String",
                StringValue: sendTo.sendToAllUserApps ? "true" : "false"
            }
        };
    }
    async isThirdPartyAppConnected(connectedAppFilter, acrossAllApps = false) {
        return this.isUserOfTenantConnected({
            userId: `api-${connectedAppFilter.applicationId.toLowerCase()}@localhost.domain`,
            tenantId: connectedAppFilter.tenantId
        }, acrossAllApps);
    }
    async sendThirdPartyAppNotification(sendTo, payload) {
        const messageContentStringified = JSON.stringify(payload);
        const messageSize = Buffer.byteLength(Buffer.from(messageContentStringified, "utf-8"));
        if (messageSize > MAX_MESSAGE_SIZE_IN_BYTES) {
            throw new Error("Cannot send the notification message, the payload size is too important (max payload is 128 KB)");
        }
        const sendMessageParams = {
            MessageBody: JSON.stringify(payload),
            MessageAttributes: this.generateSQSMessageThirdPartyAttributes(sendTo),
            QueueUrl: this.config.websocketQueueUrl
        };
        try {
            await this.SQS.sendMessage(sendMessageParams).then();
        }
        catch (err) {
            const msg = { message: "", stack: "" };
            if (err instanceof Error) {
                msg.message = err.message;
                msg.stack = err.stack || "";
            }
            throw new Error(`Failed to send message to SQS for 3rd party app : ${JSON.stringify(sendTo)}. Err ${msg.stack !== "" ? msg.stack : msg.message}`);
        }
    }
}
exports.WebsocketAWSRepository = WebsocketAWSRepository;
