"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Request = void 0;
const json_to_graphql_query_1 = require("json-to-graphql-query");
const client_error_1 = require("./client-error");
function unwrapPath(path, data) {
    return path.reduce((r, k) => r[k], data);
}
function unwrapResult(obj) {
    if (!obj || typeof obj !== 'object')
        return obj;
    if (Array.isArray(obj))
        return obj.map(unwrapResult);
    return Object.keys(obj).reduce((r, k) => {
        r[k] = unwrapResult(obj[k]);
        return r;
    }, {});
}
function unwrap(path, data) {
    if (!data.data)
        throw new Error(`request failed: ${JSON.stringify(data.errors)}`);
    return unwrapPath(path, data.data);
}
class Request {
    constructor(config, path, name, selector, isMutation) {
        this.config = config;
        this.path = path;
        this.name = name;
        this.selector = selector;
        this.isMutation = isMutation;
    }
    async execute() {
        const query = (0, json_to_graphql_query_1.jsonToGraphQLQuery)(this.selector, { pretty: true });
        if (this.config.log)
            this.config.log('GRAPHQL REQUEST', query);
        const result = await this.config.fetcher(query, this.isMutation, this.selector);
        if (result.errors)
            throw new client_error_1.ClientError(result.errors);
        if (this.config.log)
            this.config.log('GRAPHQL RESULT', JSON.stringify(result, null, 2));
        const unwrapped = unwrap(this.path, unwrapResult(result));
        return this.name ? unwrapped[this.name] : unwrapped;
    }
    /** @internal */
    unwrapBatchResult(data) {
        const unwrapped = unwrapPath(this.path.slice(1), data);
        return this.name ? unwrapped[this.name] : unwrapped;
    }
}
exports.Request = Request;
//# sourceMappingURL=request.js.map