"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.withoutEdges = withoutEdges;
exports.extractEdges = extractEdges;
exports.convertPagingOptions = convertPagingOptions;
exports.edgesSelector = edgesSelector;
exports.querySelector = querySelector;
/**
 * Recursively transforms "edges":
 *
 * Array<{edges: { node: T }}> => T[]
 *
 * "query", "queryAggregate" & "readAggregate" are PRESERVED.
 *
 * @export
 * @template T
 * @param {Edges<T>} edges
 * @returns {WithoutEdges<T>[]}
 */
function withoutEdges(edges) {
    const transform = (val) => {
        if (!val || typeof val !== 'object')
            return val;
        // if (val.query) return transform(val.query);
        if (Array.isArray(val))
            return val.map(transform);
        if (Array.isArray(val.edges))
            return val.edges.map((v) => transform(v.node));
        return Object.keys(val).reduce((r, k) => {
            r[k] = transform(val[k]);
            return r;
        }, {});
    };
    return transform(edges);
}
/**
 * Recursively extract "edges" from queries:
 *
 * * {
 *     query: Array<{edges: { node: T }}>;
 *     readAggregate: AggregateResultValues<T>;     =>      T[]
 *     queryAggregate: AggregateEdges<T>;
 * }
 *
 * "query", "queryAggregate" & "readAggregate" are REMOVED.
 * Use "withoutEdges" if you want to keep them.
 * @export
 * @template T
 * @param {Edges<T>} edges
 * @returns {ExtractEdges<T>[]}
 */
function extractEdges(edges) {
    const transform = (val) => {
        if (!val || typeof val !== 'object')
            return val;
        if (val.query)
            return transform(val.query);
        if (Array.isArray(val))
            return val.map(transform);
        if (Array.isArray(val.edges))
            return val.edges.map((v) => transform(v.node));
        return Object.keys(val).reduce((r, k) => {
            r[k] = transform(val[k]);
            return r;
        }, {});
    };
    return transform(edges);
}
/** @internal */
function convertPagingOptions(args) {
    const converted = {};
    if (args.first)
        converted.first = args.first;
    if (args.after)
        converted.after = args.after;
    if (args.last)
        converted.last = args.last;
    if (args.before)
        converted.before = args.before;
    if (args.filter)
        converted.filter = JSON.stringify(args.filter);
    if (args.orderBy)
        converted.orderBy = JSON.stringify(args.orderBy);
    return Object.keys(converted).length > 0 ? converted : undefined;
}
function edgesSelector(selector, args) {
    return {
        __args: convertPagingOptions(args || {}),
        edges: { node: selector, cursor: true },
        pageInfo: { hasNextPage: true, endCursor: true, hasPreviousPage: true, startCursor: true },
    };
}
function querySelector(selector, args) {
    return {
        query: {
            __args: convertPagingOptions(args || {}),
            edges: { node: selector, cursor: true },
            pageInfo: { hasNextPage: true, endCursor: true, hasPreviousPage: true, startCursor: true },
        },
    };
}
//# sourceMappingURL=paging.js.map