import { ClientDiagnoseSeverity } from './client-error';
import { Config } from './config';
import { Request } from './request';
import { OnlySelected, Selector } from './selector';
export type Operation<ArgT, T> = <SelectorT extends Selector<T extends (infer U)[] ? U : T>>(selector: SelectorT, args: ArgT) => Request<OnlySelected<T, SelectorT>>;
export type RunToCompletionOperation<ArgT, T> = <SelectorT extends Selector<T extends (infer U)[] ? U : T>>(selector: SelectorT, args: ArgT, options?: {
    callback?: (arg: {
        pollCount: number;
        elapsedMillis: number;
        status: string;
    }) => boolean;
}) => Request<OnlySelected<T, SelectorT>>;
/**
 * Represents an asynchronous operation with methods to start, track, stop, and run the operation to completion.
 *
 * @template ArgT - The type of the arguments passed to the operation.
 * @template T - The type of the result returned by the operation.
 *
 * @property {Operation<ArgT, { trackingId: string }>} start - Initiates the operation and returns a tracking ID.
 * @property {Operation<{ trackingId: string }, T>} track - Tracks the progress of the operation using a tracking ID.
 * @property {Operation<{ trackingId: string; reason?: string }, boolean>} stop - Stops the operation, optionally providing a reason.
 * @property {Operation<ArgT, T>} runToCompletion - Runs the operation to completion and returns the final result.
 */
export interface AsyncTrackReturn<T> {
    status: string;
    result: T;
    errorMessage: string;
    logMessages: {
        level: string;
        message: string;
    }[];
}
export interface AsyncOperation<ArgT, T> {
    start: Operation<ArgT, {
        trackingId: string;
    }>;
    track: Operation<{
        trackingId: string;
    }, AsyncTrackReturn<T>>;
    stop: Operation<{
        trackingId: string;
        reason?: string;
    }, boolean>;
    requestUserNotification: Operation<{
        trackingId: string;
    }, boolean>;
    runToCompletion: RunToCompletionOperation<ArgT, T>;
}
/** Extract the type of the result of the operation */
export type OperationResultType<Op> = Op extends Operation<any, infer T> ? T : never;
/** Extract the type of the result of the operation */
export type OperationParamType<Op> = Op extends Operation<infer T, any> ? T : never;
export type ExtractArray<T> = T extends Array<infer U> ? U : T;
export type AsyncMutationStatus = 'pending' | 'running' | 'success' | 'error';
export interface PollingOptions {
    callback?(arg: {
        pollCount: number;
        elapsedMillis: number;
        status: AsyncMutationStatus;
    }): boolean;
}
export interface AsyncMutationResponse<T> {
    status: AsyncMutationStatus;
    result: T;
    errorMessage?: string;
    logMessages: {
        message: string;
        level: keyof typeof ClientDiagnoseSeverity;
    }[];
}
export declare const runToCompletionOperation: <ArgT, T>(config: Config, path: string[]) => <SelectorT extends Selector<T>>(selector: SelectorT, args: ArgT, options?: PollingOptions) => {
    execute: () => Promise<OnlySelected<T, SelectorT>>;
};
//# sourceMappingURL=operations.d.ts.map