"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Logger = exports.getDate = exports.rethrow = void 0;
const fs = require("fs");
const logform_1 = require("logform");
const fsp = require("path");
const winston = require("winston");
const DailyRotateFile = require("winston-daily-rotate-file");
const async_local_storage_1 = require("./async-local-storage");
const colors_1 = require("./colors");
const console_transport_1 = require("./console-transport");
const rootLoggerSourceFilename = '$$';
const logLevelsOrders = {
    // WARNING : any changes to this object should be reflected to @sage/xtrem-shared/LogLevel
    off: 0,
    error: 1,
    warn: 2,
    info: 3,
    verbose: 4,
    debug: 5,
};
/** Default error handler for `logger.do`. Just rethrows */
const rethrow = err => {
    throw err;
};
exports.rethrow = rethrow;
/** We need a function for testability */
/* istanbul ignore next */
const getDate = () => new Date();
exports.getDate = getDate;
class Logger {
    static { this._globalLogger = {}; }
    static { this._allLoggers = {}; }
    static { this._allLoggersByDomain = {}; }
    static { this._nextEventId = 0; }
    static { this._startBannerDisplayed = false; }
    static { this.consoleFormatter = logform_1.format.printf((info) => {
        const context = (0, async_local_storage_1.clsContext)()?.context?.context;
        let errorHint;
        if (info.level === 'error' && context?.request != null) {
            errorHint = context.getRequestHint?.();
        }
        if (Logger._logAsJson) {
            const payload = {
                tenantId: context?.tenantId,
                userId: context?.userId,
                originId: context?.originId,
                eventId: info.paddedEventId,
                datetime: info.paddedDate,
                logLevel: info.paddedLogLevel == null ? '' : info.paddedLogLevel.toString().trim(),
                domain: info.paddedFullDomain == null ? '' : info.paddedFullDomain.toString().trim(),
                errorHint,
                source: context?.source,
                message: info.message,
            };
            if (Logger._app != null) {
                payload.app = Logger._app;
            }
            /* istanbul ignore next */
            return JSON.stringify(payload);
        }
        return [
            context?.tenantId?.slice(-5) || '-'.repeat(5),
            process.pid,
            info.paddedEventId,
            info.paddedDate,
            ...[
                info.paddedLogLevel,
                info.paddedFullDomain,
                errorHint != null ? `[${errorHint}] ${info.message}` : info.message,
            ].map(arg => (0, console_transport_1.colorizeText)(info.level, arg ?? '')),
        ].join(' | ');
    }); }
    static { this.fileFormatter = logform_1.format.printf((info) => [
        process.pid,
        info.paddedEventId,
        info.paddedDate,
        info.paddedLogLevel,
        info.paddedFullDomain,
        colors_1.Colors.clean(info.message),
    ].join(' | ')); }
    // Are all logs disabled?
    static { this.isDisabled = false; }
    constructor(sourceFilename, domain) {
        this.domain = domain;
        this._logLevel = 'info';
        this.initialized = false;
        this.timer = {
            invocationCount: 0,
            totalDuration: 0,
            sumOfSquares: 0,
        };
        if (!sourceFilename)
            throw new Error('Invalid call : __filename must be provided');
        if (sourceFilename === rootLoggerSourceFilename) {
            // rootLogger
            this.moduleName = '';
        }
        else {
            // extract domain from the package.json
            this.moduleName = Logger.getPackageName(fsp.dirname(sourceFilename));
        }
    }
    static get globalLogger() {
        return this._globalLogger[String(process.pid)];
    }
    static set globalLogger(logger) {
        this._globalLogger[String(process.pid)] = logger;
    }
    nopFunction() {
        return {
            success: () => { },
            fail: (message) => {
                this.error(message);
            },
        };
    }
    /**
     * Returns the name of the package that contains the folder.
     * The result will be sth like sage/xtrem-core
     * @param folder
     */
    static getPackageName(folder) {
        const packageFilename = 'package.json';
        let folderToTest = folder;
        while (true) {
            const path = fsp.join(folderToTest, packageFilename);
            if (fs.existsSync(path)) {
                try {
                    // eslint-disable-next-line n/global-require, import/no-dynamic-require, @typescript-eslint/no-require-imports
                    const pck = require(path);
                    const name = pck.name;
                    if (name.startsWith('@')) {
                        return name.slice(1); // remove the '@' prefix
                    }
                    return name;
                }
                catch {
                    return '';
                }
            }
            if (path === packageFilename) {
                // We have reached the top folder. No need to go further
                return '';
            }
            const parentFolder = fsp.join(folderToTest, '..');
            if (parentFolder === folderToTest)
                return '';
            folderToTest = parentFolder;
        }
    }
    /** internal */
    static get rootLogger() {
        return Logger.getLogger(rootLoggerSourceFilename, '');
    }
    static reloadConfig(config) {
        this.logsConfig = config.logs;
        Logger._logAsJson = config.deploymentMode === 'production' || this.logsConfig?.options?.json;
        if (config.env?.isCI) {
            // Disable the JSON logs when running on CI
            Logger._logAsJson = false;
            Logger.rootLogger.info(`json log has been disabled because the service is running on CI: TF_BUILD='${process.env.TF_BUILD}'`);
        }
        let fileTransportCreated = false;
        let outputFolder;
        // We have to declare a static winston logger, all the xtrem loggers will log to this unique winston
        // logger (this way, we will only have one log file).
        // By default, logs are written in the folder that contains the config file.
        this.initGlobalLogger();
        if (this.globalLogger.transports.length === 1) {
            // For now, there is only the one transport : the console transport
            // Now that we have a config, we can add a file transport
            outputFolder =
                (this.logsConfig && this.logsConfig.outputFolder) || fsp.join(config.originFolder || __dirname, 'logs');
            try {
                fs.mkdirSync(outputFolder);
                // eslint-disable-next-line no-empty
            }
            catch { }
            this.globalLogger.add(new DailyRotateFile({
                dirname: outputFolder,
                filename: `${(this.logsConfig && this.logsConfig.filenamePrefix) || 'xtrem.server'}-${process.pid}-%DATE%.log`,
                maxSize: 200 * 1024 * 1024, // 200 Mb
                maxFiles: 30,
                json: false,
                format: this.fileFormatter,
            }));
            fileTransportCreated = true;
        }
        const logsAreEmpty = !Logger._startBannerDisplayed;
        this.displayStartBanner();
        if (!logsAreEmpty)
            Logger.rootLogger.info('Configuration reloaded');
        if (fileTransportCreated)
            Logger.rootLogger.info(`Logs for process ${process.pid} will be written in folder : ${outputFolder}`);
        // Now, update the level of all the already created logs
        if (!this.logsConfig)
            return;
        const domains = this.logsConfig.domains || {};
        Object.keys(domains).forEach(domain => {
            if (this._allLoggersByDomain[domain]) {
                this._allLoggersByDomain[domain]._logLevel = domains[domain].level;
            }
        });
    }
    static initGlobalLogger() {
        if (!Logger.globalLogger) {
            // Initialization of the first logger
            const transports = [
                new console_transport_1.ConsoleTransport({
                    format: Logger.logsConfig?.options?.noColor ? this.fileFormatter : this.consoleFormatter,
                }),
            ];
            Logger.globalLogger = winston.createLogger({
                level: 'silly', // log levels are handled by our API
                transports,
            });
        }
    }
    initIfNeeded() {
        if (this.initialized)
            return;
        this.initialized = true;
        Logger.initGlobalLogger();
        // Try to retrieve the logLevel from the config file
        if (Logger.logsConfig) {
            const domains = Logger.logsConfig.domains;
            const key = this.fullDomain;
            if (domains && domains[key]) {
                const domainLevel = domains[key].level;
                if (logLevelsOrders[domainLevel] === null) {
                    throw new Error(`Logger ${key} : invalid log level : ${domains[key].level}`);
                }
                this._logLevel = domains[key].level;
                if (logLevelsOrders[this._logLevel] < logLevelsOrders.info) {
                    Logger.rootLogger.warn(`Logger '${key}' : level retrograded to '${this._logLevel}' by configuration.`);
                }
            }
        }
        Logger.displayStartBanner();
    }
    static displayStartBanner() {
        if (this._startBannerDisplayed)
            return;
        this._startBannerDisplayed = true;
        Logger.rootLogger.info('************************ SERVER STARTED ************************');
    }
    get fullDomain() {
        return this.moduleName ? `${this.moduleName}/${this.domain}` : '';
    }
    get logLevel() {
        this.initIfNeeded();
        return this._logLevel;
    }
    set logLevel(level) {
        this.initIfNeeded();
        if (logLevelsOrders[level] < logLevelsOrders[this._logLevel] && this.initialized) {
            Logger.rootLogger.warn(`Logger '${this.fullDomain}' : level retrograded to '${level}'`);
        }
        this._logLevel = level;
    }
    /**
     * Creates a new logger.
     * @param sourceFilename  should be __filename
     * @param domain
     */
    static getLogger(sourceFilename, domain) {
        const key = `${Logger.getPackageName(sourceFilename)}/${domain}`;
        let logger = Logger._allLoggers[key];
        if (!logger) {
            logger = new Logger(sourceFilename, domain);
            Logger._allLoggers[key] = logger;
            Logger._allLoggersByDomain[logger.fullDomain] = logger;
        }
        return logger;
    }
    /**
     * Logs a message with a provided log level
     */
    _log(logLevel, messageOrCallback, options) {
        this.initIfNeeded();
        if (Logger.isDisabled)
            return this.nopFunction();
        if (logLevelsOrders[this._logLevel] < logLevelsOrders[logLevel])
            return this.nopFunction();
        if (typeof messageOrCallback === 'function')
            this._internalLog({
                logLevel,
                messageOrError: messageOrCallback(),
                ignoreCallback: options.ignoreCallback,
            });
        else
            this._internalLog({ logLevel, messageOrError: messageOrCallback, ignoreCallback: options.ignoreCallback });
        return this.getLogFunctionCallback(logLevel, options);
    }
    /**
     * Logs a message with a provided log level - async version
     */
    async _logAsync(logLevel, messageOrCallback, options) {
        this.initIfNeeded();
        if (Logger.isDisabled)
            return this.nopFunction();
        if (logLevelsOrders[this._logLevel] < logLevelsOrders[logLevel])
            return this.nopFunction();
        if (typeof messageOrCallback === 'function') {
            this._internalLog({
                logLevel,
                messageOrError: await messageOrCallback(),
                ignoreCallback: options.ignoreCallback,
            });
        }
        else
            this._internalLog({
                logLevel,
                messageOrError: await messageOrCallback,
                ignoreCallback: options.ignoreCallback,
            });
        return this.getLogFunctionCallback(logLevel, options);
    }
    /**
     * Logs a message with a provided log level
     */
    log(logLevel, messageProvider, options = {}) {
        return this._log(logLevel, messageProvider, options);
    }
    /**
     * Logs a message with a provided log level - async version
     */
    logAsync(logLevel, messageProvider, options = {}) {
        return this._logAsync(logLevel, messageProvider, options);
    }
    /**
     * Logs a message with the 'info' level (level=3)
     */
    info(messageOrCallback, options = {}) {
        return this._log('info', messageOrCallback, options);
    }
    /**
     * Logs a message with the 'info' level (level=3) - async version
     */
    infoAsync(messageOrCallback, options = {}) {
        return this._logAsync('info', messageOrCallback, options);
    }
    /**
     * Logs a message with the 'warn' level (level=2)
     */
    warn(messageOrCallback, options = {}) {
        return this._log('warn', messageOrCallback, options);
    }
    /**
     * Logs a message with the 'warn' level (level=2) - async version
     */
    warnAsync(messageOrCallback, options = {}) {
        return this._logAsync('warn', messageOrCallback, options);
    }
    /**
     * Logs a message with the 'debug' level (level=5)
     */
    debug(messageProvider, options = {}) {
        return this._log('debug', messageProvider, options);
    }
    /**
     * Logs a message with the 'debug' level (level=5)
     */
    debugAsync(messageProvider, options = {}) {
        return this._logAsync('debug', messageProvider, options);
    }
    /**
     * Logs a message with the 'verbose' level (level=4)
     */
    verbose(messageProvider, options = {}) {
        return this._log('verbose', messageProvider, options);
    }
    /**
     * Logs a message with the 'verbose' level (level=4) - async version
     */
    verboseAsync(messageProvider, options = {}) {
        return this._logAsync('verbose', messageProvider, options);
    }
    /**
     * Logs a message with the 'error' level (level=1)
     */
    error(messageOrCallback, options = {}) {
        return this._log('error', messageOrCallback, options);
    }
    /**
     * Logs a message with the 'error' level (level=1) - async version
     * @param messageOrCallback
     */
    errorAsync(messageOrCallback, options = {}) {
        return this._logAsync('error', messageOrCallback, options);
    }
    getLogFunctionCallback(logLevel, options) {
        const profilingStartDate = (0, exports.getDate)().getTime();
        const eventId = Logger._nextEventId;
        return {
            success: (message) => {
                const duration = (0, exports.getDate)().getTime() - profilingStartDate;
                this.timer.invocationCount += 1;
                this.timer.totalDuration += duration;
                this.timer.sumOfSquares += duration * duration;
                let colorizedDuration = `(${duration} ms)`;
                if (options.lowThreshold != null) {
                    if (options.highThreshold == null) {
                        // only one threshold, consider it as a maximum duration
                        if (duration > options.lowThreshold)
                            colorizedDuration = colors_1.Colors.red(colorizedDuration);
                    }
                    else if (duration > options.highThreshold)
                        colorizedDuration = colors_1.Colors.red(colorizedDuration);
                    // 2 thresholds : consider them as a range [lowThreshold..highThreshold]
                    else if (duration > options.lowThreshold)
                        colorizedDuration = colors_1.Colors.yellow(colorizedDuration);
                }
                else
                    colorizedDuration = colors_1.Colors.default(colorizedDuration);
                if (this.logLevel === 'debug') {
                    const msg = `Event #${eventId}${message ? ` : ${message}` : ' is over'}`;
                    this._internalLog({ logLevel, messageOrError: `${msg} ${colorizedDuration}`, eventId });
                }
            },
            fail: (message) => {
                const duration = (0, exports.getDate)().getTime() - profilingStartDate;
                const msg = `Event #${eventId}${message ? ` : ${message}` : ' failed'}`;
                this._internalLog({ logLevel: 'error', messageOrError: `${msg} (${duration} ms)`, eventId });
            },
        };
    }
    isActive(logLevel) {
        return logLevelsOrders[this._logLevel] >= logLevelsOrders[logLevel];
    }
    static _messageOrErrorToString(messageOrError) {
        if (messageOrError instanceof Error) {
            if (messageOrError.stack)
                return messageOrError.stack;
            return `${messageOrError.name}: ${messageOrError.message}`;
        }
        return messageOrError;
    }
    _internalLog(options) {
        function padRight(str, pad) {
            return str.padEnd(pad).substring(0, pad);
        }
        function padLeft(str, pad) {
            return str.padStart(pad, '0').substring(0, pad);
        }
        const messageToLog = Logger._messageOrErrorToString(options.messageOrError);
        if (options.ignoreCallback && options.ignoreCallback(messageToLog))
            return;
        // Note 1 : if we use timestamps of Winston tranports, the timestamp in the console may differ (some ms) from
        // the timestamp written in the log file. That's why we declare a common timestamp that will be used by all
        // the transports.
        // Note 2 : all the items that will be used by the formatter must be padded so that logs are well aligned.
        // eslint-disable-next-line no-plusplus
        const evtId = options.eventId || ++Logger._nextEventId;
        const entryMetadata = {
            paddedEventId: padLeft(`${evtId}`, 6),
            paddedFullDomain: padRight(this.fullDomain.replace(/^sage\//, ''), 30),
            paddedLogLevel: padRight(options.logLevel.toString().toUpperCase(), 7),
            paddedDate: (0, exports.getDate)().toISOString().substring(11, 23), // No need to log the date, the log files are prefixed with date
        };
        Logger.globalLogger.log(options.logLevel.toString(), messageToLog, entryMetadata);
    }
    get statistics() {
        const average = (x) => (this.timer.invocationCount === 0 ? Number.NaN : x / this.timer.invocationCount);
        const mean = average(this.timer.totalDuration);
        const meanOfSquares = average(this.timer.sumOfSquares);
        return {
            count: this.timer.invocationCount,
            totalDuration: this.timer.totalDuration,
            meanDuration: mean,
            standardDeviation: Math.sqrt(meanOfSquares - mean * mean),
        };
    }
    /**
     * result = do(fn, onError);
     *
     * Executes `fn()` silently.
     * But if `fn()` fails, logs the stacktrace and rethrows.
     *
     * `onError(err)` is an optional callback which will be invoked if `fn()` fails.
     * By default, `onError` rethrows the error, but you can rethrow a different error, typically
     * to mask low level details, or add some context to the error.
     * You can also use `onError` to stop the error propagation and return a special value.
     */
    do(fn, onError = exports.rethrow) {
        try {
            return fn();
        }
        catch (err) {
            this.error(err.stack);
            return onError(err);
        }
    }
    /**
     * Async variant of do(fn, onError)
     */
    async doAsync(fn, options = {}) {
        try {
            return await fn();
        }
        catch (err) {
            this.error(err.stack, { ignoreCallback: options.ignoreCallback });
            if (options.onError)
                return options.onError(err);
            throw err;
        }
    }
    static setAppName(app) {
        this._app = app;
    }
    /**
     * Logger.disable()
     *
     * Disables all logs.
     * This method is called before running unit tests when logs/disabledForTests is true.
     */
    static disable() {
        Logger.isDisabled = true;
    }
    static isLogAsJson() {
        return !!Logger._logAsJson;
    }
    static noBanner() {
        Logger._startBannerDisplayed = true; // do not display the start banner
    }
}
exports.Logger = Logger;
//# sourceMappingURL=logger.js.map