import { objectKeys } from '@sage/xtrem-shared';
import { FieldKey } from '@sage/xtrem-shared';
import { navigationPanelId } from '../component/container/navigation-panel/navigation-panel-types';
import { AbstractUiControlObject, SectionControlObject } from '../component/control-objects';
import * as xtremRedux from '../redux';
import { xtremConsole } from './console';
import { convertDeepBindToPathNotNull } from './nested-field-utils';
import { resolveByValue } from './resolve-value-utils';
import { getCustomFields } from '../service/customization-service';
import { mergeIntoSidebarLayout } from '../component/abstract-decorator-utils';
import { getParentSection } from './abstract-fields-utils';
import { isEqual } from 'lodash';
export const isScreenDefinitionDirty = (screenDefinition) => !screenDefinition.metadata.uiComponentProperties[screenDefinition.metadata.screenId].skipDirtyCheck &&
    !!objectKeys(screenDefinition.dirtyStates).find(key => screenDefinition.dirtyStates[key] === true) &&
    screenDefinition.type !== 'sticker';
export const hasAnyDirtyScreenDefinitions = (state) => !!objectKeys(state.screenDefinitions).find(screenId => isScreenDefinitionDirty(state.screenDefinitions[screenId])) ||
    !!Object.values(state.dashboard.dashboardGroups).find(group => group.dashboardEditor.isDirty || group.widgetEditor.isDirty) ||
    isSidebarDirty(state);
export const getPageDefinitionFromState = (screenId, state = xtremRedux.getStore().getState()) => {
    const pageDefinition = getPageDefinition(screenId, state);
    if (!pageDefinition) {
        throw new Error(`No page definition found: ${screenId}`);
    }
    return pageDefinition;
};
export const getPageDefinition = (screenId, state = xtremRedux.getStore().getState()) => {
    return state.screenDefinitions[screenId];
};
export const getMainPageDefinitionFromState = (state = xtremRedux.getStore().getState()) => objectKeys(state.screenDefinitions)
    .filter(key => state.screenDefinitions[key].type === 'page')
    .map(key => state.screenDefinitions[key])
    .find(definition => definition.isMainPage);
export const getPagePropertiesFromState = (screenId, state = xtremRedux.getStore().getState()) => {
    const properties = getPageDefinitionFromState(screenId, state).metadata.uiComponentProperties[screenId];
    if (!properties) {
        xtremConsole.error(getPageDefinitionFromState(screenId, state));
        throw new Error(`No properties found for page: ${screenId}`);
    }
    return properties;
};
export const getPageControlObjectFromState = (screenId, state = xtremRedux.getStore().getState()) => getPageDefinitionFromState(screenId, state).metadata.controlObjects[screenId] || null;
export const getPagePropertiesFromPageDefinition = (pageDefinition) => pageDefinition.metadata.uiComponentProperties[pageDefinition.metadata.screenId];
export const getNavigationPanelState = (screenId, state = xtremRedux.getStore().getState()) => state.screenDefinitions[screenId]?.navigationPanel || null;
export const getNavigationPanelDefinitionFromState = (screenId, state = xtremRedux.getStore().getState()) => getPagePropertiesFromState(screenId, state).navigationPanel || null;
export const getNavigationPanelTablePropertiesFromPageDefinition = (pageDefinition, isDefaultProps = false) => pageDefinition.metadata[isDefaultProps ? 'defaultUiComponentProperties' : 'uiComponentProperties'][navigationPanelId];
export const checkIfPageIsLoaded = (screenId, state = xtremRedux.getStore().getState()) => {
    if (!getPageDefinitionFromState(screenId, state)) {
        throw new Error(`${screenId} page has already been destroyed, further operations are not allowed.`);
    }
};
export function isFieldDataLoaded(pageDefinition, elementId) {
    const parentSection = getParentSection(pageDefinition, elementId);
    if (parentSection) {
        const sectionProperties = pageDefinition.metadata.uiComponentProperties[parentSection];
        return !sectionProperties.isLazyLoaded || !!sectionProperties.isLoaded;
    }
    return true;
}
export const arePlatformLiteralsMissing = (state = xtremRedux.getStore().getState(), locale = state.applicationContext?.locale || 'en-US') => !state.translations ||
    !state.translations[locale] ||
    !objectKeys(state.translations[locale]).find(key => key.startsWith('@sage/xtrem-ui'));
export const getSidebarTableProperties = ({ elementId, pageDefinition, level = 0, }) => {
    const tableProperties = pageDefinition.metadata.uiComponentProperties[elementId]._controlObjectType === FieldKey.NestedGrid
        ? pageDefinition.metadata.uiComponentProperties[elementId].levels[level]
        : pageDefinition.metadata.uiComponentProperties[elementId];
    // We make a exception for MultifileDeposit because we hardcode the sidebar from our side so it's not defined in the page definition
    if (!tableProperties.sidebar &&
        pageDefinition.metadata.uiComponentProperties[elementId]._controlObjectType !== FieldKey.MultiFileDeposit) {
        throw new Error(`No sidebar is defined for element with ID ${elementId}`);
    }
    return tableProperties;
};
export const getSidebarNormalFields = (layoutDefinition) => {
    const collector = [];
    objectKeys(layoutDefinition || {}).forEach(sectionKey => {
        objectKeys(layoutDefinition[sectionKey].blocks || []).forEach(blockKey => {
            (layoutDefinition[sectionKey].blocks[blockKey].fields || []).forEach(f => {
                if (f instanceof AbstractUiControlObject) {
                    collector.push(f);
                }
            });
        });
    });
    return collector;
};
export const getSidebarNestedFields = (layoutDefinition) => {
    const collector = [];
    objectKeys(layoutDefinition || {}).forEach(sectionKey => {
        objectKeys(layoutDefinition[sectionKey].blocks || []).forEach(blockKey => {
            (layoutDefinition[sectionKey].blocks[blockKey].fields || []).forEach(f => {
                if (!(f instanceof AbstractUiControlObject)) {
                    collector.push(convertDeepBindToPathNotNull(f));
                }
            });
        });
    });
    return collector;
};
export function getResolvedSidebarLayout({ value, screenDefinition, level = 0, nodeTypes, elementId, layout, }) {
    const customFields = getCustomFields(value.contextNode || value.nodes?.[level], nodeTypes, value.getColumnDefinitions(level), elementId, FieldKey.Table, screenDefinition.metadata.customizations);
    return mergeIntoSidebarLayout(resolveByValue({
        screenId: screenDefinition.metadata.screenId,
        propertyValue: layout,
        rowValue: null,
        fieldValue: null,
        skipHexFormat: true,
    }), customFields);
}
export const isSidebarDirty = (state = xtremRedux.getStore().getState()) => {
    return Object.values(state.activeDialogs).some(dialog => {
        if (dialog.type !== 'table-sidebar') {
            return false;
        }
        const screenId = dialog.screenId;
        if (!screenId) {
            throw new Error('No screenId found in dialog');
        }
        const dialogContent = dialog.content;
        const tableValue = state.screenDefinitions[screenId].values[dialogContent.elementId];
        const committedRecords = tableValue.getData({
            isUncommitted: false,
            includePhantom: true,
            cleanMetadata: false,
            ...(dialogContent.recordId ? { level: dialogContent.level } : {}),
            parentId: null,
            where: {
                ...(dialogContent.recordId ? { _id: dialogContent.recordId } : { __phantom: true }),
            },
        });
        const records = tableValue.getData({
            isUncommitted: true,
            includePhantom: true,
            cleanMetadata: false,
            ...(dialogContent.recordId ? { level: dialogContent.level } : {}),
            parentId: null,
            where: {
                ...(dialogContent.recordId ? { _id: dialogContent.recordId } : { __phantom: true }),
            },
        });
        if (records?.[0]?.__dirtyColumns && (records?.[0]?.__dirtyColumns).size > 0) {
            const committedRecord = committedRecords?.[0];
            if (!committedRecord) {
                return true;
            }
            // Check if any dirty column is different from the committed record
            const isDifferent = Array.from(records[0].__dirtyColumns).some((key) => !isEqual(committedRecord[key], records[0][key]));
            if (isDifferent) {
                return true;
            }
        }
        const sidebarLayout = getResolvedSidebarLayout({
            value: tableValue,
            screenDefinition: state.screenDefinitions[screenId],
            level: dialogContent.level,
            nodeTypes: state.nodeTypes,
            elementId: dialogContent.elementId,
            layout: dialogContent.sidebarDefinition.layout,
        });
        const normalFields = getSidebarNormalFields(sidebarLayout);
        return !!normalFields.find(n => state.screenDefinitions[screenId].dirtyStates[n.id]);
    });
};
export const getVisibleSectionsFromPage = (screenId, screenDefinition = xtremRedux.getStore().getState().screenDefinitions[screenId]) => {
    const sectionOrder = screenDefinition.metadata.layout.$items.map(i => i.$containerId);
    const sections = Object.values(screenDefinition.metadata.controlObjects).filter(controlObject => controlObject instanceof SectionControlObject &&
        !resolveByValue({
            screenId,
            skipHexFormat: true,
            propertyValue: controlObject.properties.isHidden,
            rowValue: null,
            fieldValue: null,
        }));
    return sections.sort((s1, s2) => sectionOrder.indexOf(s1.id) - sectionOrder.indexOf(s2.id));
};
//# sourceMappingURL=state-utils.js.map