import { objectKeys } from '@sage/xtrem-shared';
import { get, isObject } from 'lodash';
import { getReferenceValueField } from '../component/field/reference/reference-utils';
import { FieldKey } from '../component/types';
export const isNestedField = (maybeNestedField) => !!maybeNestedField && typeof maybeNestedField === 'object' && maybeNestedField.properties && maybeNestedField.type;
export const getNestedFieldsFromProperties = (properties, level = 0) => {
    let columns = [];
    if (properties.columns) {
        columns = [...(properties.columns || [])];
    }
    if (Object.prototype.hasOwnProperty.call(properties, 'levels')) {
        columns = [...columns, ...(properties.levels[level].columns ?? [])];
    }
    const mobileCardDefinition = properties.mobileCard;
    if (mobileCardDefinition && isObject(mobileCardDefinition)) {
        columns = [...columns, ...Object.values(mobileCardDefinition).filter(v => !!v)];
    }
    const calendarProperties = properties;
    if (calendarProperties.eventCard) {
        columns = [
            ...columns,
            ...Object.values(calendarProperties.eventCard).filter(v => !!v),
        ];
    }
    if (calendarProperties.startDateField) {
        columns.push({
            type: FieldKey.Date,
            defaultUiProperties: {
                bind: convertDeepBindToPathNotNull(calendarProperties.startDateField),
            },
            properties: {
                bind: convertDeepBindToPathNotNull(calendarProperties.startDateField),
            },
        });
    }
    if (calendarProperties.endDateField) {
        columns.push({
            type: FieldKey.Date,
            defaultUiProperties: {
                bind: convertDeepBindToPathNotNull(calendarProperties.endDateField),
            },
            properties: {
                bind: convertDeepBindToPathNotNull(calendarProperties.endDateField),
            },
        });
    }
    if (isNestedField(properties.recordTitle)) {
        columns.push(properties.recordTitle);
    }
    const headerLabel = properties.headerLabel;
    if (headerLabel) {
        columns.push(headerLabel);
    }
    return columns;
};
/**
 * Resolve title field from a card definition to a display string value.
 */
export const getImagePlaceholderValue = (record, titleField) => {
    if (!record) {
        return undefined;
    }
    const properties = titleField.properties;
    const fieldValuePath = convertDeepBindToPath(titleField.properties.bind);
    if (fieldValuePath && titleField.type === FieldKey.Reference) {
        const valueField = getReferenceValueField(properties);
        const value = get(record, `${fieldValuePath}.${valueField}`);
        if (value) {
            return String(value);
        }
        return '_';
    }
    if (fieldValuePath) {
        const fieldValue = get(record, fieldValuePath);
        if (fieldValue) {
            return String(fieldValue);
        }
    }
    return '_';
};
/**
 * Turn nested bind property reference object to dot-notion reference. For example, it converts `{this:{is:{a:{nested:{bind:true}}}}}`
 * to `this.is.a.nested.bind`.
 * @param path string or deep bind object
 * @returns dot notion deep reference string
 */
export const convertDeepBindToPath = (path) => {
    if (path === null || path === undefined || typeof path === 'boolean') {
        return null;
    }
    if (typeof path === 'string') {
        return path;
    }
    if (typeof path === 'object' && objectKeys(path).length === 1) {
        const childKeys = objectKeys(path);
        const childPath = convertDeepBindToPath(path[childKeys[0]]);
        if (childPath) {
            return `${childKeys[0]}.${childPath}`;
        }
        return childKeys[0];
    }
    throw new Error(`Unsupported bind value: ${JSON.stringify(path)}`);
};
/**
 * Same as convertDeepBindToPath, but throws an error if the value is null or undefined.
 * @param path
 */
export const convertDeepBindToPathNotNull = (path) => {
    const result = convertDeepBindToPath(path);
    if (!result) {
        throw new Error('Bind cannot be falsy.');
    }
    return result;
};
/**
 * Returns to first component of a nested bind property For example, it converts `hi`, `hi.there` and `{hi:{there:true}}` to `hi`.
 * @param path
 * @returns top level property name
 */
export const getTopLevelBindFromNestedBindNotNull = (path) => {
    if (!path) {
        throw new Error('Invalid bind, cannot be empty.');
    }
    return getTopLevelBindFromNestedBind(path);
};
export function getTopLevelBindFromNestedBind(path) {
    if (typeof path === 'string') {
        return path.split('.')[0];
    }
    if (typeof path === 'object' && objectKeys(path).length === 1) {
        return objectKeys(path)[0];
    }
    if (path === undefined) {
        return path;
    }
    throw new Error(`Unsupported bind value: ${JSON.stringify(path)}`);
}
//# sourceMappingURL=nested-field-utils.js.map