import { objectKeys } from '@sage/xtrem-shared';
import { get } from 'lodash';
import { getStore } from '../redux';
import { setGlobalLoading } from '../redux/actions';
import { errorDialog } from '../service/dialog-service';
import { PromiseTracker } from '../service/promise-tracker';
import { getScreenElement } from '../service/screen-base-definition';
import { notifyConsumerOnError } from '../service/telemetry-service';
import { showToast } from '../service/toast-service';
import { xtremConsole } from './console';
import { resolveByValue } from './resolve-value-utils';
import { ServerError } from './server-error-transformer';
import { isDevMode } from './window';
const handleError = (screenId, error) => {
    getStore().dispatch(setGlobalLoading(false));
    errorDialog(screenId, 'Error', error);
    notifyConsumerOnError(error);
};
/**
 * Executes event and event-after callbacks with the screen scope. The event handler and arguments should be passed in to the function.
 * In case of an error occurs in the during the call back execution, the onError function is called.
 * */
const executeEventHandlerPrivate = async ({ eventName, screenProperties, screenId, elementId, ctx, elementDef, targetDef, args, }) => {
    const eventHandler = args && args[0]?.level !== undefined
        ? get(targetDef, `levels.${args[0]}.level.${eventName}`)
        : get(targetDef, eventName);
    const errorHandler = targetDef.onError || elementDef.onError || screenProperties.onError;
    try {
        if (eventHandler) {
            if (typeof eventHandler === 'function') {
                await eventHandler.apply(ctx, args);
            }
            else if (typeof eventHandler === 'object') {
                // eslint-disable-next-line no-restricted-syntax
                for (const fn of eventHandler) {
                    // eslint-disable-next-line no-await-in-loop
                    await fn.apply(ctx, args);
                }
            }
        }
        const afterEvents = get(targetDef, `${eventName}After`);
        if (afterEvents) {
            await executeEventHandler(`${eventName}After`, screenId, elementId, ctx, screenProperties, elementDef, targetDef, ...args);
        }
    }
    catch (error) {
        xtremConsole.error(error);
        if (targetDef.delegateErrorToCaller) {
            throw error;
        }
        else if (errorHandler) {
            const rawResult = errorHandler.apply(ctx, [error, screenId, elementId]);
            const result = await (rawResult instanceof Promise ? rawResult : Promise.resolve(rawResult));
            if (result) {
                const isServerError = error instanceof ServerError;
                showToast(isServerError ? error.toastContent : result, {
                    type: 'error',
                    timeout: 10000,
                    language: isServerError ? 'jsx' : 'markdown',
                });
            }
            else if (isDevMode()) {
                xtremConsole.warn(`Silent error on ${screenId} / ${elementId}, you should handle this error.`);
                xtremConsole.warn(error);
            }
        }
        else {
            handleError(screenId, error);
        }
    }
};
const executeEventHandler = async (eventName, screenId, elementId, ctx, screenDef, elementDef, targetDef, ...args) => {
    return PromiseTracker.withTracker(() => executeEventHandlerPrivate({
        eventName,
        screenId,
        elementId,
        ctx,
        screenProperties: screenDef,
        elementDef,
        targetDef,
        args,
    }));
};
export const executeEventHandlerWithExternalHandler = async ({ screenId, elementId, args, eventHandler, }) => {
    const state = getStore().getState();
    const screenDef = state.screenDefinitions[screenId];
    if (!screenDef) {
        throw Error(`events: ${screenId} screen definition doesn't exist in current state`);
    }
    const ctx = getScreenElement(screenDef);
    if (!ctx) {
        throw Error("events: screen context couldn't be built");
    }
    const screenProperties = screenDef.metadata.uiComponentProperties[screenId] ||
        (elementId ? screenDef.metadata.pageActions[elementId] : false);
    if (!screenProperties) {
        throw Error('events: screen or action unreachable');
    }
    const element = (elementId && screenDef.metadata.uiComponentProperties[elementId]) || screenProperties;
    return PromiseTracker.withTracker(() => executeEventHandlerPrivate({
        eventName: 'onClick',
        screenId,
        elementId,
        ctx,
        screenProperties,
        targetDef: { onClick: eventHandler },
        elementDef: element,
        args,
    }));
};
const isDisabled = (screenId, elementId) => {
    const state = getStore().getState();
    const fieldValue = state.screenDefinitions[screenId].metadata.controlObjects[elementId]?.value;
    const element = state.screenDefinitions[screenId].metadata.uiComponentProperties[elementId];
    return (element &&
        resolveByValue({
            propertyValue: element.isDisabled,
            skipHexFormat: true,
            fieldValue,
            screenId,
            rowValue: null,
        }));
};
const produceExecutionArguments = (eventName, screenId, elementId, explicitDef) => {
    const state = getStore().getState();
    const screenDef = state.screenDefinitions[screenId];
    if (!screenDef) {
        throw Error(`events: ${screenId} screen definition doesn't exist in current state`);
    }
    const screenCtx = getScreenElement(screenDef);
    if (!screenCtx) {
        throw Error("events: screen context couldn't be built");
    }
    const screen = screenDef.metadata.uiComponentProperties[screenId] ||
        (elementId ? screenDef.metadata.pageActions[elementId] : false);
    if (!screen) {
        throw Error('events: screen or action unreachable');
    }
    const element = (elementId && screenDef.metadata.uiComponentProperties[elementId]) || screen;
    const targetDef = explicitDef || element;
    return [eventName, screenId, elementId || screenId, screenCtx, screen, element, targetDef];
};
export const triggerScreenEvent = async (screenId, eventName, ...rest) => {
    const args = produceExecutionArguments(eventName, screenId);
    if (args) {
        await executeEventHandler(...args, ...rest);
    }
};
export const triggerFieldEvent = async (screenId, elementId, eventName, ...rest) => {
    const args = produceExecutionArguments(eventName, screenId, elementId);
    if (args && !isDisabled(screenId, elementId)) {
        await executeEventHandler(...args, ...rest);
    }
};
export const triggerNestedFieldEvent = async (screenId, elementId, columnDefinition, eventName, ...rest) => {
    const args = produceExecutionArguments(eventName, screenId, elementId, columnDefinition);
    if (args) {
        await executeEventHandler(...args, ...rest);
    }
};
export const triggerHandledEvent = async (screenId, elementId, handledEventContainer, ...rest) => {
    const eventName = objectKeys(handledEventContainer).find(e => e !== 'onError' && !e.endsWith('After'));
    if (!eventName) {
        return;
    }
    const args = produceExecutionArguments(eventName, screenId, elementId, handledEventContainer);
    if (!args) {
        return;
    }
    await executeEventHandler(...args, ...rest);
};
export const isBulkChange = (s1, s2) => {
    const string1 = String(s1 === null || s1 === undefined ? '' : s1);
    const string2 = String(s2 === null || s2 === undefined ? '' : s2);
    if (Math.abs(string1.length - string2.length) > 1) {
        return true;
    }
    let diffCount = 0;
    Array.from(string1).forEach((character, index) => {
        if (character !== string2.charAt(index)) {
            diffCount += 1;
        }
    });
    return diffCount > 1;
};
export const executeCallbackInScreenContext = (callback, screenId, ...args) => {
    const state = getStore().getState();
    const screenDef = state.screenDefinitions[screenId];
    if (!screenDef) {
        throw Error(`events: ${screenId} screen definition doesn't exist in current state`);
    }
    const screenCtx = getScreenElement(screenDef);
    if (!screenCtx) {
        throw Error("events: screen context couldn't be built");
    }
    callback.apply(screenCtx, args);
};
//# sourceMappingURL=events.js.map