import { camelCase } from 'lodash';
import { normalizeUnderscoreBind } from './abstract-fields-utils';
import { resolveByValue } from './resolve-value-utils';
import { splitValueToMergedValue } from './transformers';
export function scrollWithAnimationTo(element, container, offset) {
    const startTime = performance.now();
    const elementTop = element ? element.getBoundingClientRect().top : 0;
    const targetY = Math.max(0, container.scrollTop + elementTop - (offset || 0));
    const startPoint = container.scrollTop;
    const distance = targetY - startPoint;
    const scrollDuration = Math.min(Math.max(300, Math.abs(distance)), 900);
    const animationStep = (newTimestamp) => {
        if (Math.round(container.scrollTop) !== targetY && newTimestamp < startTime + scrollDuration) {
            const sinPercent = Math.sin((Math.PI / 2) * ((newTimestamp - startTime) / scrollDuration));
            const scrollValue = startPoint + sinPercent * distance;
            container.scrollTop = scrollValue;
            window.requestAnimationFrame(animationStep);
        }
    };
    window.requestAnimationFrame(animationStep);
}
export function isHidden(hiddenProperties, browser) {
    if (!browser || !hiddenProperties)
        return false;
    const isHiddenDesktop = hiddenProperties.$isHiddenDesktop ||
        hiddenProperties.isHiddenDesktop ||
        false;
    const isHiddenMobile = hiddenProperties.$isHiddenMobile ||
        hiddenProperties.isHiddenMobile ||
        false;
    return (isHiddenDesktop && browser.greaterThan.s) || (isHiddenMobile && !browser.greaterThan.s);
}
export const getDataTestIdAttribute = (fieldType, label, 
// TODO: XT-27998 `elementId` should always be a string, remove this workaround
// when `valueField` gets added to nested image fields.
elementId, prefix = 'e-field') => {
    return `e-${fieldType}-field${label ? ` ${prefix}-label-${camelCase(label)}` : ''}${elementId && typeof elementId === 'string' ? ` ${prefix}-bind-${normalizeUnderscoreBind(elementId)}` : ''}`;
};
export const getComponentClass = (props, specificClassNames, rowValue) => {
    const isHiddenValue = resolveByValue({
        screenId: props.screenId,
        fieldValue: props.value,
        propertyValue: props.fieldProperties.isHidden,
        rowValue: splitValueToMergedValue(rowValue),
        skipHexFormat: true,
    });
    const isFieldHidden = isHidden({ ...props.item, ...props.fieldProperties }, props.browser) || isHiddenValue;
    const hasValidationErrors = props.validationErrors && props.validationErrors.length > 0;
    return getFieldClass(props.screenId, props.value, `e-field ${specificClassNames}`, props.fieldProperties, hasValidationErrors, isFieldHidden, rowValue, props.contextType);
};
export const getFieldClass = (screenId, fieldValue, classNames, fieldProperties, hasValidationErrors = false, isFieldHidden = false, rowValue = {}, contextType) => {
    let classes = classNames;
    if (isFieldHidden) {
        classes += ' e-hidden';
    }
    if (hasValidationErrors) {
        classes += ' e-invalid';
    }
    if (fieldProperties.isFullWidth) {
        classes += ' e-full-width';
    }
    if (fieldProperties.isHelperTextHidden) {
        classes += ' e-helper-text-hidden';
    }
    if (fieldProperties.isTitleHidden) {
        classes += ' e-title-hidden';
    }
    if (contextType) {
        classes += ` e-context-${contextType}`;
    }
    if (resolveByValue({
        screenId,
        fieldValue,
        propertyValue: fieldProperties.isReadOnly,
        rowValue: splitValueToMergedValue(rowValue),
        skipHexFormat: true,
    })) {
        classes += ' e-read-only';
    }
    if (resolveByValue({
        screenId,
        fieldValue,
        propertyValue: fieldProperties.isDisabled,
        rowValue: splitValueToMergedValue(rowValue),
        skipHexFormat: true,
    })) {
        classes += ' e-disabled';
    }
    return classes;
};
export const copyToClipboard = (content) => {
    const selection = document && document.getSelection();
    if (selection) {
        const el = document.createElement('textarea');
        el.value = content;
        el.setAttribute('readonly', '');
        el.style.position = 'absolute';
        el.style.left = '-9999px';
        document.body.appendChild(el);
        const selected = selection.rangeCount > 0 ? selection.getRangeAt(0) : false;
        el.select();
        document.execCommand('copy');
        document.body.removeChild(el);
        if (selected) {
            selection.removeAllRanges();
            selection.addRange(selected);
        }
    }
};
export function focusTopPage() {
    setTimeout(() => {
        const pages = document.querySelectorAll('.e-page-body');
        if (pages && pages.length > 0) {
            const targetPage = pages[pages.length - 1];
            if (targetPage instanceof HTMLElement) {
                targetPage.focus({
                    preventScroll: true,
                });
            }
        }
    });
}
export const focusPageContent = () => {
    const mainElements = Array.from(document.querySelectorAll('main'));
    if (mainElements.length) {
        const selector = 'a[href], button, input, textarea, select, details, [tabindex]:not([tabindex="-1"])';
        const focusableElements = Array.from(mainElements[mainElements.length - 1].querySelectorAll(selector)).filter(element => {
            return (!element.hasAttribute('disabled') &&
                !element.hasAttribute('aria-hidden') &&
                !element.classList.contains('e-navigation-panel-toggle'));
        });
        if (focusableElements.length) {
            focusableElements[0].focus();
        }
    }
};
/** For mocking purposes */
export const getNewFileReader = () => new FileReader();
export const getTabControlElement = () => document.querySelector('.e-header-nav li[tabindex="0"]');
export const isScrolledToTheEnd = (element) => {
    const sh = element.scrollWidth;
    const st = element.scrollLeft;
    const ht = element.offsetWidth;
    if (ht === 0 || st === sh - ht) {
        return true;
    }
    return false;
};
export const isChildOfElementWithClass = (element, className) => {
    if (element === window.document) {
        return false;
    }
    if (element?.classList.contains(className)) {
        return true;
    }
    if (!element.parentElement) {
        return false;
    }
    return isChildOfElementWithClass(element.parentElement, className);
};
export const isChildOfElementWithAttributeValue = (element, attributeName, attributeValue) => {
    if (element === window.document) {
        return false;
    }
    if (element?.getAttribute(attributeName) === attributeValue) {
        return true;
    }
    if (!element.parentElement) {
        return false;
    }
    return isChildOfElementWithAttributeValue(element.parentElement, attributeName, attributeValue);
};
export const findAncestorDatasetProperty = (element, datasetPropertyName) => {
    if (element === window.document) {
        return null;
    }
    const datasetPropertyValue = element?.dataset[datasetPropertyName];
    if (datasetPropertyValue) {
        return datasetPropertyValue;
    }
    if (!element?.parentElement) {
        return null;
    }
    return findAncestorDatasetProperty(element.parentElement, datasetPropertyName);
};
//# sourceMappingURL=dom.js.map