import { FieldKey } from '@sage/xtrem-shared';
import { has, isNil, uniqBy } from 'lodash';
import * as nestedFields from '../component/nested-fields';
import { findDeepPropertyType } from './node-utils';
import { schemaTypeNameFromNodeName } from './transformers';
import { convertDeepBindToPath, convertDeepBindToPathNotNull } from './nested-field-utils';
export function getCardDefinitionFromReferenceDataType({ dataType, dataTypes, nodeTypes, contextNode, }) {
    const namedCardProperties = [
        convertDeepBindToPath(dataType.imageField?.bind),
        convertDeepBindToPath(dataType.value?.bind),
        convertDeepBindToPath(dataType.helperText?.bind),
    ];
    const columns = (dataType.columns || [])
        .map(c => createNestedFieldFromDataTypeProperty(c, nodeTypes, dataTypes, contextNode))
        .reduce((prevValue, curr) => {
        const normalizedBind = convertDeepBindToPathNotNull(curr.properties.bind);
        const propertyDetails = findDeepPropertyType(schemaTypeNameFromNodeName(contextNode), normalizedBind, nodeTypes);
        if (!namedCardProperties.includes(normalizedBind) && propertyDetails?.isOnOutputType) {
            prevValue[normalizedBind.split('.').join('__')] = curr;
        }
        return prevValue;
    }, {});
    return {
        image: dataType.imageField
            ? createNestedFieldFromDataTypeProperty(dataType.imageField, nodeTypes, dataTypes, contextNode)
            : undefined,
        title: createNestedFieldFromDataTypeProperty(dataType.value, nodeTypes, dataTypes, contextNode),
        line2: dataType.helperText
            ? createNestedFieldFromDataTypeProperty(dataType.helperText, nodeTypes, dataTypes, contextNode)
            : undefined,
        ...columns,
    };
}
export function applyDefaultValuesOnNestedField(nodeTypes, dataTypes, field, contextNode) {
    if (!field || !contextNode) {
        return;
    }
    const node = schemaTypeNameFromNodeName(contextNode);
    const propertyDetails = findDeepPropertyType(node, field.properties.bind, nodeTypes);
    const dataType = propertyDetails?.dataType ? dataTypes[propertyDetails?.dataType] : null;
    if (!propertyDetails) {
        return;
    }
    const { properties, defaultUiProperties, type } = field;
    addTitleToProperties({
        propertyDetails,
        properties: properties,
    });
    addTitleToProperties({
        propertyDetails,
        properties: defaultUiProperties,
    });
    if (!dataType) {
        return;
    }
    switch (type) {
        case FieldKey.Reference:
            addDisabledToProperties({ propertyDetails, dataType, properties });
            addDisabledToProperties({ propertyDetails, dataType, properties: defaultUiProperties });
            addValueFieldToProperties({
                dataType,
                properties: properties,
                propertyDetails,
            });
            addValueFieldToProperties({
                dataType,
                properties: defaultUiProperties,
                propertyDetails,
            });
            addHelperTextFieldToProperties({
                dataType,
                properties: properties,
                propertyDetails,
            });
            addHelperTextFieldToProperties({
                dataType,
                properties: defaultUiProperties,
                propertyDetails,
            });
            addImageFieldToProperties({
                dataType,
                properties: properties,
                propertyDetails,
            });
            addImageFieldToProperties({
                dataType,
                properties: defaultUiProperties,
                propertyDetails,
            });
            addNodeToProperties({
                dataType,
                propertyDetails,
                properties: properties,
            });
            addNodeToProperties({
                dataType,
                propertyDetails,
                properties: defaultUiProperties,
            });
            addColumnsToProperties({
                nodeTypes,
                dataTypes,
                dataType,
                properties: properties,
                propertyDetails,
            }, true);
            addColumnsToProperties({
                nodeTypes,
                dataTypes,
                dataType,
                properties: defaultUiProperties,
                propertyDetails,
            }, true);
            const columns = properties.columns;
            const subnode = properties.node;
            if (columns && subnode) {
                columns.forEach(c => {
                    applyDefaultValuesOnNestedField(nodeTypes, dataTypes, c, String(subnode));
                });
            }
            break;
        case FieldKey.Numeric:
            addDisabledToProperties({ propertyDetails, dataType, properties });
            addDisabledToProperties({ propertyDetails, dataType, properties: defaultUiProperties });
            addMaxMinToProperties({
                dataType,
                propertyDetails,
                properties: properties,
            });
            addMaxMinToProperties({
                dataType,
                propertyDetails,
                properties: defaultUiProperties,
            });
            break;
        case FieldKey.Text:
            addDisabledToProperties({ propertyDetails, dataType, properties });
            addDisabledToProperties({ propertyDetails, dataType, properties: defaultUiProperties });
            addMaxLengthToProperties({
                dataType,
                propertyDetails,
                properties: properties,
            });
            addMaxLengthToProperties({
                dataType,
                propertyDetails,
                properties: defaultUiProperties,
            });
            break;
        case FieldKey.Select:
        case FieldKey.DropdownList:
            addDisabledToProperties({ propertyDetails, dataType, properties });
            addDisabledToProperties({ propertyDetails, dataType, properties: defaultUiProperties });
            addOptionTypeToProperties({
                dataType,
                propertyDetails,
                properties: properties,
            });
            addOptionTypeToProperties({
                dataType,
                propertyDetails,
                properties: defaultUiProperties,
            });
            break;
        case FieldKey.Label:
            addOptionTypeToProperties({
                dataType,
                propertyDetails,
                properties: properties,
            });
            addOptionTypeToProperties({
                dataType,
                propertyDetails,
                properties: defaultUiProperties,
            });
            break;
        default:
        // Intentionally empty
    }
}
export function createNestedFieldFromDataTypeProperty(prop, nodeTypes, dataTypes, contextNode) {
    const node = schemaTypeNameFromNodeName(contextNode);
    const propertyDetails = findDeepPropertyType(node, prop.bind, nodeTypes) || undefined;
    const dataType = propertyDetails?.dataType ? dataTypes[propertyDetails?.dataType] : undefined;
    const propCopy = { ...prop };
    if (propertyDetails) {
        addTitleToProperties({ propertyDetails, properties: propCopy });
    }
    switch (prop.type) {
        case 'boolean':
            addDisabledToProperties({ propertyDetails, dataType, properties: propCopy });
            return nestedFields.checkbox(propCopy);
        case 'date':
            addDisabledToProperties({ propertyDetails, dataType, properties: propCopy });
            return nestedFields.date(propCopy);
        case 'integer':
            propCopy.scale = 0;
            addDisabledToProperties({ propertyDetails, dataType, properties: propCopy });
            return nestedFields.numeric(propCopy);
        case 'decimal':
        case 'double':
        case 'short':
            if (dataType && propertyDetails) {
                addMaxMinToProperties({
                    dataType,
                    propertyDetails,
                    properties: propCopy,
                });
                addScaleToProperties({
                    dataType,
                    propertyDetails,
                    properties: propCopy,
                });
            }
            addDisabledToProperties({ propertyDetails, dataType, properties: propCopy });
            return nestedFields.numeric(propCopy);
        case 'string':
            if (dataType && propertyDetails) {
                addMaxLengthToProperties({
                    dataType,
                    propertyDetails,
                    properties: propCopy,
                });
            }
            addDisabledToProperties({ propertyDetails, dataType, properties: propCopy });
            return nestedFields.text(propCopy);
        case 'json':
            return nestedFields.text(propCopy);
        case 'binaryStream':
            return nestedFields.image(propCopy);
        case 'enum':
            if (dataType && propertyDetails) {
                addOptionTypeToProperties({
                    dataType,
                    propertyDetails,
                    properties: propCopy,
                });
            }
            addDisabledToProperties({ propertyDetails, dataType, properties: propCopy });
            return nestedFields.select(propCopy);
        default:
            addDisabledToProperties({ propertyDetails, dataType, properties: propCopy });
            return nestedFields.text(propCopy);
    }
}
export function addTitleToProperties({ propertyDetails, properties, }) {
    if (propertyDetails.title && !has(properties, 'title')) {
        properties.title = propertyDetails.title;
    }
}
export function addIsTransientInput({ propertyDetails, properties, }) {
    if (!propertyDetails.isOnOutputType && !has(properties, 'isTransientInput')) {
        properties.isTransientInput = true;
    }
}
export function addMaxLengthToProperties({ dataType, properties, }) {
    if (dataType.maxLength && !has(properties, 'maxLength')) {
        properties.maxLength = dataType.maxLength;
    }
}
export function addOptionTypeToProperties({ propertyDetails, properties, }) {
    if (propertyDetails.enumType && !has(properties, 'optionType')) {
        properties.optionType = propertyDetails.enumType;
    }
}
export function addScaleToProperties({ dataType, properties, }) {
    if (!isNil(dataType.scale) && !has(properties, 'scale')) {
        properties.scale = dataType.scale;
    }
    if (dataType.type === 'integer') {
        properties.scale = 0;
    }
}
export function addMaxMinToProperties({ dataType, properties }) {
    if (!isNil(dataType.scale) && !isNil(dataType.precision) && !has(properties, 'min') && !has(properties, 'max')) {
        const max = 10 ** (dataType.precision - dataType.scale) - 10 ** -dataType.scale;
        properties.max = max;
        properties.min = -max;
    }
}
export function addNodeToProperties({ propertyDetails, properties, }) {
    if (propertyDetails.targetNode && !has(properties, 'node')) {
        properties.node = propertyDetails.targetNode;
    }
}
export function addTunnelPageToProperties({ dataType, properties, }) {
    if (dataType.tunnelPage && !has(properties, 'tunnelPage')) {
        properties.tunnelPage = dataType.tunnelPage;
    }
}
export function addDisabledToProperties({ propertyDetails, properties, }) {
    if (propertyDetails && propertyDetails.isOnInputType === false && !has(properties, 'isDisabled')) {
        properties.isDisabled = true;
    }
}
export function getDefaultColumnsFromDataType({ ignoreCardDefinition = false, nodeTypes, dataTypes, dataType, targetNode, }) {
    if (ignoreCardDefinition && targetNode) {
        return (dataType.columns || [])
            .filter(c => {
            const details = findDeepPropertyType(schemaTypeNameFromNodeName(targetNode), convertDeepBindToPathNotNull(c.bind), nodeTypes);
            return details?.isOnInputType;
        })
            .map(c => createNestedFieldFromDataTypeProperty(c, nodeTypes, dataTypes, targetNode));
    }
    return uniqBy(Object.values(getCardDefinitionFromReferenceDataType({
        nodeTypes,
        dataTypes,
        dataType,
        contextNode: targetNode,
    })).filter(c => !!c), f => convertDeepBindToPathNotNull(f.properties.bind));
}
export function addColumnsToProperties({ dataType, propertyDetails, properties, nodeTypes, dataTypes, }, ignoreCardDefinition = false) {
    if (dataType.columns && propertyDetails.targetNode && !has(properties, 'columns')) {
        properties.columns = getDefaultColumnsFromDataType({
            dataType,
            dataTypes,
            ignoreCardDefinition,
            nodeTypes,
            targetNode: propertyDetails.targetNode,
        });
    }
}
export function addMobileCardDefinitionToProperties({ dataType, propertyDetails, properties, nodeTypes, dataTypes, }) {
    if (dataType.columns && propertyDetails.targetNode && !has(properties, 'mobileCard')) {
        properties.mobileCard = getCardDefinitionFromReferenceDataType({
            nodeTypes,
            dataTypes,
            dataType,
            contextNode: propertyDetails.targetNode,
        });
    }
}
export function addValueFieldToProperties({ dataType, propertyDetails, properties, }) {
    if (dataType.value?.bind && propertyDetails.targetNode && !has(properties, 'valueField')) {
        properties.valueField = dataType.value.bind;
    }
}
export function addHelperTextFieldToProperties({ dataType, propertyDetails, properties, }) {
    if (dataType.helperText?.bind && propertyDetails.targetNode && !has(properties, 'helperTextField')) {
        properties.helperTextField = dataType.helperText.bind;
    }
}
export function addImageFieldToProperties({ dataType, propertyDetails, properties, }) {
    if (dataType.imageField?.bind && propertyDetails.targetNode && !has(properties, 'imageField')) {
        properties.imageField = dataType.imageField.bind;
    }
}
export function createDynamicPodReferenceField({ column, nodeTypes, dataTypes, }) {
    const node = column.properties.node;
    const contextNode = schemaTypeNameFromNodeName(column.properties.node);
    if (!contextNode) {
        return column;
    }
    const nodeInfo = nodeTypes[contextNode];
    if (nodeInfo?.defaultDataType) {
        const propertyDetails = {
            targetNode: String(node),
            type: nodeInfo.name,
            title: nodeInfo.title,
        };
        const dataType = dataTypes[nodeInfo?.defaultDataType];
        if (dataType && propertyDetails) {
            const propCopy = { ...column.properties };
            addTitleToProperties({ propertyDetails, properties: propCopy });
            addColumnsToProperties({
                nodeTypes,
                dataTypes,
                dataType,
                properties: propCopy,
                propertyDetails,
            });
            addValueFieldToProperties({
                dataType,
                properties: propCopy,
                propertyDetails,
            });
            addHelperTextFieldToProperties({
                dataType,
                properties: propCopy,
                propertyDetails,
            });
            return {
                ...column,
                properties: propCopy,
                defaultProperties: propCopy,
            };
        }
    }
    return column;
}
//# sourceMappingURL=data-type-utils.js.map