import { FieldKey, objectKeys } from '@sage/xtrem-shared';
import { isArray, isFunction } from 'lodash';
import { fetchReferenceItems } from '../../component/field/reference/reference-utils';
import { RANGE, RANGE_DIVIDER, SET } from '../../component/types';
import { localize } from '../../service/i18n-service';
import { triggerFieldEvent } from '../events';
import { resolveByValue } from '../resolve-value-utils';
import { AUTO_COLUMN_ID, getGraphFilter, removeNumericPostfix, setTableContext } from '../table-component-utils';
import { cleanMetadataFromRecord, splitValueToMergedValue } from '../transformers';
import { COLUMN_ID_LINE_NUMBER, COLUMN_ID_ROW_SELECTION } from './ag-grid-column-config';
import { shouldDisplayColumnInPanel } from './ag-grid-service';
import { getGroupFilterValue } from '../../service/collection-data-utils';
import { fetchCollectionDataCount } from '../../service/graphql-service';
export const ROW_HEIGHT = 40;
export const GROUP_ROW_HEIGHT = ROW_HEIGHT * 1.25;
export const getFilterValueFromAgGridFilterStatement = (filterBase, filterKey) => {
    const filterType = filterBase.filterType;
    const type = filterBase.type;
    const lookup1 = filterType === 'date'
        ? filterBase.dateFrom
        : filterBase.filter;
    const lookup2 = filterType === 'date'
        ? filterBase.dateTo
        : filterType === 'number'
            ? filterBase.filterTo
            : filterType === 'text'
                ? filterBase.filterTo
                : '';
    const filterValue = type === RANGE ? `${lookup1}${RANGE_DIVIDER}${lookup2}` : lookup1;
    if (filterType === SET) {
        return {
            filterValue: filterBase.values,
            filterType: {
                text: `${filterKey} in list`,
                value: SET,
            },
            filterById: filterBase.filterById,
        };
    }
    return {
        filterValue,
        filterType: { text: `${filterKey} ${type} ${filterValue}`, value: type },
    };
};
export function callGridMethod(gridApi, method, ...args) {
    if (!gridApi || gridApi?.isDestroyed()) {
        return undefined;
    }
    return gridApi[method]?.(...args);
}
export function getSafeGridApiContext(cb, gridApi, method, ...args) {
    if (!gridApi || gridApi?.isDestroyed()) {
        return undefined;
    }
    const result = gridApi[method]?.(...args);
    return cb(result);
}
export const mapAgGridFilterToXtremFilters = (filterModel) => (filterKey) => {
    const filterBase = filterModel[filterKey];
    const operator = filterBase.operator;
    const formattedFilterKey = removeNumericPostfix(filterKey);
    if (operator && isArray(filterBase.conditions)) {
        // Combined Filter
        return {
            // When the table value is reset the column names are getting an _{NUMBER} postfix, we remove that
            id: formattedFilterKey,
            value: filterBase.conditions.map((c) => getFilterValueFromAgGridFilterStatement(c, filterKey)),
        };
    }
    // Single Filter
    return {
        // When the table value is reset the column names are getting an _{NUMBER} postfix, we remove that
        id: formattedFilterKey,
        value: [getFilterValueFromAgGridFilterStatement(filterBase, filterKey)],
    };
};
export const tryToCommitPhantomRow = async ({ api, screenId, value, elementId, }) => {
    const phantomRow = api.getPinnedTopRow(0);
    if (!phantomRow) {
        return null;
    }
    const dirtyColumns = new Set(api.getColumnDefs()?.map(c => c.field));
    value.setRecordValue({
        recordData: { _id: phantomRow.data._id },
        level: phantomRow.data.__level,
        toBeMarkedAsDirty: Array.from(dirtyColumns),
        isOrganicChange: true,
    });
    const recordData = value.getRawRecord({
        id: phantomRow.data._id,
        level: phantomRow.data.__level,
        cleanMetadata: false,
    });
    const validationResult = await value.runValidationOnRecord({ recordData });
    if (objectKeys(validationResult || {}).length === 0) {
        api.flashCells({ rowNodes: [phantomRow] });
        const { __level: level, __parentId: parentId } = recordData;
        value.commitPhantomRow({
            id: recordData._id,
            level,
            parentId,
            isOrganicChange: true,
        });
        setTimeout(() => {
            value.createNewPhantomRow({ level, parentId });
            triggerFieldEvent(screenId, elementId, 'onRowAdded', recordData._id, splitValueToMergedValue(cleanMetadataFromRecord(recordData)));
        }, 500);
        return null;
    }
    return validationResult;
};
export const getFirstEditableColumn = (api, data = {}, includeSelect = false) => {
    return (api.getColumnDefs() ?? [])
        .filter((colDef) => !colDef.hide &&
        ((includeSelect && colDef.colId === COLUMN_ID_ROW_SELECTION) ||
            (colDef.colId !== COLUMN_ID_ROW_SELECTION &&
                !!colDef.cellEditor &&
                (colDef.editable === true ||
                    (isFunction(colDef.editable) &&
                        colDef.editable({
                            data,
                            api,
                            colDef,
                        }))))))
        .map(c => c.colId)[0];
};
export const getFilterModel = (api, groupByColumnField) => {
    const filterModel = api?.getFilterModel();
    return filterModel == null
        ? undefined
        : objectKeys(filterModel).reduce((acc, curr) => {
            const key = curr === AUTO_COLUMN_ID ? groupByColumnField : curr;
            if (key != null) {
                acc[key] = filterModel[curr];
            }
            return acc;
        }, {});
};
export function getSelectionFilter({ groupByColumn, gridApi, screenId, isSelectAllChecked, tableFieldProperties, activeOptionsMenuItem, mode = 'server', }) {
    const groupFilter = [];
    if (groupByColumn) {
        gridApi.forEachNode(node => {
            if (node.group && node.data && node.isSelected()) {
                groupFilter.push(getGroupFilterValue({
                    group: {
                        key: node.data.__groupKey,
                        value: node.data._id.split('__group-')[1],
                        aggFunc: groupByColumn.aggFunc ?? undefined,
                        type: groupByColumn.type,
                    },
                    mode,
                }));
            }
        });
    }
    return {
        ...(isSelectAllChecked &&
            getGraphFilter({
                filterModel: getFilterModel(gridApi, groupByColumn?.field),
                screenId,
                tableFieldProperties,
                activeOptionsMenuItem,
            })),
        _id: isSelectAllChecked || groupFilter.length > 0
            ? {
                _nin: getAllUnselectedIds(gridApi),
            }
            : {
                _in: gridApi
                    .getSelectedNodes()
                    .filter(r => !r.group)
                    .map(row => row.data._id),
            },
        ...(groupFilter.length > 0 && { _or: groupFilter }),
    };
}
export const getAllUnselectedIds = (gridApi) => {
    const unselected = [];
    gridApi.forEachNode(node => {
        if (node.data && !node.isSelected() && !node.data.__isGroup) {
            unselected.push(node.data._id);
        }
    });
    return unselected;
};
export const getColumnStatesForColumnPanel = (screenId, columnDefinitions, columnStates) => {
    if (!columnDefinitions) {
        return [];
    }
    const columns = [];
    if (columnDefinitions[0]?.context.columnId === COLUMN_ID_LINE_NUMBER) {
        const columnState = (columnStates || []).find(state => COLUMN_ID_LINE_NUMBER && state.colId && state.colId === COLUMN_ID_LINE_NUMBER);
        columns.push({
            colId: COLUMN_ID_LINE_NUMBER,
            isHidden: columnState?.hide === true,
            isMandatory: false,
            isSpecialColumn: true,
            title: localize('@sage/xtrem-ui/table-line-number', 'Line number'),
        });
    }
    const columnsToDisplayInPanel = [
        ...columnDefinitions.filter(c => !!c.colId && shouldDisplayColumnInPanel(c.context.columnDefinition)),
    ];
    columns.push(...columnsToDisplayInPanel
        .sort((a, b) => {
        if (columnStates) {
            const aColumnStateIndex = columnStates.findIndex(acs => acs.colId === a.colId);
            const bColumnStateIndex = columnStates.findIndex(bcs => bcs.colId === b.colId);
            return aColumnStateIndex - bColumnStateIndex;
        }
        const aColumnDefinitionIndex = columnDefinitions.findIndex(acs => acs.colId === a.colId);
        const bColumnDefinitionIndex = columnDefinitions.findIndex(bcs => bcs.colId === b.colId);
        return aColumnDefinitionIndex - bColumnDefinitionIndex;
    })
        .map(c => {
        const columnState = (columnStates || []).find(state => c.colId && state.colId && state.colId === c.colId);
        const properties = c.context.columnDefinition?.properties;
        const title = resolveByValue({
            propertyValue: properties.title,
            rowValue: null,
            fieldValue: null,
            skipHexFormat: true,
            screenId,
        });
        return {
            colId: String(c.colId),
            title,
            isMandatory: !!properties.isMandatory || c.cellRenderer === 'agGroupCellRenderer',
            isHidden: columnState ? columnState.hide === true : !!properties.isHiddenOnMainField,
        };
    }));
    return columns;
};
export async function pasteToSelectedCell(api) {
    if (!api) {
        return;
    }
    const focusedCell = api.getFocusedCell();
    if (!focusedCell) {
        return;
    }
    const colDef = focusedCell.column?.getColDef();
    const editable = colDef?.editable;
    if (typeof editable === 'boolean' && !editable) {
        return;
    }
    const isKeyboardWriteEnabled = await navigator.permissions.query({ name: 'clipboard-write' });
    if (isKeyboardWriteEnabled.state !== 'granted') {
        return;
    }
    const text = await navigator.clipboard.readText();
    const rowNode = api.getDisplayedRowAtIndex(focusedCell.rowIndex);
    if (!rowNode) {
        return;
    }
    if (typeof editable === 'function' && !editable(rowNode.data)) {
        return;
    }
    const userColDef = focusedCell.column.getUserProvidedColDef();
    if (userColDef?.type === FieldKey.Reference) {
        const items = await fetchReferenceItems(text, userColDef);
        if (items.length === 1) {
            rowNode.setDataValue(focusedCell.column.getColId(), items[0].__collectionItem);
        }
        return;
    }
    rowNode.setDataValue(focusedCell.column.getColId(), text);
}
export async function copySelectedCellValue(cellValue) {
    const isKeyboardWriteEnabled = await navigator.permissions.query({ name: 'clipboard-write' });
    if (isKeyboardWriteEnabled.state !== 'granted') {
        return;
    }
    await navigator.clipboard.writeText(cellValue);
}
export const getTotalRecordCount = async ({ api, fieldProperties, activeOptionsMenuItem, screenId, node, groupByColumnField, }) => {
    const filter = getGraphFilter({
        filterModel: getFilterModel(api, groupByColumnField),
        screenId,
        tableFieldProperties: fieldProperties,
        activeOptionsMenuItem,
    });
    return new Promise(resolve => {
        if (fieldProperties._controlObjectType !== FieldKey.MultiReference) {
            fetchCollectionDataCount({ rootNode: node, filter: JSON.stringify(filter) }).then(totalRowCount => {
                resolve(totalRowCount);
            });
        }
        else {
            resolve(0);
        }
    }).then(totalRowCount => {
        setTableContext(api, c => {
            c.totalRowCount = totalRowCount;
        });
        return totalRowCount;
    });
};
//# sourceMappingURL=ag-grid-table-utils.js.map