import { objectKeys } from '@sage/xtrem-shared';
import { get, isEmpty, set, uniq } from 'lodash';
import { getReferencePath, getReferenceValueField } from '../component/field/reference/reference-utils';
import { ReadonlyFieldControlObject } from '../component/readonly-field-control-object';
import { ContainerKey, FieldKey } from '../component/types';
import { getScreenElement } from '../service/screen-base-definition';
import { GraphQLTypes } from '../types';
import { xtremConsole } from './console';
import { getLayoutChildren } from './layout-utils';
import { convertDeepBindToPath, convertDeepBindToPathNotNull } from './nested-field-utils';
import { isDevMode } from './window';
export function getNestedFieldElementId(field) {
    if (!field) {
        return null;
    }
    if (field.type === FieldKey.Reference || field.type === FieldKey.MultiReference) {
        const fieldProperties = field.properties;
        const valueField = getReferenceValueField(fieldProperties)?.split('.').join('__');
        const helperTextField = fieldProperties.helperTextField;
        if (!fieldProperties.bind || (!valueField && !helperTextField)) {
            throw new Error(`Missing information for the following reference field: ${JSON.stringify(field)}`);
        }
        return `${convertDeepBindToPathNotNull(fieldProperties.bind)}__${convertDeepBindToPathNotNull(valueField || String(helperTextField))}`;
    }
    return convertDeepBindToPath(field.properties.bind) ?? null;
}
export const filterFields = (controlObjects, requiredSections = [], pageDefinition, includeFieldsWithNoParent = false) => {
    let requiredFields = [];
    const hasLazyLoadedFields = !isEmpty(requiredSections) && pageDefinition;
    if (!isEmpty(requiredSections) && pageDefinition?.metadata.layout) {
        requiredFields = pageDefinition.metadata.layout.$items
            .filter(c => requiredSections.indexOf(String(c.$containerId)) !== -1)
            .flatMap(i => getLayoutChildren(i));
        if (includeFieldsWithNoParent) {
            objectKeys(controlObjects).forEach(key => {
                if (controlObjects[key] instanceof ReadonlyFieldControlObject &&
                    !getParentSection(pageDefinition, key)) {
                    requiredFields.push(key);
                }
            });
        }
        requiredFields = uniq(requiredFields);
    }
    return objectKeys(controlObjects)
        .filter(key => controlObjects[key] instanceof ReadonlyFieldControlObject &&
        (!hasLazyLoadedFields || requiredFields.includes(key)))
        .reduce((result, key) => ({ ...result, [key]: controlObjects[key] }), {});
};
export function getValueTypes(controlObjects, uiComponentProperties) {
    return objectKeys(controlObjects).reduce((result, key) => {
        if (controlObjects[key] instanceof ReadonlyFieldControlObject) {
            result[uiComponentProperties?.[key]?.bind ?? key] = controlObjects[key];
        }
        return result;
    }, {});
}
export const setDefaultProperties = (decoratorProperties, defaultProperties, controlObjectType, extensionPackageName) => {
    objectKeys(defaultProperties)
        .filter(key => get(decoratorProperties, key) === undefined)
        .forEach(key => {
        set(decoratorProperties, key, get(defaultProperties, key));
    });
    decoratorProperties._controlObjectType = controlObjectType || decoratorProperties._controlObjectType;
    if (!decoratorProperties._declaredInExtension) {
        decoratorProperties._declaredInExtension = extensionPackageName;
    }
    return decoratorProperties;
};
const hasParentInHierarchy = (screen, controlObject, elementId) => {
    const parentGetter = controlObject.parent;
    if (parentGetter) {
        const parent = parentGetter.apply(screen);
        if (parent?.id === elementId) {
            return true;
        }
        if (parent?.parent) {
            return hasParentInHierarchy(screen, parent, elementId);
        }
    }
    return false;
};
export const getContainerChildFields = (screen, controlObjects, elementId) => controlObjects.filter(controlObject => controlObject.id === elementId || hasParentInHierarchy(screen, controlObject, elementId));
export const getAllParents = (screen, fields) => {
    return fields.reduce((reduced, field) => {
        const parentGetter = field.parent;
        if (parentGetter) {
            const parent = parentGetter.apply(screen);
            return reduced.concat([parent, ...getAllParents(screen, [parent])]);
        }
        return reduced;
    }, []);
};
export function getParentSection(screenDefinition, elementId) {
    const elementProperties = screenDefinition.metadata.uiComponentProperties[elementId];
    if (!elementProperties) {
        return null;
    }
    if (elementProperties._controlObjectType === ContainerKey.Section) {
        return elementId;
    }
    const parentElement = elementProperties.parent?.apply(getScreenElement(screenDefinition));
    if (parentElement?.id) {
        return getParentSection(screenDefinition, parentElement.id);
    }
    return null;
}
export const getPropertyScalarType = (nodeTypes, propertyType, fieldType, valueField) => {
    // If we can't find a scalar type for the field, we prevent it from being displayed in the filter manager
    const selectFieldScalarType = fieldType === FieldKey.Select ? GraphQLTypes.Enum : null;
    const matchingScalarType = objectKeys(GraphQLTypes).find(typeKey => GraphQLTypes[typeKey] === propertyType);
    const referenceFieldPropertyType = valueField && nodeTypes[propertyType]
        ? get(nodeTypes[propertyType].properties.type, getReferencePath(valueField))
        : null;
    return selectFieldScalarType || matchingScalarType || referenceFieldPropertyType;
};
export function handleChange(bind, value, // TODO Type this properly
setFieldValue, validate, onChange) {
    setFieldValue(bind, value)
        .then(() => {
        let validationPromise = Promise.resolve(undefined);
        if (validate) {
            validationPromise = validate(bind, value);
        }
        else if (isDevMode()) {
            xtremConsole.info(`Validation is being skipped on ${bind} since it does not provide a validate handler`);
        }
        if (onChange && validationPromise) {
            // eslint-disable-next-line no-console
            validationPromise.then(() => onChange(value)).catch(xtremConsole.error);
        }
        else if (isDevMode()) {
            xtremConsole.info(`Change event cannot be triggered on ${bind} since it does not provide a change handler`);
        }
    })
        .catch(err => {
        xtremConsole.error(`Failed to set the value of ${bind}`, err);
    });
}
export const normalizeUnderscoreBind = (bind) => bind.split('__')[0];
export const findColumnDefinitionByBind = (columns, bind) => columns.find(c => getNestedFieldElementId(c) === bind || getNestedFieldElementId(c).startsWith(`${bind}__`));
export const filterColumnDefinitionByBind = (columns, bind) => columns.filter(c => getNestedFieldElementId(c) === bind || getNestedFieldElementId(c).startsWith(`${bind}__`));
//# sourceMappingURL=abstract-fields-utils.js.map