/**
 * @packageDocumentation
 * @module root
 * */
import { getStore } from '../redux';
import { refreshNavigationPanel } from '../redux/actions';
import { applyDefaultValues } from '../redux/actions/field-value-actions';
import { setNavigationPanelIsHidden } from '../redux/actions/navigation-panel-actions';
import { ATTACHMENTS_ELEMENT_ID, DUPLICATE_INDICATOR_QUERY_EDITOR_PARAM, NEW_PAGE, QUERY_PARAM_TUNNEL_SEGMENTS, } from '../utils/constants';
import { checkIfPageIsLoaded, getPageDefinitionFromState, getPagePropertiesFromPageDefinition, getPagePropertiesFromState, } from '../utils/state-utils';
import { DeveloperApi, ScreenBase } from './screen-base';
export class PageDeveloperApi extends DeveloperApi {
    constructor(page) {
        super(page);
        this._detailPanel = page._pageMetadata.detailPanelThunk ? page._pageMetadata.detailPanelThunk(page) : null;
        if (page.$header) {
            this._headerLineBlock = Object.seal(page._pageMetadata.layoutBlocks[page.$header.id]);
        }
    }
    get screenId() {
        return this.page.id;
    }
    /** Light-weight function that returns the ID of the page from the query parameters */
    get recordId() {
        const state = getStore().getState();
        const pageDefinition = getPageDefinitionFromState(this.screenBase._pageMetadata.screenId, state);
        const params = pageDefinition?.queryParameters || {};
        return params._id && params._id !== NEW_PAGE && !params[DUPLICATE_INDICATOR_QUERY_EDITOR_PARAM]
            ? String(params._id)
            : undefined;
    }
    get isNavigationPanelHidden() {
        checkIfPageIsLoaded(this.screenId);
        const pageDefinition = getPageDefinitionFromState(this.screenId);
        return pageDefinition?.navigationPanel?.isHidden || false;
    }
    set isNavigationPanelHidden(state) {
        checkIfPageIsLoaded(this.screenId);
        const dispatch = getStore().dispatch;
        setNavigationPanelIsHidden(state, this.screenId)(dispatch);
    }
    get hasAttachmentsDialog() {
        return !!this.screenBase._pageMetadata.attachmentInformation;
    }
    get attachments() {
        return (this.screenBase._pageMetadata.controlObjects[ATTACHMENTS_ELEMENT_ID] ||
            null);
    }
    get headerLineBlock() {
        return this._headerLineBlock;
    }
    get queryParameters() {
        const state = getStore().getState();
        const pageDefinition = getPageDefinitionFromState(this.screenBase._pageMetadata.screenId, state);
        const params = { ...(pageDefinition?.queryParameters || {}) };
        if (params._id === NEW_PAGE) {
            delete params._id;
        }
        return params;
    }
    /** Container that can be used to display additional information outside of the page layout */
    get detailPanel() {
        return this._detailPanel;
    }
    /** This page */
    get page() {
        return this.screenBase._pageMetadata.controlObjects[this.screenBase._pageMetadata.screenId];
    }
    /** Visualization mode */
    get mode() {
        const pageProperties = getPagePropertiesFromState(this.screenBase._pageMetadata.screenId);
        return pageProperties.mode || 'default';
    }
    async refreshNavigationPanel(recordAdded = false) {
        const store = getStore();
        const dispatch = store.dispatch;
        await dispatch(refreshNavigationPanel(this.screenBase._pageMetadata.screenId, recordAdded));
    }
    get isInDialog() {
        const state = getStore().getState();
        const pageDefinition = getPageDefinitionFromState(this.screenBase._pageMetadata.screenId, state);
        return Boolean(pageDefinition && pageDefinition.isInDialog);
    }
    get isInDuplicationDialog() {
        const state = getStore().getState();
        const pageDefinition = getPageDefinitionFromState(this.screenBase._pageMetadata.screenId, state);
        const params = pageDefinition?.queryParameters || {};
        return !!params[DUPLICATE_INDICATOR_QUERY_EDITOR_PARAM];
    }
    get isInTunnel() {
        const state = getStore().getState();
        const pageDefinition = getPageDefinitionFromState(this.screenBase._pageMetadata.screenId, state);
        return this.isInDialog && !!pageDefinition.queryParameters[QUERY_PARAM_TUNNEL_SEGMENTS];
    }
    get isNewPage() {
        const state = getStore().getState();
        const pageDefinition = getPageDefinitionFromState(this.screenBase._pageMetadata.screenId, state);
        const params = { ...(pageDefinition?.queryParameters || {}) };
        return !params._id || params._id === NEW_PAGE;
    }
    /**
     * Fetch default values from the server even if those fields are dirty.
     * If the `skipSet` argument is set to true, the values are requested from the server but not applied to the page.
     */
    async fetchDefaults(requestedFieldIds, skipSet = false) {
        checkIfPageIsLoaded(this.screenId);
        const store = getStore();
        const state = store.getState();
        const dispatch = store.dispatch;
        const screenDefinition = getPageDefinitionFromState(this.screenBase._pageMetadata.screenId, state);
        const pageProperties = getPagePropertiesFromPageDefinition(screenDefinition);
        if (pageProperties.isTransient) {
            throw new Error('Default values cannot be requested on a transient page.');
        }
        if (!pageProperties.node) {
            throw new Error('Default values cannot be requested on a page without a node.');
        }
        const resolvedFieldIds = requestedFieldIds.map(e => (typeof e === 'string' ? e : e.id));
        return applyDefaultValues({
            dispatch,
            screenDefinition,
            plugins: state.plugins,
            nodeTypes: state.nodeTypes,
            requestedFieldIds: resolvedFieldIds,
            skipDispatch: skipSet,
        });
    }
}
export class Page extends ScreenBase {
    constructor() {
        super();
        this.$ = new PageDeveloperApi(this);
    }
}
export class PageFragment extends Page {
}
PageFragment.prototype.___isPageFragment = true;
//# sourceMappingURL=page.js.map