import { CUSTOM_DATA_PROPERTY, objectKeys } from '@sage/xtrem-shared';
import { getStore } from '../redux';
import { addSchemaInfo } from '../redux/actions/common-actions';
import { GraphQLKind, GraphQLTypes, textStreams } from '../types';
import { getArtifactDescription } from '../utils/transformers';
import { executeGraphqlQuery } from './graphql-utils';
import * as metadataService from './metadata-service';
export const fetchNodeNames = async (onlyPublishedNodes = false) => {
    const response = await executeGraphqlQuery({
        query: {
            getNodeNames: {
                __args: {
                    onlyPublishedNodes,
                },
                name: true,
                title: true,
            },
        },
        endpoint: '/metadata',
    });
    if (Array.isArray(response.data?.getNodeNames)) {
        return response.data.getNodeNames
            .sort((a, b) => a.title.localeCompare(b.title))
            .reduce((prevValue, currentValue) => {
            return { ...prevValue, [currentValue.name]: currentValue.title };
        }, {});
    }
    throw new Error('Failed to load node names from the server');
};
function getPropertyName(propertyDetails) {
    if (propertyDetails?.isCustom) {
        const prefix = `${CUSTOM_DATA_PROPERTY}.`;
        if (propertyDetails?.name?.startsWith(prefix)) {
            return propertyDetails.name;
        }
        return `${prefix}${propertyDetails.name}`;
    }
    if (propertyDetails?.type === GraphQLTypes.OutputTextStream) {
        return `${propertyDetails.name}.value`;
    }
    if (!propertyDetails.name) {
        throw new Error('Property name is missing');
    }
    return propertyDetails.name;
}
function getEnumType(propertyDetails, enumTypes) {
    if (propertyDetails.type === GraphQLTypes.Enum && propertyDetails.enumType) {
        const { name } = getArtifactDescription(propertyDetails.enumType);
        return enumTypes[name]?.values;
    }
    return undefined;
}
function getNode(propertyDetails) {
    if (propertyDetails.kind === GraphQLKind.Object) {
        return propertyDetails.type;
    }
    if (propertyDetails.type === GraphQLTypes.Enum) {
        return propertyDetails.enumType;
    }
    return undefined;
}
const remapProperties = (properties, enumTypes, getCollections) => {
    return objectKeys(properties)
        .filter(propertyKey => propertyKey !== CUSTOM_DATA_PROPERTY &&
        propertyKey !== '_etag' &&
        (getCollections || !properties[propertyKey].isCollection))
        .reduce((prevValue, propertyKey) => {
        const propertyDetails = properties[propertyKey];
        if (propertyKey === '_id') {
            prevValue[propertyKey] = {
                ...propertyDetails,
                type: GraphQLTypes.IntOrString,
                kind: GraphQLKind.Scalar,
                name: propertyKey,
                canFilter: true,
                canSort: true,
                label: '_id',
                isStored: true,
                isOnInputType: false,
                isOnOutputType: true,
                dataType: '',
                targetNode: '',
                enumType: null,
                isCustom: false,
                isMutable: false,
            };
        }
        else {
            prevValue[propertyKey] = {
                ...propertyDetails,
                name: getPropertyName(propertyDetails),
                canFilter: propertyDetails?.canFilter || false,
                canSort: (propertyDetails.canSort && !textStreams.includes(propertyDetails.type)) ||
                    false,
                label: propertyDetails.title || propertyKey,
                isStored: propertyDetails.isStored || false,
                isOnInputType: propertyDetails.isOnInputType || false,
                isOnOutputType: propertyDetails.isOnOutputType || false,
                dataType: propertyDetails.dataType || '',
                targetNode: propertyDetails.targetNode || '',
                enumType: propertyDetails.enumType || null,
                isCustom: propertyDetails.isCustom || false,
                isMutable: propertyDetails.isMutable || false,
                enumValues: getEnumType(propertyDetails, enumTypes),
                node: getNode(propertyDetails),
            };
        }
        return prevValue;
    }, {});
};
export const fetchNodeDetails = async ({ locale, nodeName, getCollections = false, withSortValue = false, }) => {
    const store = getStore();
    const nodeDetailsResponse = await metadataService.fetchNodeDetails([nodeName], 1, undefined, withSortValue);
    store.dispatch(addSchemaInfo(nodeDetailsResponse.nodeTypes, nodeDetailsResponse.dataTypes, nodeDetailsResponse.enumTypes, locale));
    const nodeDetails = nodeDetailsResponse.nodeTypes[nodeName];
    if (!nodeDetails) {
        throw new Error(`No details found for node: ${nodeName}`);
    }
    return remapProperties(nodeDetails.properties, nodeDetailsResponse.enumTypes, getCollections);
};
export const fetchNodePackageName = async (nodeName) => {
    const packageNames = await metadataService.fetchNodePackageNames([nodeName], 1);
    return packageNames[0];
};
export const mapNodeDetailsToTreeProperty = (parent, nodeDetails) => {
    return nodeDetails.map(nodeInfo => ({
        data: {
            ...nodeInfo,
            canSort: parent.data.canSort && nodeInfo.canSort,
            canFilter: parent.data.canFilter && nodeInfo.canFilter,
        },
        key: parent.key ? `${parent.key}.${nodeInfo.name}` : nodeInfo.name,
        id: parent.key ? `${parent.key}.${nodeInfo.name}` : nodeInfo.name,
        labelKey: parent.label ? `${parent.label}.${nodeInfo.label}` : nodeInfo.label,
        labelPath: parent.label ? `${parent.label}.${nodeInfo.label}` : nodeInfo.label,
        label: nodeInfo.label,
        canBeExpanded: (nodeInfo.kind === 'OBJECT' && !textStreams.includes(nodeInfo.type)) ||
            nodeInfo.kind === 'INTERFACE',
        canBeSelected: (nodeInfo.kind !== 'OBJECT' || textStreams.includes(nodeInfo.type)) &&
            nodeInfo.kind !== 'INTERFACE',
    }));
};
export const rawNodeDetailsToTreeProperty = (parent, rawNodeDetails, getCollections = false) => {
    const { nodeTypes, enumTypes } = metadataService.processNodeDetailsResponse(rawNodeDetails);
    const nodeDetails = nodeTypes[parent.data.type];
    const remappedNodeDetails = remapProperties(nodeDetails.properties, enumTypes, getCollections);
    return mapNodeDetailsToTreeProperty(parent, Object.values(remappedNodeDetails));
};
//# sourceMappingURL=node-information-service.js.map