import { cloneDeep, isNil, isString } from 'lodash';
import { xtremConsole } from '../utils/console';
import { capitalize, getArtifactDescription } from '../utils/transformers';
import { hasNodeCacheLogging, isDevMode } from '../utils/window';
import { queryToGraphQuery } from './graphql-utils';
import { sha256Hash } from './encryption-service';
export class NodeCacheService {
    static { this.cache = {}; }
    static async getCacheKeys(node, query) {
        const nodeDescription = getArtifactDescription(node);
        const stringifiedQuery = isString(query) ? query.toString() : queryToGraphQuery(query);
        const queryHash = await sha256Hash(stringifiedQuery);
        return { nodeName: nodeDescription.name, queryHash };
    }
    static setGlobalCache(cache) {
        if (isDevMode()) {
            window.__XTREM_NODE_CACHE = cache;
        }
    }
    static flushCache(nodeNames) {
        const nodeNamesToFlush = nodeNames ?? Object.keys(NodeCacheService.cache);
        if (hasNodeCacheLogging()) {
            xtremConsole.log(`%cNODE CACHE - %cFLUSH %c${nodeNamesToFlush.join(', ')}`, 'font-weight: bold;', 'color: orange; font-weight: bold;', 'font-weight: bold;');
        }
        nodeNamesToFlush.forEach(nodeName => {
            delete NodeCacheService.cache[nodeName];
        });
        NodeCacheService.setGlobalCache(NodeCacheService.cache);
    }
    static flushCacheAfterMutation(mutation) {
        Object.keys(mutation).forEach(packageName => {
            Object.keys(mutation[packageName]).forEach(nodeName => {
                const node = capitalize(nodeName);
                delete NodeCacheService.cache[node];
            });
        });
        NodeCacheService.setGlobalCache(NodeCacheService.cache);
    }
    static async get(fullNodeName, query) {
        const { nodeName, queryHash } = await NodeCacheService.getCacheKeys(fullNodeName, query);
        const cachedItem = NodeCacheService.cache[nodeName]?.[queryHash];
        if (hasNodeCacheLogging()) {
            const isHit = !isNil(cachedItem);
            xtremConsole.log(`%cNODE CACHE - %c${isHit ? 'HIT' : 'MISS'} %c${nodeName}\n%c${query.toString()}`, 'font-weight: bold;', isHit ? 'color: green; font-weight: bold;' : 'color: red; font-weight: bold;', 'font-weight: bold;', 'font-weight: lighter; font-size: xx-small;');
        }
        return cloneDeep(cachedItem);
    }
    static async set(fullNodeName, query, result) {
        const { nodeName, queryHash } = await NodeCacheService.getCacheKeys(fullNodeName, query);
        if (!NodeCacheService.cache[nodeName]) {
            NodeCacheService.cache[nodeName] = {};
        }
        NodeCacheService.cache[nodeName][queryHash] = cloneDeep(result);
        NodeCacheService.setGlobalCache(NodeCacheService.cache);
    }
}
//# sourceMappingURL=node-cache-service.js.map