import lokijs, { LokiOps } from '@sage/lokijs';
import { deepMerge, objectKeys } from '@sage/xtrem-shared';
import { cloneDeep, get, groupBy, initial, isEmpty, isNil, mergeWith, noop } from 'lodash';
import { navigationPanelId } from '../component/container/navigation-panel/navigation-panel-types';
import { xtremConsole } from '../utils/console';
import { SERVER_VALIDATION_RULE_PREFIX } from '../utils/constants';
import { transformToLokiJsFilter } from '../utils/lokijs-filter-transformer';
import { cleanMetadataFromRecord } from '../utils/transformers';
import { isDevMode } from '../utils/window';
import { RecordActionType } from './collection-data-types';
import { sortResultSet } from './collection-data-utils';
const mappingNoop = (d) => d;
const COMPOSITE_KEY_SEPARATOR = '.';
export const extendLokiOperators = () => {
    // Custom operator
    LokiOps.$empty = (obj, flag) => {
        const empty = isEmpty(obj);
        return flag ? empty : !empty;
    };
};
extendLokiOperators();
// eslint-disable-next-line new-cap
const lokiDb = new lokijs('CollectionDatabase');
export const destroyScreenCollections = (screenId, keepNavigationPanelDb = false) => lokiDb
    .listCollections()
    .filter(c => c.name.indexOf(`${screenId}-`) === 0)
    .filter(c => !keepNavigationPanelDb || c.name.indexOf(navigationPanelId) === -1)
    .forEach(c => lokiDb.removeCollection(c.name));
export class LokiDb {
    static getCollectionId(screenId, elementId, dbKey) {
        return `${screenId}-${elementId}${dbKey ? `-${dbKey}` : ''}`;
    }
    constructor({ screenId, elementId, initialValues, isTransient, dbKey }) {
        this.minId = 0;
        this.result = null;
        this.screenId = screenId;
        this.elementId = elementId;
        const collectionId = LokiDb.getCollectionId(screenId, elementId, dbKey);
        if (lokiDb.getCollection(collectionId)) {
            lokiDb.removeCollection(collectionId);
        }
        this.collection = lokiDb.addCollection(collectionId, {
            unique: ['__path'],
            indices: ['__dirty', '__action', '_id', '__path', '__compositeKey', '__level', '__uncommitted'],
            disableMeta: true,
            caseInsensitive: true,
        });
        initialValues.forEach(initialValue => {
            const hasId = Object.prototype.hasOwnProperty.call(initialValue, '_id');
            this.addRecord({
                data: { ...initialValue },
                dirty: !hasId,
                action: !hasId && !isTransient ? RecordActionType.ADDED : 'none',
            });
        });
    }
    static groupBy(input, groupKey) {
        return groupBy(input, element => get(element, groupKey));
    }
    calculateAggregatedValue({ aggregationKey, aggregationMethod, ...findAllParams }) {
        const items = this.findAll({ ...findAllParams }).value();
        const getSum = () => {
            return items.reduce((acc, item) => {
                const value = get(item, aggregationKey);
                if (value != null && Number.isNaN(Number(value))) {
                    throw new Error(`Unsupported sum aggregation on value "${value}"`);
                }
                return value == null ? acc : (acc || 0) + value;
            }, Number.NaN);
        };
        switch (aggregationMethod) {
            case 'min':
                return items.reduce((acc, item) => {
                    const value = get(item, aggregationKey);
                    return value == null || (acc || Number.MAX_SAFE_INTEGER) < value ? acc : value;
                }, Number.NaN);
            case 'max':
                return items.reduce((acc, item) => {
                    const value = get(item, aggregationKey);
                    return value == null || (acc || Number.MIN_SAFE_INTEGER) > value ? acc : value;
                }, Number.NaN);
            case 'sum':
                return getSum();
            case 'avg':
                const sum = getSum();
                return Number.isNaN(sum) ? sum : sum / items.length;
            case 'count':
                return items.reduce((acc, item) => {
                    return get(item, aggregationKey) ? (acc || 0) + 1 : acc;
                }, Number.NaN);
            default:
                throw new Error(`Unsupported group method: "${aggregationMethod}"`);
        }
    }
    withTemporaryRecords(body, temporaryRecords = []) {
        let cleanup = noop;
        try {
            const references = temporaryRecords
                .filter(data => isNil(this.collection.findOne({ _id: { $eq: data._id } })))
                .map(data => {
                const addedRecord = this.collection.add(this.createRecord({ data, action: RecordActionType.TEMPORARY }));
                return addedRecord.$loki;
            });
            cleanup = () => references.forEach(r => this.collection.remove(r));
            return body(this);
        }
        finally {
            cleanup();
        }
    }
    getCompositeKey({ _id, __level = 0, __isGroup: isGroup, __groupKey: groupKey, __uncommitted, }) {
        if (isGroup === undefined || isGroup === false) {
            return `${_id}${COMPOSITE_KEY_SEPARATOR}${__level}${__uncommitted ? '_UNCOMMITTED' : ''}`;
        }
        return `${_id}${COMPOSITE_KEY_SEPARATOR}${__level}${COMPOSITE_KEY_SEPARATOR}${groupKey ?? ''}${COMPOSITE_KEY_SEPARATOR}${__uncommitted ? '_UNCOMMITTED' : ''}group`;
    }
    getPathProperty(record) {
        const path = [String(record.__level || 0), ...this.getIdPathToRecordByRecord(record)].join(',');
        return record.__uncommitted ? `${path},uncommitted` : path;
    }
    getIdAndLevelFromCompositeKey(compositeKey) {
        const parts = compositeKey.split(COMPOSITE_KEY_SEPARATOR);
        if (parts.length === 0) {
            throw new Error(`Invalid composite key, separator not found: ${compositeKey}`);
        }
        return {
            _id: parts[0],
            __level: isNil(parts[1]) ? undefined : Number(parts[1]),
        };
    }
    ensureResult() {
        if (this.result === null) {
            throw new Error('Empty result!');
        }
    }
    generateIndex() {
        const ids = this.collection
            .chain()
            .data({ removeMeta: true })
            .map(r => cloneDeep(r))
            .reduce((previousValue, currentRow) => {
            const id = parseInt(currentRow._id || '', 10);
            if (!currentRow.__isGroup && Number.isInteger(id)) {
                previousValue.push(id);
            }
            return previousValue;
        }, [0]);
        this.minId = Math.min(...ids, this.minId) - 1;
        return String(this.minId);
    }
    /** Prepare the collection information for server side processing. Generated IDs and LokJS flags removed */
    exportDataForServerProcessing() {
        return this.collection
            .chain()
            .data({ removeMeta: true })
            .map(r => cloneDeep(r))
            .map(({ _id, ...rest }) => {
            const updatedRecord = Number.parseInt(_id, 10) < 0 ? { ...rest } : { _id, ...rest };
            // The __action field is not removed as the server needs that value for knowing how the change is applied
            return cleanMetadataFromRecord(updatedRecord, false);
        });
    }
    findOne({ id, level = 0, parentId, where = {}, cleanMetadata = true, includeUnloaded = false, isUncommitted = false, }) {
        const result = this.collection.findOne({
            ...where,
            ...{ __uncommitted: isUncommitted ? { $eq: true } : { $in: [0, undefined] } },
            ...(id && { _id: { $eq: String(id) } }),
            __level: level === 0 ? { $in: [0, undefined] } : { $eq: level },
            ...(parentId && { __parentId: { $eq: parentId } }),
            ...(!includeUnloaded && { __unloaded: { $ne: true } }),
        });
        return cleanMetadata ? cleanMetadataFromRecord(cloneDeep(result)) : cloneDeep(result);
    }
    findAll({ level, parentId, where = {}, includeUnloaded = false, includePhantom = false, isUncommitted = false, } = {}) {
        const parentCondition = parentId === undefined
            ? { __parentId: { $exists: false } }
            : parentId === null
                ? {}
                : { __parentId: { $eq: parentId } };
        const levelCondition = level === undefined || level === 0
            ? { __level: { $in: [0, undefined] } }
            : level === null
                ? {}
                : { __level: { $eq: level } };
        const unloadedCondition = !includeUnloaded ? { __unloaded: { $ne: true } } : {};
        const uncommittedCondition = { __uncommitted: isUncommitted ? { $eq: true } : { $in: [0, undefined] } };
        this.result = (this.result || this.collection.chain())
            .find({ ...(!includePhantom && { __phantom: { $ne: true } }), ...where })
            .find({
            ...levelCondition,
            ...parentCondition,
            ...unloadedCondition,
            ...uncommittedCondition,
        });
        return this;
    }
    where(fun) {
        this.result = (this.result || this.collection.chain()).where(fun);
        return this;
    }
    clear() {
        this.collection.clear();
        return this;
    }
    remove({ data, beforeRemove = noop, afterRemove = noop, }) {
        beforeRemove();
        this.collection.remove(data);
        afterRemove();
        return this;
    }
    sort({ columnDefinitions, orderBy, }) {
        if (orderBy) {
            this.ensureResult();
            this.result = sortResultSet({
                resultSet: this.result,
                orderBy,
                columnDefinitions,
            });
        }
        return this;
    }
    filter({ where, groupByKey }) {
        if (where && !isEmpty(where)) {
            this.ensureResult();
            const filter = groupByKey
                ? { _or: [{ ...this.filter, __isGroup: { _neq: true } }, { __isGroup: { _eq: true } }] }
                : this.filter;
            const lokiJsFilter = transformToLokiJsFilter(filter);
            try {
                this.result = this.result.find(transformToLokiJsFilter(lokiJsFilter));
            }
            catch (e) {
                if (isDevMode()) {
                    xtremConsole.error(e);
                    xtremConsole.error(`Failed to filter dataset using in-memory database for ${this.screenId} - ${this.elementId}`);
                    xtremConsole.error('GraphQL filter:');
                    xtremConsole.error(JSON.stringify(filter, null, 4));
                    xtremConsole.error('LokiJS filter:');
                    xtremConsole.error(JSON.stringify(lokiJsFilter, null, 4));
                }
            }
        }
        return this;
    }
    update({ data, beforeUpdate = mappingNoop, afterUpdate = noop, cleanMetadata = true, }) {
        const updated = beforeUpdate(data);
        this.collection.update(updated);
        afterUpdate(updated);
        return cleanMetadata ? cleanMetadataFromRecord(updated) : updated;
    }
    getIdPathToRecordByIdAndLevel({ id, level, }) {
        const record = this.findOne({ id, level, cleanMetadata: false });
        return this.getIdPathToRecordByRecord(record);
    }
    getIdPathToRecordByRecord(startRecord) {
        const result = [];
        let record = startRecord;
        if (!record) {
            return [];
        }
        if (record._id) {
            result.push(record._id);
        }
        while (record?.__parentId) {
            // eslint-disable-next-line no-prototype-builtins
            record = record.hasOwnProperty('__level')
                ? this.findOne({ id: record.__parentId, level: record.__level - 1, cleanMetadata: false })
                : this.findOne({ id: record.__parentId, cleanMetadata: false });
            if (record && record._id) {
                result.push(record._id);
            }
        }
        return result.reverse();
    }
    createRecord({ data, level, dirty = true, action = RecordActionType.ADDED, beforeCreate = mappingNoop, parentId, isUncommitted = false, }) {
        const _id = !Object.prototype.hasOwnProperty.call(data, '_id') ||
            data._id === null ||
            data._id === undefined
            ? this.generateIndex()
            : String(data._id);
        const actualAction = action === 'none' ? undefined : action;
        const record = {
            ...(data && beforeCreate(cloneDeep(data))),
            _id,
            __dirty: dirty,
            // If we are creating an uncommitted row by cloning an existing one, we need to keep the dirty status of the columns
            __dirtyColumns: !isUncommitted
                ? new Set()
                : data.__dirtyColumns || new Set(),
            ...(actualAction && { __action: actualAction }),
            ...(level !== undefined && level !== 0 && { __level: level }),
            ...(parentId !== undefined && { __parentId: parentId }),
            ...(isUncommitted && { __uncommitted: true }),
        };
        record.__compositeKey = this.getCompositeKey(record);
        record.__path = this.getPathProperty(record);
        return record;
    }
    addRecord({ data, level, cleanMetadata = true, dirty = true, action = RecordActionType.ADDED, beforeInsert = (d) => d, afterInsert = noop, parentId, isUncommitted, }) {
        const record = this.createRecord({
            data,
            level,
            dirty,
            action,
            beforeCreate: beforeInsert,
            parentId,
            isUncommitted,
        });
        this.collection.add(record);
        if (action !== 'none') {
            afterInsert({ record: cloneDeep(record), action });
        }
        return cleanMetadata ? cleanMetadataFromRecord(cloneDeep(record)) : cloneDeep(record);
    }
    skip(offset) {
        this.ensureResult();
        this.result = this.result.offset(offset);
        return this;
    }
    take(limit) {
        this.ensureResult();
        this.result = this.result.limit(limit);
        return this;
    }
    getAncestorIds({ id, level }) {
        return initial(this.getIdPathToRecordByIdAndLevel({ id, level }));
    }
    value({ cleanMetadata = true, cleanDatabaseMeta = true, } = {}) {
        this.ensureResult();
        const data = this.result.data({ removeMeta: cleanDatabaseMeta }).map(cloneDeep);
        const result = cleanMetadata ? data.map((r) => cleanMetadataFromRecord(r)) : data;
        this.result = null;
        return result;
    }
    getValidationStateByColumn(levels) {
        return Array.from({ length: levels })
            .map((_, index) => index)
            .map(level => {
            return mergeWith({}, ...this.getAllInvalidRecords({ level, parentId: null, includePhantom: true })
                .value({ cleanMetadata: false })
                .map(r => r.__validationState), (s = [], o) => [...s, o]);
        });
    }
    getAllServerRecordWithServerSideValidationIssues() {
        return (this.collection
            .chain()
            .find({
            // Deleted or temporary records should not be validated
            __action: { $nin: [RecordActionType.REMOVED, RecordActionType.TEMPORARY] },
            __phantom: { $ne: true },
        })
            // Find items where there is a server side validation errors
            .where(r => !!r.__validationState &&
            Object.values(r.__validationState)
                .map(v => v.validationRule)
                .filter(k => k.startsWith(SERVER_VALIDATION_RULE_PREFIX)).length > 0)
            .data()
            .map(r => r.__compositeKey));
    }
    getInvalidLoadedRecords({ level, parentId, withoutServerErrors, includePhantom = false, } = {}) {
        return this.findAll({
            where: {
                __action: { $nin: [RecordActionType.REMOVED, RecordActionType.TEMPORARY] },
                __validationState: { $ne: undefined },
            },
            level,
            parentId,
            includePhantom,
        }).where(r => {
            const validationRules = objectKeys(r.__validationState || {}).map(k => r.__validationState[k].validationRule);
            return (validationRules.length > 0 &&
                (!withoutServerErrors ||
                    validationRules.filter(vr => !vr.startsWith(SERVER_VALIDATION_RULE_PREFIX)).length > 0));
        });
    }
    getInvalidUnloadedRecords({ level, parentId, withoutServerErrors, includePhantom = false, } = {}) {
        return this.findAll({
            where: {
                __action: { $nin: [RecordActionType.REMOVED, RecordActionType.TEMPORARY] },
                __unloaded: { $eq: true },
            },
            level,
            parentId,
            includeUnloaded: true,
            includePhantom,
        }).where(r => !!r.__validationState &&
            objectKeys(r.__validationState).length > 0 &&
            (!withoutServerErrors ||
                objectKeys(r.__validationState).filter(k => k.indexOf(SERVER_VALIDATION_RULE_PREFIX) === 0).length >
                    0));
    }
    getAllInvalidRecords({ level, parentId, includePhantom = false, } = {}) {
        return this.findAll({
            where: {
                __action: { $ne: RecordActionType.TEMPORARY },
            },
            level,
            parentId,
            includeUnloaded: true,
            includePhantom,
        }).where(r => !!r.__validationState && objectKeys(r.__validationState).length > 0);
    }
    isCollectionValid() {
        return (this.collection
            .chain()
            // Deleted or temporary records should not be validated
            .find({
            __action: { $nin: [RecordActionType.REMOVED, RecordActionType.TEMPORARY] },
        })
            .where(r => !!r.__validationState && objectKeys(r.__validationState).length > 0)
            .count() === 0);
    }
    isCollectionDirty() {
        return (this.collection
            .chain()
            .find({ __dirty: { $eq: true } })
            .count() > 0);
    }
    startRecordTransaction({ recordId, recordLevel, }) {
        const result = this.collection.find({
            _id: { $eq: String(recordId) },
            __level: recordLevel === 0 ? { $in: [0, undefined] } : { $eq: recordLevel },
        });
        if (result.length === 0) {
            throw new Error('Cannot start transaction on a non-existing record');
        }
        if (result.length > 1) {
            throw new Error('Too many records found, maybe a transaction has already been started on the record');
        }
        const record = cloneDeep(result[0]);
        delete record.$loki; // TODO: Set it correctly
        record.__uncommitted = true;
        this.collection.add(this.createRecord({
            data: record,
            isUncommitted: true,
            action: record.__action || RecordActionType.MODIFIED,
        }));
    }
    // The return value indicates if the committed record was a new row.
    commitRecord({ recordId, recordLevel }) {
        const uncommittedRecord = this.collection.find({
            _id: { $eq: String(recordId) },
            __level: recordLevel === 0 ? { $in: [0, undefined] } : { $eq: recordLevel },
            __uncommitted: { $eq: true },
        });
        if (uncommittedRecord.length === 0) {
            throw new Error(`Cannot commit a record that is not in transaction ${recordId} ${recordLevel || ''}`);
        }
        if (uncommittedRecord.length > 1) {
            throw new Error('Too many records found, the dataset seems to be corrupted');
        }
        let updatedRecord;
        if (uncommittedRecord[0].__phantom) {
            // In case of a new record we don't need to look up the original record, given that doesn't exist.
            updatedRecord = {
                ...uncommittedRecord[0],
                __uncommitted: undefined,
                __phantom: undefined,
            };
            updatedRecord.__compositeKey = this.getCompositeKey(updatedRecord);
            updatedRecord.__path = this.getPathProperty(updatedRecord);
            this.collection.update(updatedRecord);
            return true;
        }
        // For updates we need to merge the uncommitted changes on the top of the original record, then update the original record and remove the uncommitted working copy
        const originalRecord = this.collection.find({
            _id: { $eq: String(recordId) },
            __level: recordLevel === 0 ? { $in: [0, undefined] } : { $eq: recordLevel },
            __uncommitted: { $in: [false, undefined] },
        });
        if (originalRecord.length === 0) {
            throw new Error(`Cannot find original record of the transaction ${recordId} ${recordLevel || ''}`);
        }
        updatedRecord = deepMerge(originalRecord[0], uncommittedRecord[0], false, true);
        updatedRecord.$loki = originalRecord[0].$loki;
        updatedRecord.__uncommitted = undefined;
        updatedRecord.__compositeKey = originalRecord[0].__compositeKey;
        updatedRecord.__path = originalRecord[0].__path;
        this.collection.update(updatedRecord);
        // We need to remove the uncommitted working copy from the database because now it would be duplicated
        this.collection.remove(uncommittedRecord);
        return false;
    }
    cancelRecordTransaction({ recordId, recordLevel, }) {
        const uncommittedRecord = this.collection.find({
            _id: { $eq: String(recordId) },
            __level: recordLevel === 0 ? { $in: [0, undefined] } : { $eq: recordLevel },
            __uncommitted: { $eq: true },
        });
        if (uncommittedRecord.length === 0) {
            throw new Error(`Cannot cancel a record that is not in transaction ${recordId} ${recordLevel || ''}`);
        }
        if (uncommittedRecord.length > 1) {
            throw new Error('Too many records found, the dataset seems to be corrupted');
        }
        this.collection.remove(uncommittedRecord);
    }
}
//# sourceMappingURL=loki.js.map