import { isValidIsoDate, isValidLocalizedDate, parseIsoDate, parseLocalizedDate } from '@sage/xtrem-date-time';
import { deepMerge, flat, objectKeys } from '@sage/xtrem-shared';
import { EnumType } from 'json-to-graphql-query';
import { camelCase, get, has, isEmpty, isPlainObject, isUndefined, mapValues, merge, omit, reduceRight, remove, set, uniq, without, } from 'lodash';
import { getReferencePath, getReferenceValueFieldPath } from '../../component/field/reference/reference-utils';
import { withoutNestedTechnical, } from '../../component/nested-fields';
import { FieldKey } from '../../component/types';
import { GraphQLKind, GraphQLTypes } from '../../types';
import { filterFields, getNestedFieldElementId } from '../../utils/abstract-fields-utils';
import { deepFindPropertyValues } from '../../utils/common-util';
import { xtremConsole } from '../../utils/console';
import { HEADER_IMAGE, HEADER_TITLE, NEW_PAGE } from '../../utils/constants';
import { convertDeepBindToPath, convertDeepBindToPathNotNull, getNestedFieldsFromProperties, getTopLevelBindFromNestedBind, } from '../../utils/nested-field-utils';
import { findDeepPropertyDetails, findDeepPropertyType } from '../../utils/node-utils';
import { parseScreenValues, pascalCase, schemaTypeNameFromNodeName } from '../../utils/transformers';
import { AllAndOnly, reduceDict } from '../../utils/type-utils';
import { isDevMode } from '../../utils/window';
import { CollectionValue } from '../collection-data-service';
import { CollectionFieldTypes } from '../collection-data-types';
import { mergeGraphQLFilters } from '../filter-service';
import { escapeStringRegexp, wrapQuery } from '../graphql-utils';
import { getScreenElement } from '../screen-base-definition';
import { formatInputProperty } from '../value-serializer-service';
import { getFieldProperties } from './field-query-builder';
export const RESTRICTED_COLUMN_PREFIX = '__restricted_';
export const RESTRICTED_COLUMN_ALL_PREFIX = '__all_';
export const quantifiers = AllAndOnly()(['_atLeast', '_atMost', '_every', '_none']);
export * from './field-query-builder';
export const filterReservedWords = AllAndOnly()([
    '_and',
    '_or',
    '_not',
    '_eq',
    '_ne',
    '_in',
    '_nin',
    '_contains',
    '_containsRange',
    '_containedBy',
    'start',
    'end',
    '_gt',
    '_gte',
    '_lt',
    '_lte',
    '_atLeast',
    '_atMost',
    '_every',
    '_none',
    '_regex',
    '_options',
    '_mod',
]);
const filterReducer = reduceDict();
const DATA_TYPE_ENUM_VALUES_FRAGMENT = {
    value: true,
    title: true,
};
const DATA_TYPE_FIELD_FRAGMENT = {
    bind: true,
    title: true,
    type: true,
};
export const DATA_TYPE_FRAGMENT = {
    name: true,
    type: true,
    title: true,
    node: true,
    precision: true,
    scale: true,
    maxLength: true,
    value: DATA_TYPE_FIELD_FRAGMENT,
    helperText: DATA_TYPE_FIELD_FRAGMENT,
    imageField: DATA_TYPE_FIELD_FRAGMENT,
    columns: DATA_TYPE_FIELD_FRAGMENT,
    values: DATA_TYPE_ENUM_VALUES_FRAGMENT,
    tunnelPage: true,
    tunnelPageId: DATA_TYPE_FIELD_FRAGMENT,
};
export function getNodeDetailsPropertiesFragment() {
    return {
        name: true,
        title: true,
        canSort: true,
        canFilter: true,
        type: true,
        isCustom: true,
        isMutable: true,
        isStored: true,
        isOnInputType: true,
        isOnOutputType: true,
        enumType: true,
        dataType: true,
        dataTypeDetails: DATA_TYPE_FRAGMENT,
        targetNode: true,
    };
}
export function getNodeDetailsFragment() {
    return {
        packageName: true,
        name: true,
        title: true,
        defaultDataType: true,
        defaultDataTypeDetails: DATA_TYPE_FRAGMENT,
        hasAttachments: true,
        hasNotes: true,
        properties: getNodeDetailsPropertiesFragment(),
        mutations: {
            name: true,
            title: true,
            parameters: {
                name: true,
                title: true,
            },
        },
    };
}
export function getNodeCustomDetailsFragment() {
    return {
        name: true,
        properties: getNodeDetailsPropertiesFragment(),
    };
}
export const getPlatformStringLiteralsSchemaFragment = () => ({
    __args: {
        filter: {
            packageOrPage: '@sage/xtrem-ui,@sage/xtrem-date-time,@sage/xtrem-ui-components,@sage/xtrem-document-editor',
        },
    },
    key: true,
    content: true,
});
export function getBindAndSelectorFromDeepBinding(bind, contextNode, nodeTypes, alias) {
    const remapAlias = (s) => {
        if (alias) {
            const copy = [...s];
            copy[0] = alias;
            return copy.join('.');
        }
        return s.join('.');
    };
    const valuePath = convertDeepBindToPathNotNull(bind);
    const segments = valuePath.split('.');
    const node = schemaTypeNameFromNodeName(contextNode);
    for (let i = 0; i < segments.length; i += 1) {
        const bindSegments = segments.slice(0, i + 1);
        const type = findDeepPropertyDetails(node, bindSegments.join('.'), nodeTypes);
        if (type?.type === GraphQLTypes.Json) {
            const jsonSelector = segments.slice(i + 1).join('.');
            return {
                valuePath: remapAlias(bindSegments),
                jsonSelector,
            };
        }
    }
    return {
        valuePath: remapAlias(segments),
    };
}
export const addTotalCountTransform = (filter, initial) => filterReducer(filter, initial, (_value, acc, key, path, source) => {
    if (quantifiers.includes(key)) {
        const parentKey = path[path.length - 2];
        const parentPath = path.slice(0, -1);
        const parentValue = get(source, parentPath);
        const fieldName = pascalCase(parentKey);
        acc[`${RESTRICTED_COLUMN_ALL_PREFIX}${fieldName}`] = {
            __aliasFor: parentKey,
            query: {
                totalCount: true,
            },
        };
        acc[`${RESTRICTED_COLUMN_PREFIX}${fieldName}`] = {
            __aliasFor: parentKey,
            query: {
                totalCount: true,
                __args: { filter: JSON.stringify(omit(parentValue, [key])) },
            },
        };
        delete acc[parentKey];
    }
    return acc;
});
export const addReferenceIdsAssign = (filter, node, nodeTypes) => filterReducer(filter, {}, (_value, acc, _key, path) => {
    const isReserved = (k) => filterReservedWords.includes(k);
    const cleanPath = remove([...path], k => typeof k !== 'number' && !isReserved(k));
    const targetType = findDeepPropertyDetails(schemaTypeNameFromNodeName(node), cleanPath.join('.'), nodeTypes, true);
    const isNodeProperty = cleanPath.length === 1;
    const isNodeReference = cleanPath.length >= 2;
    const isNode = (isNodeProperty || isNodeReference) && targetType?.type !== GraphQLTypes.Json;
    if (isNodeProperty && targetType?.kind === GraphQLKind.Object) {
        return merge(acc, { [cleanPath[0]]: { _id: true } });
    }
    if (isNode) {
        return merge(acc, reduceRight(cleanPath, (acc2, k) => (isEmpty(acc2) ? { [k]: true } : { [k]: { ...acc2, _id: true } }), {}));
    }
    return acc;
});
export const parseFilterEnumFields = (source) => {
    let result = source;
    if (isPlainObject(source)) {
        result = mapValues(source, (x) => Array.isArray(x) ? x.map(parseFilterEnumFields) : parseFilterEnumFields(x));
        if (result._isEnum) {
            // TODO Should we throw an exception if an operator other than _eq is used or if no _eq is provided?
            result._eq = new EnumType(result._eq);
            delete result._isEnum;
        }
    }
    return result;
};
export const convertFilterDecoratorToGraphQLFilter = (screenDefinition, filter, recordContext) => {
    if (filter) {
        if (typeof filter === 'function') {
            const screenBase = getScreenElement(screenDefinition);
            return parseFilterEnumFields(filter.apply(screenBase, [recordContext]));
        }
        return parseFilterEnumFields(filter);
    }
    return undefined;
};
const buildNodeQueryFromProperties = (properties, startFrom = {}, isRequestingDefaults = false) => {
    let node = { ...startFrom };
    properties.forEach(prop => {
        if (typeof prop === 'string') {
            if (isRequestingDefaults && prop === '_sortValue') {
                return;
            }
            node[prop] = true;
        }
        if (typeof prop === 'object') {
            const merged = merge(prop, node);
            node = isRequestingDefaults ? sortValueCleaner(merged) : merged;
        }
    });
    return node;
};
const sortValueCleaner = (input) => {
    if (!input || typeof input !== 'object') {
        return input;
    }
    if (Array.isArray(input)) {
        return input
            .filter(e => e !== '_sortValue')
            .map((e) => {
            return sortValueCleaner(e);
        });
    }
    return objectKeys(input).reduce((r, k) => {
        if (k === '_sortValue') {
            return r;
        }
        r[k] = sortValueCleaner(input[k]);
        return r;
    }, {});
};
const buildCollectionNodeWithQuery = (options, wrapWithEdges = true, node, nodeTypes) => {
    if (!wrapWithEdges) {
        let singleNode = { _id: true };
        if (options.properties) {
            options.properties.forEach(prop => {
                if (typeof prop === 'string') {
                    singleNode[prop] = true;
                }
                if (typeof prop === 'object') {
                    singleNode = deepMerge(prop, singleNode);
                }
            });
        }
        return singleNode;
    }
    const collection = {
        query: {
            edges: {
                node: {
                    _id: true,
                },
                cursor: true,
            },
            // If we would have an empty object, the library puts empty parentheses and breaks the query
            __args: options.queryArguments && objectKeys(options.queryArguments).length > 0
                ? options.queryArguments
                : undefined,
        },
    };
    if (!options.noPageInfo) {
        collection.query.pageInfo = {
            startCursor: true,
            endCursor: true,
            hasPreviousPage: true,
            hasNextPage: true,
        };
    }
    if (options.properties) {
        collection.query.edges.node = buildNodeQueryFromProperties(options.properties, collection.query.edges.node);
    }
    if (options.queryArguments?.filter) {
        const parsedFilter = JSON.parse(options.queryArguments.filter);
        collection.query.edges.node = merge(collection.query.edges.node, addReferenceIdsAssign(parsedFilter, node, nodeTypes));
        collection.query.edges.node = addTotalCountTransform(parsedFilter, collection.query.edges.node);
    }
    return collection;
};
export const queryBuilder = (node, options) => {
    const query = buildCollectionNodeWithQuery(options);
    if (options.alias) {
        query.__aliasFor = camelCase(String(node).split('/').pop());
    }
    return query;
};
export const getUsedEnums = (pageMetadata, knownEnumDefinitions) => {
    const enums = deepFindPropertyValues('optionType', pageMetadata.uiComponentProperties).map(schemaTypeNameFromNodeName);
    return uniq(without(enums, ...knownEnumDefinitions)).filter(v => !!v);
};
export const buildEnumQuery = (enumName) => {
    const typeName = schemaTypeNameFromNodeName(enumName);
    return {
        [typeName]: {
            __aliasFor: '__type',
            __args: { name: typeName },
            enumValues: {
                name: true,
            },
        },
    };
};
export const getUsedNodes = (pageMetadata, knownTypeDefinitions, rootNode) => {
    const nodes = deepFindPropertyValues('node', pageMetadata.uiComponentProperties).filter(node => typeof node === 'string');
    if (rootNode) {
        nodes.push(rootNode);
    }
    if (knownTypeDefinitions) {
        return uniq(without(nodes, ...knownTypeDefinitions)).filter(v => !!v);
    }
    return nodes;
};
/**
 * @param withSortValue When true, the _sortValue property will be included in the returned properties
 */
export const buildTypeQuery = (nodeNames, knownTypes, depth = 3, withSortValue = false) => {
    const missingNodeNames = nodeNames.map(schemaTypeNameFromNodeName);
    const knownNodeNames = knownTypes.map(schemaTypeNameFromNodeName);
    return {
        getNodeDetailsList: {
            __args: { missingNodeNames, knownNodeNames, depth, withSortValue },
            ...getNodeDetailsFragment(),
        },
        getNodeCustomDetailsList: {
            __args: { missingNodeNames, knownNodeNames, depth },
            ...getNodeCustomDetailsFragment(),
        },
    };
};
export const getRequestableFieldIds = (fieldControlObjects, screenDefinition) => objectKeys(fieldControlObjects).filter(f => {
    return (!screenDefinition.metadata.uiComponentProperties?.[f]?.isTransient &&
        !screenDefinition.metadata.uiComponentProperties?.[f]?.isTransientInput);
});
export const buildEnumQueries = (enums) => enums.map(buildEnumQuery).reduce((optionsQuery, nextQuery) => ({ ...optionsQuery, ...nextQuery }), {});
export const buildTableTotalCountQuery = ({ filter }) => {
    return {
        query: {
            __args: { filter },
            totalCount: true,
        },
    };
};
export const buildFieldQuery = (rootNode, fieldControlObject, screenDefinition, fieldProperties, elementId, recordId, nodeTypes, userSettings) => {
    const queryProps = getFieldProperties(elementId, fieldControlObject.componentType, fieldProperties, screenDefinition, fieldProperties.bind, nodeTypes, rootNode, userSettings);
    const filter = JSON.stringify({ _id: { _eq: recordId } });
    const options = { queryArguments: { filter }, properties: [queryProps] };
    return queryBuilder(rootNode, options);
};
export const buildRootNodeQuery = (rootNode, fieldControlObjects, screenDefinition, _id, alias, nodeTypes, isDuplicate = false, skipHeaderFields) => {
    const fieldIds = getRequestableFieldIds(fieldControlObjects, screenDefinition);
    const screenId = screenDefinition.metadata.screenId;
    const uiComponentProps = screenDefinition.metadata.uiComponentProperties[screenId];
    let image;
    let title;
    if (uiComponentProps && uiComponentProps.navigationPanel) {
        if (uiComponentProps.navigationPanel.listItem.image) {
            image = getBindForQuery(uiComponentProps.navigationPanel.listItem.image);
        }
        title = getBindForQuery(uiComponentProps.navigationPanel.listItem.title);
    }
    const queryProps = getFieldsProperties(fieldIds, fieldControlObjects, screenDefinition, nodeTypes, rootNode);
    let extendedTitleProps = [];
    let extendedImageProps = [];
    if (title && !skipHeaderFields) {
        extendedTitleProps = title.includes('.')
            ? buildAliasObject(title, HEADER_TITLE)
            : [{ [HEADER_TITLE]: { __aliasFor: title } }];
    }
    if (image && !skipHeaderFields) {
        extendedImageProps = image.includes('.')
            ? buildAliasObject(image, HEADER_IMAGE)
            : [{ [HEADER_IMAGE]: { __aliasFor: image, value: true } }];
    }
    const extendedQueryProps = [...queryProps, ...extendedImageProps, ...extendedTitleProps];
    return wrapQuery(rootNode, {
        [isDuplicate ? 'getDuplicate' : 'read']: {
            __args: { _id: String(_id) },
            ...buildNodeQueryFromProperties([...extendedQueryProps, '_etag', '_id']),
        },
    }, alias);
};
const buildAliasObject = (value, key) => {
    const object = [...value.split('.')].reverse().reduce((prevFilter, currentKey, index) => {
        if (key === HEADER_IMAGE && index === 0) {
            return { [currentKey]: { value: true } };
        }
        return { [currentKey]: index === 0 ? true : prevFilter };
    }, {});
    object[value.split('.')[0]].__aliasFor = value.split('.')[0];
    const currentObjectKey = objectKeys(object)[0];
    object[key] = object[currentObjectKey];
    delete object[currentObjectKey];
    return [object];
};
const getBindForQuery = (field) => {
    return field.type === FieldKey.Reference
        ? `${convertDeepBindToPathNotNull(field.properties.bind)}.${getReferencePath(field.properties.valueField)}`
        : convertDeepBindToPathNotNull(field?.properties.bind);
};
const getFieldsProperties = (fieldIds, fieldControlObjects, screenDefinition, nodeTypes, node) => {
    return fieldIds.reduce((props, fieldId) => {
        const fieldType = fieldControlObjects[fieldId].componentType;
        const fieldProperties = screenDefinition.metadata.uiComponentProperties[fieldId];
        const userSettings = screenDefinition.userSettings[fieldId]?.$current;
        return [
            ...props,
            getFieldProperties(fieldId, fieldType, fieldProperties, screenDefinition, fieldProperties.bind, nodeTypes, node, userSettings),
        ];
    }, []);
};
export const getFilterAsString = (filters, screenDefinition, recordContext) => {
    const graphQLFilters = mergeFilters(filters, screenDefinition, recordContext);
    return !isEmpty(graphQLFilters) ? JSON.stringify(graphQLFilters) : undefined;
};
export const mergeFilters = (filters, screenDefinition, recordContext) => {
    const graphQLFilters = filters.reduce((currentGraphQLFilters, filter) => {
        const graphQLFilter = convertFilterDecoratorToGraphQLFilter(screenDefinition, filter, recordContext);
        if (graphQLFilter) {
            currentGraphQLFilters.push(graphQLFilter);
        }
        return currentGraphQLFilters;
    }, []);
    if (!isEmpty(graphQLFilters)) {
        return graphQLFilters.length === 1 ? graphQLFilters[0] : mergeGraphQLFilters(graphQLFilters);
    }
    return {};
};
export const applyDeepNestedKey = (elementId, fieldDetails, addAlias = false, bind, parentNode, nodeTypes) => {
    let pathComponents = convertDeepBindToPathNotNull(bind || elementId).split('.');
    let valueToSet = fieldDetails;
    let pathParent = schemaTypeNameFromNodeName(parentNode);
    for (let i = 0; i < pathComponents.length; i += 1) {
        const propertyType = findDeepPropertyDetails(pathParent, pathComponents[i], nodeTypes);
        if (propertyType?.type === GraphQLTypes.Json) {
            /**
             * We cannot deep query JSON columns, we need to fetch the JSON column, then
             * parse it and then finally return the property value we really need.
             *  */
            pathComponents = [...pathComponents].slice(0, i + 1);
            // In case of a JSON column, we need to fetch the whole column and then parse it
            valueToSet = true;
            break;
        }
        pathParent = propertyType?.type;
    }
    const object = [...pathComponents].reverse().reduce((prevFilter, currentKey) => {
        return { [currentKey]: prevFilter };
    }, valueToSet);
    if (!addAlias) {
        return object;
    }
    // On primitive types we just return an alias object
    if (typeof object[pathComponents[0]] !== 'object') {
        if (elementId === pathComponents[0]) {
            return object;
        }
        return {
            [elementId]: { __aliasFor: pathComponents[0] },
        };
    }
    // Swap the alias with the object key
    const currentObjectKey = objectKeys(object)[0];
    if (currentObjectKey !== elementId) {
        object[pathComponents[0]].__aliasFor = pathComponents[0];
        object[elementId] = object[currentObjectKey];
        delete object[currentObjectKey];
    }
    return object;
};
/**
 * Inspects the filter object, and prepare it for server side consumption.
 * It remaps id field names to match current server inconsistencies and creates nested sorting conditions for reference
 * fields.
 * @param orderBy
 */
export const createOrderByQueryArgument = (orderBy) => {
    const remappedOrderBy = {};
    objectKeys(orderBy).forEach(fieldKey => {
        const filterValue = orderBy[fieldKey];
        const keyToSet = fieldKey.split('__').join('.'); // Split by the nested bind separator
        set(remappedOrderBy, keyToSet, filterValue);
    });
    return JSON.stringify(remappedOrderBy);
};
export const getAggregationValues = (aggregations) => Object.entries(flat(aggregations)).reduce((acc, [key, value]) => {
    set(acc, `${key}.${value}`, true);
    return acc;
}, {});
export const nestedFieldsToQueryProperties = ({ screenDefinition, nestedFields, node, nodeTypes, }) => nestedFields
    ?.filter(nestedField => !nestedField.properties.isTransient && !nestedField.properties.isTransientInput)
    .map(nestedField => {
    return getFieldProperties(getTopLevelBindFromNestedBind(nestedField.properties.bind), nestedField.type, nestedField.properties, screenDefinition, nestedField.properties.bind, nodeTypes, node);
});
export const getNestedFieldsQuery = ({ screenDefinition, nestedFields, queryArguments, alias, wrapNodeWithEdges = true, node, nodeTypes, }) => {
    const properties = nestedFieldsToQueryProperties({ screenDefinition, nestedFields, node, nodeTypes });
    return buildCollectionNodeWithQuery({ alias, properties, queryArguments }, wrapNodeWithEdges, node, nodeTypes);
};
// This function avoids adding undefined properties to the GraphQL queries that would result in errors
export const parseQueryArguments = (inputArguments) => {
    const outputArguments = {};
    if (inputArguments.after) {
        outputArguments.after = inputArguments.after;
    }
    if (inputArguments.before) {
        outputArguments.before = inputArguments.before;
    }
    if (inputArguments.filter) {
        outputArguments.filter = inputArguments.filter;
    }
    if (inputArguments.first) {
        outputArguments.first = inputArguments.first;
    }
    if (inputArguments.last) {
        outputArguments.last = inputArguments.last;
    }
    if (inputArguments.orderBy) {
        outputArguments.orderBy = inputArguments.orderBy;
    }
    return outputArguments;
};
export const getDirtyNestedValues = ({ screenDefinition, elementId, recordId, level = 0, }) => {
    if (!recordId) {
        return {};
    }
    const value = screenDefinition.values[elementId];
    if (!value || value instanceof CollectionValue === false) {
        return {};
    }
    const actualLevel = level === 0 ? undefined : level;
    const row = screenDefinition.values[elementId].getRawRecord({
        id: recordId,
        cleanMetadata: false,
        level: actualLevel,
    });
    if (!row) {
        return {};
    }
    return Array.from(row.__dirtyColumns || []).reduce((acc, dirtyColumn) => {
        return set(acc, dirtyColumn, get(row, dirtyColumn));
    }, {});
};
export const buildRowDefaultQuery = ({ screenDefinition, elementId, isNewRow = true, alias, recordId, data, nodeTypes, }) => {
    if (!isNewRow && recordId === undefined) {
        throw new Error('A row id is needed in order to fetch default values for an existing row.');
    }
    const fieldProperties = screenDefinition.metadata.uiComponentProperties[elementId];
    const inputData = data || getDirtyNestedValues({ screenDefinition, elementId, recordId });
    const queryData = formatInputProperty({
        property: { type: schemaTypeNameFromNodeName(fieldProperties.node || ''), kind: GraphQLKind.Object },
        value: inputData,
        nodeTypes,
        isTopLevel: true,
    });
    let additionalData;
    if (!queryData) {
        additionalData = { [fieldProperties.bind ? String(fieldProperties.bind) : elementId]: [{}] };
    }
    else {
        additionalData = { [fieldProperties.bind ? String(fieldProperties.bind) : elementId]: [queryData] };
    }
    return buildDefaultsQuery({
        screenDefinition,
        alias,
        requestedFieldIds: [elementId],
        additionalData,
        sendNonDirtyFields: true,
    });
};
export const buildNestedDefaultsQuery = ({ screenDefinition, elementId, isNewRow = true, alias, recordId, level = 0, data, nodeTypes, }) => {
    if (!isNewRow && recordId === undefined) {
        throw new Error('A row id is needed in order to fetch default values for an existing row.');
    }
    const fieldProperties = screenDefinition.metadata.uiComponentProperties[elementId];
    const isNestedGrid = Object.prototype.hasOwnProperty.call(fieldProperties, 'levels');
    const inputData = data || getDirtyNestedValues({ screenDefinition, elementId, recordId, level });
    const columnDefinitions = isNestedGrid
        ? fieldProperties.levels[level].columns
        : getNestedFieldsFromProperties(fieldProperties);
    const node = String(isNestedGrid
        ? fieldProperties.levels[level].node
        : fieldProperties.node);
    let columns = [];
    if (!isUndefined(recordId) || !isNewRow || data) {
        const dirtyKeys = objectKeys(inputData).map(key => key.split('__')[0]);
        columns = columnDefinitions.filter(column => dirtyKeys.indexOf(getNestedFieldElementId(column).split('__')[0]) === -1);
    }
    else {
        columns = columnDefinitions;
    }
    columns = columns.filter(f => f.properties.bind !== '_sortValue');
    if (isEmpty(columns)) {
        return {};
    }
    const queryProps = [
        {
            ...getNestedFieldsQuery({
                screenDefinition,
                nestedFields: columns,
                wrapNodeWithEdges: false,
                node,
                nodeTypes,
            }),
        },
    ];
    const fieldsToRequest = buildNodeQueryFromProperties(queryProps);
    // We don't request the fake ID that the defaults service returns
    delete fieldsToRequest._id;
    const queryData = formatInputProperty({
        property: { type: schemaTypeNameFromNodeName(node), kind: GraphQLKind.Object },
        value: inputData,
        nodeTypes,
        isTopLevel: true,
    });
    return wrapQuery(node, {
        getDefaults: {
            ...((!isNewRow || data) && {
                __args: {
                    data: queryData,
                },
            }),
            ...fieldsToRequest,
        },
    }, alias);
};
/**
 * Builds a query to request default values from the server.
 * If a list of fields is not provided as the last argument, the function checks the screen definition and requests values for the non-dirty fields.
 */
export const buildDefaultsQuery = ({ screenDefinition, alias, requestedFieldIds, clean = false, additionalData, sendNonDirtyFields = false, node, nodeTypes, values, }) => {
    const screenId = screenDefinition.metadata.screenId;
    const controlObjects = filterFields(screenDefinition.metadata.controlObjects);
    const isNewPage = Boolean(screenDefinition.queryParameters?._id === NEW_PAGE);
    const isMainList = screenDefinition.selectedRecordId === null &&
        screenDefinition.navigationPanel &&
        screenDefinition.isMainPage === true &&
        !isNewPage;
    const hasFetchedDefaults = objectKeys(controlObjects).some(key => {
        const ctrlObjProps = controlObjects[key].properties;
        return ctrlObjProps?.fetchesDefaults || ctrlObjProps.columns?.some((c) => c.properties?.fetchesDefaults);
    });
    const hasQuickEntry = objectKeys(controlObjects).some(key => {
        return controlObjects[key].properties?.canAddNewLine;
    });
    if (isMainList || (!hasFetchedDefaults && !hasQuickEntry)) {
        return {};
    }
    const parsedScreenValues = parseScreenValues(screenId, false, true);
    // Only dirty fields that the user manually modified should be sent to the server
    let dataValues = {};
    if (!clean && !sendNonDirtyFields) {
        dataValues = objectKeys(parsedScreenValues).reduce((prevValue, fieldId) => {
            if ((screenDefinition.dirtyStates[fieldId] === true || has(screenDefinition.initialValues, fieldId)) &&
                (!requestedFieldIds || requestedFieldIds.indexOf(fieldId) === -1)) {
                const bind = screenDefinition.metadata.uiComponentProperties?.[fieldId]?.bind ||
                    fieldId;
                set(prevValue, convertDeepBindToPathNotNull(bind), parsedScreenValues[fieldId]);
            }
            return prevValue;
        }, {});
        if (values) {
            Object.entries(values).forEach(([key, value]) => {
                const bind = convertDeepBindToPathNotNull(key);
                set(dataValues, bind, value);
            });
        }
    }
    else if (sendNonDirtyFields) {
        // All fields should be sent to the server
        dataValues = parsedScreenValues;
    }
    const fieldIds = requestedFieldIds ||
        getRequestableFieldIds(controlObjects, screenDefinition).filter(fieldId => {
            // Dirty fields will not be requested as we don't want the user's manual changes to be overwritten by the default values.
            return !screenDefinition.dirtyStates[fieldId];
        });
    if (isEmpty(fieldIds)) {
        return {};
    }
    const queryProps = getFieldsProperties(fieldIds, controlObjects, screenDefinition, nodeTypes, node);
    const screenProperties = screenDefinition.metadata.uiComponentProperties[screenId];
    const fieldsToRequest = buildNodeQueryFromProperties(queryProps, {}, true);
    // We don't request the fake ID that the defaults service returns
    delete fieldsToRequest._id;
    if (isEmpty(fieldsToRequest)) {
        return {};
    }
    // If the fake $new ID is used in duplication mode, we remove it from the payload because that is only used in the client
    if (dataValues._id === NEW_PAGE) {
        delete dataValues._id;
    }
    else if (!dataValues._id && screenDefinition.selectedRecordId !== NEW_PAGE) {
        dataValues._id = screenDefinition.selectedRecordId;
    }
    // TODO Filter out dirty fields
    return wrapQuery(screenProperties.node || '', {
        getDefaults: {
            __args: {
                data: { ...dataValues, ...additionalData },
            },
            ...fieldsToRequest,
        },
    }, alias);
};
export const getNavigationPanelOrderByDefinition = (listItem, explicitOrderBy) => {
    if (explicitOrderBy) {
        return explicitOrderBy;
    }
    const sortableFields = [
        FieldKey.Text,
        FieldKey.Reference,
        FieldKey.Label,
        FieldKey.Numeric,
        FieldKey.Date,
        FieldKey.Progress,
    ];
    const titleProps = listItem.title
        .properties;
    const line2Props = listItem.line2
        ?.properties;
    const titleRightProps = listItem.titleRight
        ?.properties;
    const valueField = titleProps &&
        !titleProps.isTransient &&
        titleProps._controlObjectType &&
        sortableFields.includes(titleProps._controlObjectType) &&
        getReferenceValueFieldPath(titleProps);
    const orderBy = valueField ? set({}, valueField, 1) : {};
    if (line2Props?.bind &&
        !line2Props?.isTransient &&
        line2Props._controlObjectType &&
        sortableFields.includes(line2Props._controlObjectType)) {
        const line2ValueField = getReferenceValueFieldPath(line2Props);
        if (line2ValueField) {
            set(orderBy, line2ValueField, 1);
        }
    }
    else if (titleRightProps?.bind &&
        !titleRightProps.isTransient &&
        titleRightProps._controlObjectType &&
        sortableFields.includes(titleRightProps._controlObjectType)) {
        const titleLineRightValueField = getReferenceValueFieldPath(titleRightProps);
        if (titleLineRightValueField) {
            set(orderBy, titleLineRightValueField, 1);
        }
    }
    return orderBy;
};
const getDateForFiltering = (value, locale) => {
    let dateValue = null;
    if (isValidIsoDate(value, locale)) {
        dateValue = parseIsoDate(value, locale);
    }
    if (isValidLocalizedDate(value, locale, 'FullDate')) {
        dateValue = parseLocalizedDate(value, locale, 'FullDate');
    }
    if (!dateValue) {
        return null;
    }
    return dateValue.format('YYYY-MM-DD', locale);
};
export const buildSearchBoxFilterForNestedFields = (nestedFields, nodeTypes, filterText, locale, node) => {
    const filterStatements = [];
    const trimmedFilterText = escapeStringRegexp(filterText.trim());
    // If we dont have node it means the page or the field is transient so we dont check the nodeTypes to filter
    if (!node) {
        withoutNestedTechnical(nestedFields)
            .filter(itemDefinition => itemDefinition && itemDefinition.properties.canFilter !== false)
            .forEach(itemDefinition => {
            const bind = convertDeepBindToPathNotNull(itemDefinition.properties.bind);
            const regexTextFilter = { _regex: trimmedFilterText, _options: 'i' };
            filterStatements.push(applyDeepNestedKey(bind, regexTextFilter, false, undefined, node ? String(node) : undefined, nodeTypes));
        });
        return filterStatements;
    }
    withoutNestedTechnical(nestedFields)
        .filter(itemDefinition => {
        const bind = convertDeepBindToPathNotNull(itemDefinition.properties.bind);
        const graphqlPropertyType = findDeepPropertyType(schemaTypeNameFromNodeName(node), bind, nodeTypes);
        return itemDefinition && itemDefinition.properties.canFilter !== false && graphqlPropertyType?.canFilter;
    })
        .forEach(itemDefinition => {
        const bind = convertDeepBindToPathNotNull(itemDefinition.properties.bind);
        const regexTextFilter = { _regex: trimmedFilterText, _options: 'i' };
        const graphqlPropertyType = findDeepPropertyType(schemaTypeNameFromNodeName(node), bind, nodeTypes);
        // TODO: Handle enums, numbers and additional types
        if (graphqlPropertyType?.type === GraphQLTypes.String ||
            (graphqlPropertyType?.type === GraphQLTypes.IntOrString && graphqlPropertyType.name === '_id')) {
            // If the type is string, we apply a partial regex match
            filterStatements.push(set({}, bind, regexTextFilter));
        }
        else if (graphqlPropertyType?.type === GraphQLTypes.Date &&
            getDateForFiltering(filterText.trim(), locale)) {
            filterStatements.push(set({}, bind, { _eq: getDateForFiltering(filterText.trim(), locale) }));
        }
        else if (itemDefinition.type === FieldKey.Reference) {
            // If field type is reference, we do the same type check for the value and helper text fields.
            const referenceProperties = itemDefinition.properties;
            const referredNode = String(referenceProperties.node);
            const valueFieldKey = convertDeepBindToPath(referenceProperties.valueField);
            const isNestedReference = valueFieldKey && valueFieldKey.indexOf('.') !== -1;
            const valueFieldType = findDeepPropertyType(schemaTypeNameFromNodeName(referredNode), referenceProperties.valueField, nodeTypes);
            // Here there is a nested reference, so we probably don't know the exact type, so let's just hope for the best!
            if (valueFieldType?.type === GraphQLTypes.String || isNestedReference) {
                if (isNestedReference && isDevMode()) {
                    xtremConsole.warn(`Type could not be determined for '${valueFieldKey}' nested reference field, assuming String type.`);
                }
                const filter = set({}, `${convertDeepBindToPathNotNull(itemDefinition.properties.bind)}.${valueFieldKey}`, regexTextFilter);
                filterStatements.push(filter);
            }
            const helperTextFieldKey = convertDeepBindToPath(referenceProperties.helperTextField);
            const helperTextFieldType = findDeepPropertyType(schemaTypeNameFromNodeName(referredNode), referenceProperties.helperTextField, nodeTypes);
            if (helperTextFieldKey && helperTextFieldType?.type === GraphQLTypes.String) {
                const filter = set({}, `${convertDeepBindToPathNotNull(itemDefinition.properties.bind)}.${helperTextFieldKey}`, regexTextFilter);
                filterStatements.push(filter);
            }
        }
    });
    return filterStatements;
};
export const buildSearchBoxFilter = (fieldProperties, nodeTypes, locale, collectionFieldType, searchBoxText, exactMatch = false, isArray = false) => {
    if (!searchBoxText)
        return undefined;
    const nodeType = schemaTypeNameFromNodeName(String(fieldProperties.node));
    const isSearchQueryANumber = Number.isFinite(Number(searchBoxText));
    const orFilters = [];
    if (collectionFieldType === CollectionFieldTypes.LOOKUP_DIALOG) {
        const referenceProperties = fieldProperties;
        if (!Object.prototype.hasOwnProperty.call(referenceProperties, 'valueField') &&
            !Object.prototype.hasOwnProperty.call(referenceProperties, 'helperTextField')) {
            throw new Error(`Neither 'valueField' nor 'helperTextField' found: ${JSON.stringify(referenceProperties)}`);
        }
        const condition = {
            _regex: escapeStringRegexp(searchBoxText),
            ...(!exactMatch && { _options: 'i' }),
        };
        if (referenceProperties.helperTextField) {
            const helperTextFieldType = findDeepPropertyType(nodeType, referenceProperties.helperTextField, nodeTypes, true);
            if (helperTextFieldType?.canFilter &&
                (helperTextFieldType?.type === GraphQLTypes.String ||
                    ((helperTextFieldType?.type === GraphQLTypes.IntOrString ||
                        helperTextFieldType?.type === GraphQLTypes.Int) &&
                        isSearchQueryANumber))) {
                const helperTextFieldPath = convertDeepBindToPathNotNull(referenceProperties.helperTextField);
                const filter = set(isArray ? set({}, [...helperTextFieldPath.split('.').slice(0, -1), '_atLeast'], 1) : {}, helperTextFieldPath, condition);
                orFilters.push(filter);
            }
        }
        if (referenceProperties.valueField) {
            const valueFieldType = findDeepPropertyType(nodeType, referenceProperties.valueField, nodeTypes, true);
            if (valueFieldType?.canFilter &&
                (valueFieldType?.type === GraphQLTypes.String ||
                    ((valueFieldType?.type === GraphQLTypes.IntOrString || valueFieldType?.type === GraphQLTypes.Int) &&
                        isSearchQueryANumber))) {
                const valueFieldPath = convertDeepBindToPathNotNull(referenceProperties.valueField);
                const filter = set(isArray ? set({}, [...valueFieldPath.split('.').slice(0, -1), '_atLeast'], 1) : {}, valueFieldPath, condition);
                orFilters.push(filter);
            }
        }
        if (referenceProperties.columns && referenceProperties.shouldSuggestionsIncludeColumns) {
            orFilters.push(...buildSearchBoxFilterForNestedFields(referenceProperties.columns, nodeTypes, searchBoxText, locale, referenceProperties.node));
        }
    }
    else {
        const tableProperties = fieldProperties;
        orFilters.push(...buildSearchBoxFilterForNestedFields(tableProperties.columns || [], nodeTypes, searchBoxText, locale, tableProperties.node));
    }
    return isEmpty(orFilters) ? undefined : { _or: orFilters };
};
//# sourceMappingURL=index.js.map