import { ClientError, Graph, GraphMutation } from '@sage/xtrem-client';
import { InteropError, objectKeys } from '@sage/xtrem-shared';
import { get, merge, uniq } from 'lodash';
import { ContainerKey, FieldKey } from '../component/types';
import * as xtremRedux from '../redux';
import { refreshNavigationPanel, removePageServerErrors, setIdToQueryParameters } from '../redux/actions';
import { filterFields } from '../utils/abstract-fields-utils';
import { xtremConsole } from '../utils/console';
import { getNonLazySectionsFromScreenDefinition } from '../utils/page-utils';
import { getNavigationPanelDefinitionFromState } from '../utils/state-utils';
import { getFieldProperties } from './graphql-query-builder';
import { executeGraphqlQuery, removeEdges } from './graphql-utils';
import { localize } from './i18n-service';
import { NodeCacheService } from './node-cache-service';
import { formatScreenValues } from './value-formatter-service';
export const nonSavableFieldType = [FieldKey.Aggregate, FieldKey.Count, FieldKey.Separator];
const fetcher = (query, isMutation, selector, endpoint = '/api') => {
    const result = executeGraphqlQuery({ query, endpoint }).catch(errorResponse => errorResponse?.response?.data || fetcher);
    if (isMutation && selector.mutation) {
        NodeCacheService.flushCacheAfterMutation(selector.mutation);
    }
    return result;
};
const getApiError = (data) => {
    const message = data.errors.map(e => e.message).join('\n');
    const diagnoses = data.extensions?.diagnoses || [];
    data.errors.forEach(e => {
        diagnoses.push(...(e.extensions?.diagnoses || []));
    });
    return new InteropError(message, diagnoses);
};
export class ReadOnlyGraphQLApi {
    constructor() {
        this.graph = new Graph({
            fetcher,
        });
    }
    node(nodeName) {
        return this.graph.node(nodeName);
    }
    raw(query, isMetaDataQuery = false) {
        if (query.includes('mutation')) {
            xtremConsole.warn('Mutations on this context are not supported. Use at your own risk.');
        }
        return fetcher(query, false, {}, isMetaDataQuery ? '/metadata' : '/api').then(data => {
            if (data) {
                if (data.errors) {
                    throw getApiError(data);
                }
                return { ...data.data, diagnoses: data.extensions?.diagnoses };
            }
            throw new Error(localize('@sage/xtrem-ui/invalid-response-no-data', 'Invalid response, no data provided'));
        });
    }
}
export class GraphQLApi {
    constructor(screenBase) {
        this.screenBase = screenBase;
        this.graph = new Graph({
            fetcher,
        });
    }
    getQueryParameters() {
        return this.screenBase.$.queryParameters;
    }
    node(nodeName) {
        return this.graph.node(nodeName);
    }
    /**
     * Executes a raw GraphQL query string against the server endpoint.
     * @param query the query string
     * @param skipErrorProcessing if this property is set to true, the framework will not try to intercept the error and
     * @returns
     */
    async raw(query, skipErrorProcessing = false, isMetaDataQuery = false) {
        const data = await fetcher(query, false, {}, isMetaDataQuery ? '/metadata' : '/api');
        if (data) {
            if (data.errors) {
                if (!skipErrorProcessing) {
                    await this.screenBase.$.processServerErrors(new ClientError(data));
                }
                throw getApiError(data);
            }
            else {
                return { ...data.data, diagnoses: data.extensions?.diagnoses };
            }
        }
        else {
            throw new Error(localize('@sage/xtrem-ui/invalid-response-no-data', 'Invalid response, no data provided'));
        }
    }
    nodeName(node) {
        const pageProperties = this.screenBase._pageMetadata.uiComponentProperties[this.screenBase._pageMetadata.screenId];
        const nodeName = node || pageProperties.node;
        if (!nodeName) {
            throw new Error(localize('@sage/xtrem-ui/no-node', 'No node was provided neither {{0}} specifies one', [
                this.screenBase._pageMetadata.screenId,
            ]));
        }
        return nodeName;
    }
    async applyUpdateToValue(nodeName, response, args, recordAdded = false) {
        if (!args || !args.nodeName) {
            const store = xtremRedux.getStore();
            const state = store.getState();
            const values = removeEdges(response, true, true);
            const screenId = this.screenBase._pageMetadata.screenId;
            const controlObjects = this.getLoadedFieldControlObjects(state.screenDefinitions[screenId]);
            this.screenBase.$.values = formatScreenValues({
                screenId,
                controlObjects,
                plugins: state.plugins,
                nodeTypes: state.nodeTypes,
                values,
                parentNode: nodeName.toString(),
                userSettings: state.screenDefinitions[screenId].userSettings,
            });
            if (values._id) {
                const dispatch = store.dispatch;
                dispatch(setIdToQueryParameters(screenId, values._id));
                if (getNavigationPanelDefinitionFromState(this.screenBase._pageMetadata.screenId, state)) {
                    // This promise is intentionally not awaited so we do not block the UI while the navigation panel is refreshed
                    dispatch(refreshNavigationPanel(this.screenBase._pageMetadata.screenId, recordAdded));
                }
            }
        }
    }
    getLoadedFieldControlObjects(screenDefinition) {
        const nonLazySections = getNonLazySectionsFromScreenDefinition(screenDefinition);
        const loadedSection = objectKeys(this.screenBase._pageMetadata.controlObjects).filter(key => this.screenBase._pageMetadata.controlObjects[key].componentType === ContainerKey.Section &&
            screenDefinition.metadata.uiComponentProperties[key]?.isLoaded);
        const sectionsToLoad = uniq([...nonLazySections, ...loadedSection]);
        return filterFields(this.screenBase._pageMetadata.controlObjects, sectionsToLoad, screenDefinition, true);
    }
    getNonTransientFieldsQuery(args, forQuery = false) {
        const state = xtremRedux.getStore().getState();
        const screenDefinition = state.screenDefinitions[this.screenBase._pageMetadata.screenId];
        const uiComponentProperties = this.screenBase._pageMetadata.uiComponentProperties;
        const controlObjects = this.getLoadedFieldControlObjects(screenDefinition);
        return (objectKeys(controlObjects)
            // Transient fields are not sent to the server in any case
            .filter(key => !get(uiComponentProperties, `${key}.isTransient`) &&
            !get(uiComponentProperties, `${key}.isTransientInput`))
            // Some fields are not saved as they are always the result of a server side calculation.
            .filter(key => forQuery || nonSavableFieldType.indexOf(controlObjects[key].componentType) === -1)
            // Fields that the functional developer not requested are filtered out
            .filter(key => (args && args.fieldsToReturn ? args.fieldsToReturn.indexOf(key) !== -1 : true))
            .map(key => {
            const fieldProperties = getFieldProperties(key, controlObjects[key].componentType, get(uiComponentProperties, key), screenDefinition, get(uiComponentProperties, `${key}.bind`), state.nodeTypes, this.nodeName(args?.nodeName), screenDefinition.userSettings?.[key]?.$current);
            return typeof fieldProperties === 'string'
                ? {
                    [fieldProperties]: true,
                }
                : fieldProperties;
        })
            .reduce((reduced, next) => merge(reduced, next), {}));
    }
    async create(args) {
        // Allow any event handlers to settle and finish updating the value
        await new Promise(resolve => setTimeout(resolve, 20));
        const nodeName = this.nodeName(args && args.nodeName);
        const values = args?.values ?? this.screenBase.getSerializedValues();
        // ID is not sent whenever a new record is being created
        if (values._id) {
            delete values._id;
        }
        const valuesKeys = this.getNonTransientFieldsQuery(args, true);
        try {
            const node = this.graph.node(nodeName);
            const response = await node.create({ ...valuesKeys, _id: true }, { data: values }).execute();
            await this.applyUpdateToValue(nodeName.toString(), response, args, true);
            const dispatch = xtremRedux.getStore().dispatch;
            dispatch(removePageServerErrors(this.screenBase._pageMetadata.screenId));
        }
        catch (error) {
            if (error.errors) {
                await this.screenBase.$.processServerErrors(error);
            }
            else {
                xtremRedux.getStore().getState()?.applicationContext?.onTelemetryEvent?.('error', error.message);
                throw new Error(localize('@sage/xtrem-ui/create-error', 'Create error: {{0}}', [error]));
            }
        }
    }
    async update(args, skipUpdate = false) {
        // Allow any event handlers to settle and finish updating the value
        await new Promise(resolve => setTimeout(resolve, 20));
        const nodeName = this.nodeName(args && args.nodeName);
        const values = args?.values ?? this.screenBase.getSerializedValues();
        const valuesKeys = this.getNonTransientFieldsQuery(args, true);
        const recordId = this.getQueryParameters()._id;
        const argsId = args?._id || args?.values?._id;
        const additionalValues = { _id: argsId || String(recordId) };
        try {
            const node = this.graph.node(nodeName);
            const response = await node
                .update({ ...valuesKeys, _id: true, _etag: true }, { data: { ...additionalValues, ...values } })
                .execute();
            if (((recordId && !argsId) || argsId === recordId) && !skipUpdate) {
                await this.applyUpdateToValue(nodeName.toString(), response, args);
                const dispatch = xtremRedux.getStore().dispatch;
                dispatch(removePageServerErrors(this.screenBase._pageMetadata.screenId));
            }
        }
        catch (error) {
            if (error.errors) {
                await this.screenBase.$.processServerErrors(error);
            }
            else {
                throw new Error(localize('@sage/xtrem-ui/update-error', 'Update error: {{0}}', [error]));
            }
        }
    }
    async delete(args) {
        const nodeName = this.nodeName(args && args.nodeName);
        const recordId = args?._id || this.getQueryParameters()._id;
        if (!recordId) {
            throw new Error(localize('@sage/xtrem-ui/no-record-id-provided', 'No record id was provided through query parameters'));
        }
        try {
            const node = this.graph.node(nodeName);
            await node.deleteById(String(recordId)).execute();
            const dispatch = xtremRedux.getStore().dispatch;
            if (getNavigationPanelDefinitionFromState(this.screenBase._pageMetadata.screenId) &&
                (!args?._id || args?._id === this.getQueryParameters()._id)) {
                // This promise is intentionally not awaited so we do not block the UI while the navigation panel is refreshed
                dispatch(refreshNavigationPanel(this.screenBase._pageMetadata.screenId));
            }
        }
        catch (error) {
            if (error.errors) {
                await this.screenBase.$.processServerErrors(error);
            }
            else {
                throw new Error(localize('@sage/xtrem-ui/delete-error', 'Delete error: {{0}}', [error]));
            }
        }
    }
    async duplicate(args) {
        const nodeName = this.nodeName(args && args.nodeName);
        const recordId = args?._id || this.getQueryParameters()._id;
        if (!recordId) {
            throw new Error(localize('@sage/xtrem-ui/no-record-id-provided', 'No record id was provided through query parameters'));
        }
        const values = args?.values;
        const duplicateArgs = { _id: String(recordId) };
        if (values) {
            duplicateArgs.data = values;
        }
        try {
            const node = this.graph.node(nodeName);
            const response = await node.duplicate({ _id: true }, duplicateArgs).execute();
            const dispatch = xtremRedux.getStore().dispatch;
            dispatch(removePageServerErrors(this.screenBase._pageMetadata.screenId));
            return String(response._id);
        }
        catch (error) {
            if (error.errors) {
                await this.screenBase.$.processServerErrors(error);
                return null;
            }
            throw new Error(localize('@sage/xtrem-ui/duplicate-error', 'Duplication error: {{0}}', [error]));
        }
    }
}
export class GraphQLMutationApi {
    constructor() {
        this.graph = new GraphMutation({
            fetcher,
        });
    }
    node(nodeName) {
        return this.graph.node(nodeName);
    }
}
//# sourceMappingURL=graphql-api.js.map