import { isObject } from 'lodash';
import { xtremConsole } from '../utils/console';
import { isDevMode } from '../utils/window';
/**
 * Derives a key from the given password and salt using PBKDF2.
 * @param password The password to derive the key from.
 * @param salt The salt to use in the key derivation.
 * @returns A promise that resolves to the derived key.
 */
async function deriveKey(password, salt) {
    const encoder = new TextEncoder();
    const keyMaterial = await window.crypto.subtle.importKey('raw', encoder.encode(password), 'PBKDF2', false, [
        'deriveKey',
    ]);
    return window.crypto.subtle.deriveKey({
        name: 'PBKDF2',
        salt,
        iterations: 100000,
        hash: 'SHA-256',
    }, keyMaterial, { name: 'AES-GCM', length: 256 }, false, ['encrypt', 'decrypt']);
}
export async function encryptJsonDocument(content, encryptionKey) {
    try {
        const salt = window.crypto.getRandomValues(new Uint8Array(16));
        const iv = window.crypto.getRandomValues(new Uint8Array(12));
        const text = JSON.stringify(content);
        const key = await deriveKey(encryptionKey, salt.buffer);
        const encoded = new TextEncoder().encode(text);
        const encrypted = await window.crypto.subtle.encrypt({ name: 'AES-GCM', iv }, key, encoded);
        // Store salt and iv with ciphertext (salt|iv|ciphertext)
        // This is safe to have them in the encrypted data, it does not compromise the security
        const combined = new Uint8Array(salt.length + iv.length + encrypted.byteLength);
        combined.set(salt, 0);
        combined.set(iv, salt.length);
        combined.set(new Uint8Array(encrypted), salt.length + iv.length);
        return combined;
    }
    catch (e) {
        if (isDevMode()) {
            xtremConsole.error('Failed to encrypt cache entry.');
            xtremConsole.error(e);
        }
        return null;
    }
}
export async function decryptJsonDocument(content, encryptionKey) {
    try {
        let combined;
        if (content instanceof Uint8Array) {
            combined = content;
        }
        else if (isObject(content)) {
            // BL: Probably the unit test IndexDB mock does not return Uint8Array, so we handle it here.
            const contentArray = Object.keys(content).reduce((acc, key) => {
                acc.push(content[key]);
                return acc;
            }, []);
            combined = new Uint8Array(contentArray);
        }
        else {
            xtremConsole.error('Invalid content type for decryption. Expected Uint8Array or object.');
            return null;
        }
        // extract salt, iv, and ciphertext
        const salt = combined.slice(0, 16);
        const iv = combined.slice(16, 28);
        const cipherText = combined.slice(28);
        const key = await deriveKey(encryptionKey, salt.buffer);
        const decrypted = await window.crypto.subtle.decrypt({
            name: 'AES-GCM',
            iv,
        }, key, cipherText);
        const decoder = new TextDecoder();
        const decodedValue = decoder.decode(decrypted);
        return JSON.parse(decodedValue);
    }
    catch (e) {
        if (isDevMode()) {
            xtremConsole.error('Failed to decrypt cache entry.');
            xtremConsole.error(e);
        }
        return null;
    }
}
export async function sha256Hash(text) {
    const msgUint8 = new TextEncoder().encode(text);
    const hashBuffer = await window.crypto.subtle.digest('SHA-256', msgUint8);
    const hashArray = Array.from(new Uint8Array(hashBuffer));
    const hashHex = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
    return hashHex;
}
//# sourceMappingURL=encryption-service.js.map