import { Decimal } from '@sage/xtrem-decimal';
import { deepMerge, flat, objectKeys } from '@sage/xtrem-shared';
import { get, intersection, isArray, isEmpty, isEqual, isNil, isPlainObject, omitBy, partition, set, setWith, sortBy, uniq, } from 'lodash';
import { navigationPanelId } from '../component/container/navigation-panel/navigation-panel-types';
import { FieldKey } from '../component/types';
import { getStore } from '../redux';
import { GraphQLTypes } from '../types';
import { xtremConsole } from '../utils/console';
import { DEEP_BIND_QUERY_ALIAS_GLUE, ELEMENT_ID_APPLICATION_CODE_LOOKUP, SERVER_VALIDATION_RULE_PREFIX, } from '../utils/constants';
import { getIsoDate } from '../utils/formatters';
import { transformToLokiJsFilter } from '../utils/lokijs-filter-transformer';
import { convertDeepBindToPath, getNestedFieldsFromProperties } from '../utils/nested-field-utils';
import { findDeepPropertyDetails } from '../utils/node-utils';
import { resolveByValue } from '../utils/resolve-value-utils';
import { findNestedFieldProperties } from '../utils/server-error-transformer';
import { getPageDefinitionFromState, getPagePropertiesFromPageDefinition } from '../utils/state-utils';
import { cleanMetadataFromRecord, schemaTypeNameFromNodeName, splitValueToMergedValue } from '../utils/transformers';
import { isDevMode } from '../utils/window';
import { CollectionFieldTypes, RecordActionType } from './collection-data-types';
import { CollectionDataRow, getGroupFilterValue, getGroupKey, transformFilterForCollectionValue, } from './collection-data-utils';
import { dispatchCloseSideBar, dispatchFieldValidation, dispatchSetFieldDirty, dispatchUpdateNestedFieldValidationErrorsForRecord, } from './dispatch-service';
import { getGraphQLFilter, getTypedNestedFields, mergeGraphQLFilters } from './filter-service';
import { buildSearchBoxFilter, convertFilterDecoratorToGraphQLFilter } from './graphql-query-builder';
import { fetchCollectionData, fetchCollectionRecord, fetchNestedDefaultValues, fetchReferenceFieldData, } from './graphql-service';
import { removeEdgesDeep } from './graphql-utils';
import { LokiDb } from './loki';
import { checkNestedRecordValidationStatus } from './validation-service';
import { formatCollectionItem } from './value-formatter-service';
export class CollectionValue {
    constructor({ activeOptionsMenuItem, bind, columnDefinitions, contextNode, dbKey, elementId, fieldType, filter = [], hasNextPage, initialValues = [], isNoServerLookups, isTransient, levelMap, locale = getStore().getState().applicationContext?.locale || 'en-US', mapServerRecordFunctions, nodes, nodeTypes = getStore().getState().nodeTypes, orderBy = [], parentElementId, recordContext, screenId, referenceLookupContextLevel, sublevelProperty, }) {
        this.initialFilter = [];
        this.filter = [];
        /**
         * Stores a filter snapshot whenever a search text is combined with
         * active filters (i.e. nav panel in split view)
         * so that it can be later restored by calling "restoreFilterSnapshot"
         */
        this.filterSnapshot = [];
        this.forceRefetch = false;
        this.lastFetchedPage = [];
        this.orderBy = [];
        this.initialOrderBy = [];
        this.valueChangeSubscribers = [];
        this.valueChangeUncommittedSubscribers = [];
        this.validityChangeSubscribers = [];
        this.validityChangeUncommittedSubscribers = [];
        this.initialValues = [];
        this.forbiddenKeys = [
            '__aggFunc',
            '__level',
            '__dirtyColumns',
            '__forceRowUpdate',
            '__validationState',
            '__compositeKey',
            '__unloaded',
            '__action',
            '__isGroup',
            '__phantom',
            '__hasChildRecords',
            '$loki',
        ];
        this.formatRecordValue = (collectionItem, previousRecord, shouldCallRemapServer = false, level = 0) => {
            const nodeIndex = collectionItem.__level ?? level;
            const formattedValue = formatCollectionItem({
                screenId: this.screenId,
                nodeTypes: this.nodeTypes,
                collectionItem,
                columnsDefinitions: this.getColumnDefinitions(nodeIndex),
                parentNode: this.getNode(nodeIndex),
                contextRow: previousRecord,
            });
            if (shouldCallRemapServer && this.mapServerRecordFunctions && this.mapServerRecordFunctions[nodeIndex]) {
                const mapFunction = this.mapServerRecordFunctions[nodeIndex];
                const clearRecord = cleanMetadataFromRecord(formattedValue);
                const remappedRecord = resolveByValue({
                    propertyValue: mapFunction,
                    screenId: this.screenId,
                    rowValue: null,
                    fieldValue: clearRecord,
                    skipHexFormat: true,
                });
                const modifiedMetadataProperties = intersection(objectKeys(remappedRecord), this.forbiddenKeys);
                if (modifiedMetadataProperties.length > 0) {
                    throw new Error(`The remap function is not allowed to modify metadata properties. Modified properties: ${modifiedMetadataProperties.join(', ')}`);
                }
                if (remappedRecord._id !== formattedValue._id) {
                    throw new Error('The remap function is not allowed to modify the _id property.');
                }
                return { ...formattedValue, ...remappedRecord };
            }
            if (this.sublevelProperty && this.fieldType === CollectionFieldTypes.TREE) {
                const childRecordCount = get(collectionItem, `${this.sublevelProperty}.query.totalCount`, 0);
                formattedValue.__isGroup = childRecordCount > 0;
            }
            return formattedValue;
        };
        /** Compacts query results with __ aliases into a single object for better rendering and data management purposes */
        this.repackRecordData = (rv) => {
            // We don't change primitive values
            const recordValue = rv;
            if (!recordValue ||
                typeof recordValue === 'boolean' ||
                typeof recordValue === 'string' ||
                typeof recordValue === 'number') {
                return recordValue;
            }
            return (objectKeys(recordValue)
                // Keys must be sorted so first so if there is a mix of normal keys and keys with `__` in them, we process the normal keys first.
                .sort()
                .reduce((previousValue, key) => {
                // If the key includes a forbidden key we skip it
                if (this.forbiddenKeys.includes(key)) {
                    return { ...previousValue, [key]: recordValue[key] };
                }
                const parts = key.split(DEEP_BIND_QUERY_ALIAS_GLUE);
                if (parts.length > 1 && parts[0] !== '' && parts[1] !== '') {
                    const keyToUse = parts[0];
                    const isArrayValue = isArray(recordValue[key]);
                    const defaultValue = isArrayValue ? [] : {};
                    const currentValue = previousValue[keyToUse] || defaultValue;
                    let repackedValue = {};
                    if (isArrayValue) {
                        repackedValue = deepMerge(currentValue, recordValue[key], isArrayValue).map(this.repackRecordData);
                        return { ...previousValue, [keyToUse]: repackedValue };
                    }
                    if (isPlainObject(recordValue[key])) {
                        repackedValue = this.repackRecordData(recordValue[key]);
                    }
                    else {
                        repackedValue[parts[1]] = recordValue[key];
                    }
                    previousValue[keyToUse] =
                        repackedValue instanceof Object && objectKeys(repackedValue).length === 0
                            ? null
                            : { ...repackedValue, ...previousValue[keyToUse] };
                }
                else if (recordValue[key] instanceof Decimal ||
                    recordValue[key]?.constructor?.name === 'Decimal' ||
                    !!recordValue[key]?.constructor?.Decimal) {
                    previousValue[key] = recordValue[key].toNumber();
                }
                else if (recordValue[key] instanceof Date) {
                    previousValue[key] = getIsoDate(recordValue[key]);
                }
                else if (isArray(recordValue[key])) {
                    previousValue[key] = recordValue[key].map(this.repackRecordData);
                }
                else if (isPlainObject(recordValue[key])) {
                    previousValue[key] = this.repackRecordData(recordValue[key]);
                }
                else {
                    previousValue[key] = recordValue[key];
                }
                return previousValue;
            }, {}));
        };
        this.validateField = async ({ skipDispatch = false, withoutServerErrors = false, } = {}) => {
            const isValid = this.db.isCollectionValid();
            if (isValid) {
                if (!skipDispatch) {
                    dispatchFieldValidation(this.screenId, this.elementId);
                }
                return [];
            }
            const invalidLoaded = this.getInvalidLoadedRecords({ level: null, parentId: null, withoutServerErrors });
            const invalidUnloaded = this.getInvalidUnloadedRecords({ level: null, parentId: null, withoutServerErrors });
            const allInvalid = [...invalidLoaded, ...invalidUnloaded];
            const expectedLength = Math.min(10, allInvalid.length);
            const fieldProperties = getStore().getState().screenDefinitions[this.screenId].metadata.uiComponentProperties[this.elementId];
            let errors;
            let validationResult;
            if (invalidLoaded.length >= expectedLength) {
                errors = this.getErrorMessages({
                    records: invalidLoaded,
                    fieldProperties,
                });
                validationResult = this.getErrorList(errors);
                if (!skipDispatch) {
                    dispatchFieldValidation(this.screenId, this.elementId, validationResult);
                }
                return validationResult;
            }
            if (fieldProperties.mainField && fieldProperties.mainField !== '_id') {
                const idsToFetch = invalidUnloaded.slice(0, expectedLength - invalidLoaded.length).map(r => r._id);
                const pageDefinition = getPageDefinitionFromState(this.screenId);
                const pageProperties = getPagePropertiesFromPageDefinition(pageDefinition);
                const idFilter = { _id: { _in: idsToFetch } };
                const result = await fetchCollectionData({
                    screenDefinition: pageDefinition,
                    rootNode: pageProperties.node,
                    rootNodeId: pageDefinition.selectedRecordId || '',
                    elementId: this.elementId,
                    nestedFields: this.getColumnDefinitions(0),
                    queryArguments: { filter: JSON.stringify(idFilter) },
                    bind: this.bind,
                });
                this.addQueryResultToCollection({
                    result,
                    level: 0,
                    markAsAdded: false,
                });
            }
            const invalidRecords = this.getAllInvalidRecords({ level: null });
            errors = this.getErrorMessages({
                records: invalidRecords,
                fieldProperties,
            });
            validationResult = this.getErrorList(errors);
            if (!skipDispatch) {
                dispatchFieldValidation(this.screenId, this.elementId, validationResult);
            }
            return validationResult;
        };
        this.getErrorList = (errors) => {
            return errors.map(({ mainFieldName, mainFieldValue, columnId, columnName, message, recordId, validationRule, level }) => {
                let messagePrefix = mainFieldName ? `${mainFieldName}: ${mainFieldValue} -` : `Id: ${recordId}`;
                if (columnId) {
                    messagePrefix += ` ${columnName}`;
                }
                return {
                    screenId: this.screenId,
                    elementId: this.elementId,
                    columnId,
                    recordId,
                    message,
                    messagePrefix,
                    validationRule,
                    level,
                };
            });
        };
        this.isValueEmpty = (value) => {
            if (typeof value === 'number') {
                return false;
            }
            if (typeof value === 'boolean') {
                return false;
            }
            return isEmpty(value);
        };
        this.getFormattedActiveRecords = () => {
            return this.db
                .findAll({
                level: null,
                parentId: null,
                where: {
                    __action: { $nin: [RecordActionType.REMOVED, RecordActionType.TEMPORARY] },
                },
            })
                .value({ cleanMetadata: true, cleanDatabaseMeta: true })
                .map(splitValueToMergedValue);
        };
        this.runValidationOnRecord = async ({ recordData, changedColumnId, isUncommitted, columnsToRevalidate, }) => {
            const level = recordData.__level ?? 0;
            const columnDefinitions = this.getColumnDefinitions(level);
            const rowValue = cleanMetadataFromRecord(recordData);
            // Execute validators of the row that is changed
            const currentValidationState = await checkNestedRecordValidationStatus({
                rowValue,
                elementId: this.elementId,
                screenId: this.screenId,
                recordId: recordData._id,
                columnDefinitions,
                columnsToValidate: columnsToRevalidate
                    ? new Set(columnsToRevalidate)
                    : recordData.__dirtyColumns || new Set(),
            });
            // If the validation result is an empty object, we normalize to to undefined
            let normalizedCurrentValidationState = isEmpty(currentValidationState) ? undefined : currentValidationState;
            const normalizedPreviousValidationState = isEmpty(recordData.__validationState)
                ? undefined
                : recordData.__validationState;
            // If the user just edited a record, keep the server side validation errors unless they belong to the cell that the user just edited
            if (changedColumnId && normalizedPreviousValidationState) {
                /**
                 * Copy over server validation rules to the new validation state of the record because the client record validation can't regenerate
                 * the server side errors.
                 *
                 * The client side errors take precedence, so if there is a validation error in the new validation result we will not keep
                 * the server generated value because the client validation error blocks the user from even submitting the page value for server
                 * evaluation
                 *  */
                Object.values(normalizedPreviousValidationState).forEach((e) => {
                    if (e.columnId &&
                        e.columnId !== changedColumnId &&
                        e.validationRule.startsWith(SERVER_VALIDATION_RULE_PREFIX) &&
                        !Object.values(normalizedCurrentValidationState || {}).find(v => v.columnId === e.columnId)) {
                        if (!normalizedCurrentValidationState) {
                            normalizedCurrentValidationState = {};
                        }
                        normalizedCurrentValidationState[e.columnId] = e;
                    }
                });
            }
            // Only trigger event if the validation state of the record changed compared to its previous state.
            if (!isEqual(normalizedCurrentValidationState, normalizedPreviousValidationState)) {
                // Get a fresh copy of the record from the database because the values could have changed while the validation run
                const updatedRecord = {
                    ...this.getRawRecord({
                        id: recordData._id,
                        level: recordData.__level,
                        cleanMetadata: false,
                        isUncommitted,
                    }),
                    __validationState: normalizedCurrentValidationState,
                };
                this.db.update({
                    data: updatedRecord,
                    beforeUpdate: record => this.formatRecordValue(this.repackRecordData(record), undefined, true, level),
                    afterUpdate: () => this.notifyValidationSubscribers([updatedRecord], isUncommitted),
                    isUncommitted,
                });
                const errors = isEmpty(normalizedCurrentValidationState) ||
                    updatedRecord.__phantom === true ||
                    updatedRecord.__uncommitted === true
                    ? []
                    : this.getErrorList(this.getErrorMessages({ records: [updatedRecord] }));
                dispatchUpdateNestedFieldValidationErrorsForRecord({
                    screenId: this.screenId,
                    elementId: this.elementId,
                    recordId: recordData._id,
                    level,
                    validationResult: errors,
                    isUncommitted,
                });
            }
            return currentValidationState;
        };
        this.getNode = (level) => {
            if (this.sublevelProperty) {
                return this.nodes[0];
            }
            const targetLevel = isNil(level) ? 0 : level;
            return this.nodes[targetLevel];
        };
        this.getColumnDefinitions = (level) => {
            if (this.sublevelProperty) {
                return this.columnDefinitions[0];
            }
            const targetLevel = isNil(level) ? 0 : level;
            return this.columnDefinitions[targetLevel];
        };
        this.activeOptionsMenuItem = activeOptionsMenuItem || undefined;
        this.bind = bind || undefined;
        this.columnDefinitions = columnDefinitions;
        this.elementId = elementId;
        this.fieldType = fieldType;
        this.locale = locale;
        this.initialFilter = Array.from(Array(nodes.length).keys()).map(i => filter[i]);
        setWith(this.filter, [0, 0], filter?.[0], Object);
        this.hasNextPage = hasNextPage;
        this.initialValues = initialValues;
        this.isTransient = isTransient;
        this.isNoServerLookups = !!isNoServerLookups;
        this.levelMap = levelMap;
        this.contextNode = contextNode ? String(contextNode) : undefined;
        this.nodes = nodes;
        this.nodeTypes = nodeTypes;
        this.initialOrderBy = Array.from(Array(nodes.length).keys()).map(i => orderBy[i]);
        setWith(this.orderBy, [0, 0], orderBy?.[0], Object);
        setWith(this.lastFetchedPage, [0, 0], 0, Object);
        this.parentElementId = parentElementId;
        this.recordContext = recordContext;
        this.screenId = screenId;
        this.mapServerRecordFunctions = mapServerRecordFunctions;
        this.referenceLookupContextLevel = referenceLookupContextLevel;
        this.sublevelProperty = sublevelProperty;
        this.db = new LokiDb({
            dbKey,
            elementId: this.elementId,
            initialValues: [],
            isTransient: this.isTransient,
            screenId: this.screenId,
        });
        this.initialValues.forEach(recordData => {
            // eslint-disable-next-line @typescript-eslint/naming-convention
            const { [this.levelMap?.[0] ?? '']: _, ...data } = recordData;
            const newRecord = this.db.addRecord({
                data: data,
                level: 0,
                beforeInsert: record => this.formatRecordValue(this.repackRecordData(record), undefined, true, 0),
                cleanMetadata: false,
                action: 'none',
                dirty: false,
            });
            this.addNestedLevelRecursive({
                recordData: recordData,
                level: 0,
                shouldNotifySubscribers: false,
                parentId: newRecord._id,
                action: 'none',
                dirty: false,
            });
        });
    }
    async getPhantomRecords({ level, parentId, } = {}) {
        const records = this.db
            .findAll({ level, parentId, where: { __phantom: { $eq: true } } })
            .value({ cleanMetadata: false });
        if (records.length > 0) {
            return records;
        }
        const newRecord = await this.createNewPhantomRow({
            level,
            parentId,
            shouldNotifySubscribers: false,
            cleanMetadata: false,
        });
        return [...records, newRecord];
    }
    hasDirtyPhantomRows() {
        const records = this.db.findAll({ where: { __phantom: { $eq: true }, __dirty: { $eq: true } } }).value();
        return records.length !== 0;
    }
    calculateAggregatedValue({ aggregationMethod, aggregationKey, level = 0, parentId, where = transformToLokiJsFilter(this.getInternalFilter({ level: level || 0, parentId: parentId || undefined })), includeUnloaded, includePhantom, isUncommitted, }) {
        if (this.hasNextPage) {
            return Number.NaN;
        }
        return this.db.calculateAggregatedValue({
            aggregationMethod,
            aggregationKey,
            level,
            parentId,
            where,
            includeUnloaded,
            includePhantom,
            isUncommitted,
        });
    }
    async getDefaultPhantomRow({ level = 0 } = {}) {
        const fieldProperties = getStore().getState().screenDefinitions[this.screenId].metadata.uiComponentProperties[this.elementId];
        const emptyRecord = this.getColumnDefinitions(level).reduce((acc, colDef) => {
            const bind = String(colDef.properties.bind);
            if (bind !== '_id') {
                acc[bind] = null;
            }
            return acc;
        }, {});
        const { _id: discardedId, ...record } = fieldProperties.isTransient
            ? emptyRecord
            : (await fetchNestedDefaultValues({
                screenId: this.screenId,
                elementId: this.elementId,
                data: { __phantom: true },
            })).nestedDefaults || emptyRecord;
        return {
            ...record,
            __phantom: true,
            __dirtyColumns: new Set(),
        };
    }
    async resetRowToDefaults({ id, level, parentId, shouldNotifySubscribers = true, isOrganicChange = false, resetDirtiness = false, }) {
        const record = this.db.findOne({ id, level, parentId, cleanMetadata: false });
        const defaults = splitValueToMergedValue(await this.getDefaultPhantomRow());
        const updatedData = { ...record, ...defaults, __validationState: {} };
        if (resetDirtiness) {
            updatedData.__dirty = false;
            updatedData.__dirtyColumns = new Set();
        }
        this.db.update({
            data: updatedData,
            beforeUpdate: toBeUpdated => this.formatRecordValue(this.repackRecordData(toBeUpdated), undefined, true, level),
            afterUpdate: () => {
                updatedData.__forceRowUpdate = true;
                if (shouldNotifySubscribers) {
                    this.notifyValueChangeSubscribers(RecordActionType.MODIFIED, updatedData);
                }
                if (isOrganicChange) {
                    this.notifyValidationSubscribers([updatedData]);
                    dispatchUpdateNestedFieldValidationErrorsForRecord({
                        screenId: this.screenId,
                        elementId: this.elementId,
                        recordId: updatedData._id,
                        level,
                    });
                }
            },
        });
    }
    async createNewPhantomRow({ level, parentId, shouldNotifySubscribers = true, cleanMetadata = true, isUncommitted = false, } = {}) {
        /**
         * If a getDefaults call take very long, this function might be called again while the defaults are being fetched from the server.
         * We should not execute two default calls at once, first its bad server performance, second it can result two phantom rows in the collection
         * value which cause all sort of problems.
         *  */
        // Determine which promise to use based on isUncommitted
        const contextKey = `${level ?? 0}-${parentId ?? 'root'}-${isUncommitted}`;
        const existingPhantomRow = this.db.findOne({
            cleanMetadata: false,
            level,
            parentId,
            where: { __phantom: { $eq: true } },
            isUncommitted,
        });
        if (existingPhantomRow) {
            return existingPhantomRow;
        }
        if (!this.phantomRowPromises) {
            this.phantomRowPromises = new Map();
        }
        const existingPromise = this.phantomRowPromises.get(contextKey);
        if (existingPromise) {
            return existingPromise;
        }
        const promise = new Promise((resolve, reject) => {
            (async () => {
                try {
                    const recordData = await this.getDefaultPhantomRow({ level });
                    const undeletedPhantomRow = this.db.findOne({
                        cleanMetadata: false,
                        level,
                        parentId,
                        where: { __phantom: { $eq: true } },
                        isUncommitted,
                    });
                    if (undeletedPhantomRow) {
                        resolve(undeletedPhantomRow);
                        return;
                    }
                    const record = this.addRecord({
                        recordData,
                        level,
                        shouldNotifySubscribers,
                        parentId,
                        cleanMetadata,
                        isUncommitted,
                        dirty: false,
                    });
                    resolve(record);
                }
                catch (error) {
                    reject(error);
                }
                finally {
                    this.phantomRowPromises?.delete(contextKey);
                }
            })();
        });
        this.phantomRowPromises.set(contextKey, promise);
        return promise;
    }
    async resetPhantomRowToDefault() {
        const recordData = await this.getDefaultPhantomRow();
        const oldRecord = this.db.findOne({ where: { __phantom: { $eq: true } }, cleanMetadata: false });
        const updatedData = { ...oldRecord, ...recordData };
        if (!oldRecord) {
            this.db.addRecord({
                data: recordData,
                dirty: false,
                afterInsert: () => {
                    recordData.__forceRowUpdate = true; // We only want to tell to desktop table to force reload the row but not save in db for next uses.
                    this.notifyValueChangeSubscribers(RecordActionType.ADDED, recordData);
                },
            });
        }
        else {
            this.db.update({
                data: updatedData,
                afterUpdate: () => {
                    updatedData.__forceRowUpdate = true; // We only want to tell to desktop table to force reload the row but not save in db for next uses.
                    this.notifyValueChangeSubscribers(RecordActionType.MODIFIED, updatedData);
                },
            });
        }
    }
    createRecord(args) {
        return this.db.createRecord(args);
    }
    commitPhantomRow({ id, level, parentId, shouldNotifySubscribers = true, isOrganicChange = false, }) {
        const record = this.db.findOne({ id, level, parentId, cleanMetadata: false });
        // eslint-disable-next-line @typescript-eslint/naming-convention
        const { __phantom: _, ...updatedData } = record;
        this.db.update({
            data: updatedData,
            beforeUpdate: toBeUpdated => this.formatRecordValue(this.repackRecordData(toBeUpdated), undefined, !isOrganicChange, level),
            afterUpdate: () => {
                if (shouldNotifySubscribers) {
                    this.notifyValueChangeSubscribers(RecordActionType.ADDED, updatedData);
                }
            },
        });
    }
    getTreeFromNormalizedCollection({ records, includeActions = true, excludeEmptyChildCollections = false, isRequestingDefaults = false, removeNegativeId = false, cleanInputTypes = true, }) {
        if (records.length === 0) {
            return [];
        }
        if (this.levelMap === undefined) {
            throw new Error('Missing level information!');
        }
        const levelLength = objectKeys(this.levelMap).length;
        /**
         * full path of IDs to all records
         * e.g.
         * 1
         * ├── -1
         * └── 1
         *     ├── 2
         *     └── 3
         * becomes:
         * [
         *   [1]
         *   [1,-1],
         *   [1,1],
         *   [1,1,2],
         *   [1,1,3],
         * ]
         */
        const paths = records.map(element => {
            return this.db.getIdPathToRecordByIdAndLevel({ id: element._id, level: element.__level });
        });
        /**
         * unique IDs by level with new items at the end
         * e.g.
         *
         * [
         *   [1],
         *   [1,-1],
         *   [3,2],
         * ]
         *
         */
        const uniqueIdsByLevel = Array.from({ length: levelLength }, (_, i) => i).reduce((acc, i) => {
            acc[i] = sortBy(uniq(paths.map(path => path[i]).filter(Boolean))).reverse();
            return acc;
        }, []);
        // cache for explored records (cache key = <record.__compositeKey>)
        const explored = {};
        // cache for record's children (cache key = <record_id>.<level> )
        const deps = {};
        const tree = [];
        // process 'uniqueIdsByLevel' backwards, i.e. start with more nested records
        Array.from({ length: levelLength }, (_, i) => levelLength - i - 1).forEach(i => {
            const ids = uniqueIdsByLevel[i];
            ids.forEach(id => {
                const level = i === 0 ? undefined : i;
                const element = this.db.findOne({ id, level, cleanMetadata: false });
                if (element.__compositeKey) {
                    if (i > 0) {
                        deps[`${element.__parentId}.${i - 1}`] = [
                            ...(deps[`${element.__parentId}.${i - 1}`] || []),
                            element.__compositeKey,
                        ];
                    }
                    explored[element.__compositeKey] = element;
                    const childProperty = this.levelMap && this.levelMap[i];
                    // add all children
                    if (childProperty !== undefined) {
                        const children = deps[element.__compositeKey] || [];
                        // do not set empty child collections so that the server doesn't override any previous value
                        if (!excludeEmptyChildCollections || children.length !== 0) {
                            set(element, childProperty, children.map(dep => {
                                return new CollectionDataRow(explored[dep], this.columnDefinitions, this.nodes, this.nodeTypes, {
                                    includeActions,
                                    removeNegativeId,
                                    isRequestingDefaults,
                                    cleanInputTypes,
                                    isTree: !!this.sublevelProperty,
                                }).getChangeset();
                            }));
                        }
                    }
                }
                if (i === 0) {
                    const topLevelRecord = new CollectionDataRow(element, this.columnDefinitions, this.nodes, this.nodeTypes, {
                        includeActions,
                        removeNegativeId,
                        isRequestingDefaults,
                        cleanInputTypes,
                        isTree: !!this.sublevelProperty,
                    }).getChangeset();
                    // add only top-level records to tree
                    tree.push(topLevelRecord);
                }
            });
        });
        return tree;
    }
    getOrderBy({ level, parentId }) {
        if (this.sublevelProperty) {
            return isEmpty(this.initialOrderBy[0]) ? { _id: -1 } : this.initialOrderBy[0];
        }
        const initialOrderBy = this.initialOrderBy[level ?? 0];
        const fallbackOrderBy = isEmpty(initialOrderBy) ? { _id: -1 } : initialOrderBy;
        const storedOrderBy = get(this.orderBy, [level ?? 0, parentId ?? 0]);
        return isEmpty(storedOrderBy) ? fallbackOrderBy : storedOrderBy;
    }
    getFilter({ level, parentId }) {
        let filter = get(this.filter, [this.sublevelProperty ? 0 : level, parentId ?? 0], this.initialFilter[this.sublevelProperty ? 0 : level]);
        if (!level && this.activeOptionsMenuItem && filter) {
            filter = mergeGraphQLFilters([filter, this.activeOptionsMenuItem.graphQLFilter]);
        }
        else if (!level && this.activeOptionsMenuItem?.graphQLFilter) {
            filter = this.activeOptionsMenuItem.graphQLFilter;
        }
        return omitBy(filter, v => v === undefined);
    }
    getInternalFilter({ level, parentId }) {
        return transformFilterForCollectionValue(omitBy(this.getFilter({ level, parentId }), v => v === undefined));
    }
    resetFilter() {
        setWith(this.filter, [0, 0], this.initialFilter?.[0], Object);
    }
    async fetchInvalidUnloadedRecords() {
        const idsToFetch = this.getInvalidUnloadedRecords().map(r => r._id);
        const pageDefinition = getPageDefinitionFromState(this.screenId);
        const pageProperties = getPagePropertiesFromPageDefinition(pageDefinition);
        const idFilter = { _id: { _in: idsToFetch } };
        const result = await fetchCollectionData({
            screenDefinition: pageDefinition,
            rootNode: pageProperties.node,
            rootNodeId: pageDefinition.selectedRecordId || '',
            elementId: this.elementId,
            nestedFields: this.getColumnDefinitions(0),
            queryArguments: { filter: JSON.stringify(idFilter) },
            bind: this.bind,
            nodeTypes: this.nodeTypes,
            node: this.getNode(0),
        });
        this.addQueryResultToCollection({
            result,
            level: 0,
            markAsAdded: false,
        });
    }
    isReferenceField(fieldProperties, pageProperties) {
        // If it has valueField its a referenceField so we know for sure it's okay
        if (fieldProperties._controlObjectType === FieldKey.Reference ||
            fieldProperties._controlObjectType === FieldKey.Pod ||
            fieldProperties._controlObjectType === FieldKey.FilterSelect ||
            Object.prototype.hasOwnProperty.call(fieldProperties, 'valueField')) {
            return true;
        }
        if (fieldProperties._controlObjectType === FieldKey.Table ||
            fieldProperties._controlObjectType === FieldKey.Tree ||
            fieldProperties._controlObjectType === FieldKey.Calendar ||
            fieldProperties._controlObjectType === FieldKey.MultiFileDeposit ||
            fieldProperties._controlObjectType === FieldKey.NestedGrid) {
            return false;
        }
        // If not we need to check if we are checking some reference from other fields like pod
        // If the nodeType of the property is a Object and it DONT start with a underscore it means that its a reference to other node.
        const propertyType = findDeepPropertyDetails(schemaTypeNameFromNodeName(pageProperties?.node), this.bind || this.elementId, this.nodeTypes);
        return propertyType?.kind === 'OBJECT' && !propertyType?.type.startsWith('_');
    }
    async fetchServerRecords({ pageDefinition, pageSize, tableFieldProperties, group, axiosCancelToken, level = 0, parentId, cursor, totalCount = false, }) {
        if (this.isNoServerLookups) {
            return {};
        }
        const pageProperties = getPagePropertiesFromPageDefinition(pageDefinition);
        const selectedRecordId = parentId || pageDefinition.selectedRecordId || '';
        const queryArguments = {
            first: pageSize,
        };
        const sortOrder = this.getOrderBy({ level, parentId });
        let orderBy = sortOrder;
        if (group !== undefined && group?.value === undefined) {
            // eslint-disable-next-line @sage/redos/no-vulnerable
            const nestedValueMatch = group.key.match(/(.*)\./);
            orderBy = deepMerge(
            // always order by id of nested references
            group?.type === FieldKey.Reference && nestedValueMatch
                ? set(set({}, group.key, 1), `${nestedValueMatch[1]}._id`, 1)
                : {}, 
            // for groups add only sort conditions on the grouping column
            objectKeys(flat(sortOrder))
                .filter(k => k.startsWith(group.key))
                .reduce((acc, curr) => {
                set(acc, curr, get(sortOrder, curr));
                return acc;
            }, {}));
        }
        if (orderBy && !isEmpty(orderBy)) {
            queryArguments.orderBy = JSON.stringify(orderBy);
        }
        let filter = this.getFilter({ level, parentId });
        if (filter || group !== undefined) {
            const groupFilter = group ? getGroupFilterValue({ group, mode: 'server' }) : {};
            filter = deepMerge(filter, groupFilter);
            queryArguments.filter = JSON.stringify(filter);
        }
        if (cursor) {
            queryArguments.after = cursor;
        }
        let graphqlQueryResult;
        const parentLevel = level > 0
            ? pageDefinition.metadata.uiComponentProperties[this.elementId]?.levels?.[level - 1]
            : undefined;
        const rootNode = parentLevel ? String(parentLevel.node) : pageProperties.node;
        const rootNodeId = parentLevel ? parentLevel.childProperty : this.elementId;
        const bind = parentLevel ? undefined : convertDeepBindToPath(tableFieldProperties.bind || this.elementId);
        const targetProperty = findDeepPropertyDetails(schemaTypeNameFromNodeName(rootNode), bind, this.nodeTypes, false);
        /**
         * A table is considered transient if the field is transient OR the page that the field is on is transient OR
         * the table doesn't belong to any node OR has not been saved to the server yet and we cannot query it OR it is
         * bound to a JSON deep nested array.
         *  */
        const isTransient = !selectedRecordId ||
            this.isTransient ||
            pageProperties.isTransient ||
            !pageProperties.node ||
            !this.nodes.length ||
            targetProperty?.type === GraphQLTypes.Json;
        if (this.isReferenceField(tableFieldProperties, pageProperties) ||
            this.elementId === navigationPanelId ||
            this.elementId === ELEMENT_ID_APPLICATION_CODE_LOOKUP) {
            const referenceFieldProperties = tableFieldProperties;
            const data = await fetchReferenceFieldData({
                fieldProperties: referenceFieldProperties,
                screenId: this.screenId,
                elementId: this.elementId,
                filter,
                orderBy,
                after: queryArguments.after,
                valueField: referenceFieldProperties.valueField || this.elementId,
                parentElementId: this.parentElementId,
                recordContext: this.recordContext,
                contextNode: this.contextNode,
                axiosCancelToken,
                level: this.referenceLookupContextLevel,
                group,
                pageSize,
            });
            const result = removeEdgesDeep(data, true, true);
            const dataAccessor = result?.lookups ? `lookups.${tableFieldProperties?.bind}` : 'query';
            graphqlQueryResult = get(result, dataAccessor);
        }
        else if (!isTransient) {
            const treeProperties = parentLevel ? undefined : tableFieldProperties;
            graphqlQueryResult = await fetchCollectionData({
                screenDefinition: pageDefinition,
                rootNode,
                rootNodeId: selectedRecordId,
                elementId: rootNodeId,
                nestedFields: getNestedFieldsFromProperties(tableFieldProperties, level),
                queryArguments,
                bind: bind || undefined,
                group,
                treeProperties,
                totalCount,
                nodeTypes: this.nodeTypes,
                node: this.getNode(level),
            });
        }
        this.addQueryResultToCollection({ result: graphqlQueryResult, level, parentId, markAsAdded: false, group });
        return graphqlQueryResult;
    }
    notifyValueChangeSubscribers(type, recordValue, isUncommitted = false) {
        if (!isUncommitted && this.db.isCollectionDirty()) {
            dispatchSetFieldDirty({
                screenId: this.screenId,
                elementId: this.elementId,
            });
        }
        const subscriptions = isUncommitted ? this.valueChangeUncommittedSubscribers : this.valueChangeSubscribers;
        subscriptions.forEach(subscriber => {
            subscriber(type, recordValue);
        });
    }
    getErrorMessages({ records = this.getAllInvalidRecords({ level: null, parentId: null }), fieldProperties = getStore().getState().screenDefinitions[this.screenId].metadata.uiComponentProperties[this.elementId], withoutServerErrors = false, } = {}) {
        const fieldTitle = resolveByValue({
            screenId: this.screenId,
            skipHexFormat: true,
            propertyValue: fieldProperties.title,
            rowValue: undefined,
        });
        return records.reduce((acc, curr) => {
            const { __validationState: validationState, _id: recordId } = curr;
            if (validationState) {
                objectKeys(validationState).forEach(valKey => {
                    const { message, columnId, validationRule } = validationState[valKey];
                    if (withoutServerErrors && validationRule.indexOf(SERVER_VALIDATION_RULE_PREFIX) === 0) {
                        return acc;
                    }
                    if (!columnId) {
                        acc.push({
                            fieldName: fieldTitle,
                            mainFieldValue: String(get(curr, fieldProperties.mainField || '_id')),
                            message,
                            recordId,
                            validationRule,
                        });
                        return acc;
                    }
                    const mainFieldName = findNestedFieldProperties(this.screenId, fieldProperties.mainField || '_id', fieldProperties, 0);
                    const columnName = findNestedFieldProperties(this.screenId, columnId, fieldProperties, 0);
                    return acc.push({
                        fieldName: fieldTitle,
                        mainFieldName,
                        mainFieldValue: String(get(curr, fieldProperties.mainField || '_id')),
                        columnId,
                        columnName,
                        message,
                        recordId,
                        validationRule,
                        level: curr.__level,
                    });
                });
            }
            return acc;
        }, []);
    }
    getAllInvalidRecords({ level, parentId, } = {}) {
        return this.db
            .getAllInvalidRecords({ level, parentId })
            .sort({
            orderBy: this.getOrderBy({ level, parentId }),
            columnDefinitions: this.getColumnDefinitions(level),
        })
            .value({ cleanMetadata: false });
    }
    getInvalidUnloadedRecords({ level, parentId, withoutServerErrors, } = {}) {
        return this.db
            .getInvalidUnloadedRecords({ level, parentId, withoutServerErrors })
            .sort({
            orderBy: this.getOrderBy({ level, parentId }),
            columnDefinitions: this.getColumnDefinitions(level),
        })
            .value({ cleanMetadata: false });
    }
    getInvalidLoadedRecords({ level, parentId, withoutServerErrors, } = {}) {
        return this.db
            .getInvalidLoadedRecords({ level, parentId, withoutServerErrors })
            .sort({
            orderBy: this.getOrderBy({ level, parentId }),
            columnDefinitions: this.getColumnDefinitions(level),
        })
            .value({ cleanMetadata: false });
    }
    async fetchAllErrors() {
        const fieldProperties = getStore().getState().screenDefinitions[this.screenId].metadata.uiComponentProperties[this.elementId];
        const invalidUnloaded = this.getInvalidUnloadedRecords();
        if (fieldProperties.mainField !== '_id') {
            const idsToFetch = invalidUnloaded.slice(0, 500).map(r => r._id);
            const pageDefinition = getPageDefinitionFromState(this.screenId);
            const pageProperties = getPagePropertiesFromPageDefinition(pageDefinition);
            const idFilter = { _id: { _in: idsToFetch } };
            const result = await fetchCollectionData({
                screenDefinition: pageDefinition,
                rootNode: pageProperties.node,
                rootNodeId: pageDefinition.selectedRecordId || '',
                elementId: this.elementId,
                nestedFields: this.getColumnDefinitions(0),
                queryArguments: { filter: JSON.stringify(idFilter), first: idsToFetch.length },
                bind: this.bind,
            });
            this.addQueryResultToCollection({
                result,
                level: 0,
                markAsAdded: false,
            });
        }
    }
    hasMultipleLevels() {
        return this.levelMap !== undefined;
    }
    getValidationStateByColumn() {
        return this.db.getValidationStateByColumn(this.columnDefinitions.length);
    }
    notifyValidationSubscribers(recordsWithChangedValidationStatus, isUncommitted = false) {
        // Calculate global validation status.
        const globalValidationState = this.getValidationStateByColumn();
        const subscribers = isUncommitted ? this.validityChangeUncommittedSubscribers : this.validityChangeSubscribers;
        // Notify the subscribers for every record
        recordsWithChangedValidationStatus.forEach(r => subscribers.forEach(subscriber => {
            subscriber({
                globalValidationState,
                recordValidationState: r.__validationState || {},
                recordId: r._id,
                recordLevel: r.__level,
            });
        }));
    }
    getUpdatedRecord({ previous, newData, toBeMarkedAsDirty, addToDirty = [], }) {
        const recordData = this.formatRecordValue(newData, previous);
        const updatedKeys = (toBeMarkedAsDirty ||
            objectKeys(recordData).reduce((acc, key) => {
                return key === '_id' ||
                    key.startsWith('__') ||
                    (this.isValueEmpty(get(previous, key)) && this.isValueEmpty(recordData[key])) ||
                    isEqual(recordData[key], get(previous, key))
                    ? acc
                    : acc.concat(key);
            }, [])).concat(addToDirty);
        const newDirtyColumns = Array.from(newData.__dirtyColumns || []);
        const dirtyColumns = [...updatedKeys, ...newDirtyColumns].reduce((acc, key) => {
            return acc.add(key);
        }, previous.__dirtyColumns || new Set());
        /**
         * When items removed from array field (multi reference, multi select etc), the deep merge operation above will persist and duplicate the values,
         *  so we need to ensure that the array remains unique
         *  */
        const mergedRecordValue = deepMerge(previous, recordData, false, true);
        return {
            ...mergedRecordValue,
            __dirty: toBeMarkedAsDirty ? toBeMarkedAsDirty.length > 0 : true,
            __dirtyColumns: dirtyColumns,
            // If it's a new record which has not been saved yet, we keep it as is
            __action: previous?.__action === RecordActionType.ADDED ? RecordActionType.ADDED : RecordActionType.MODIFIED,
        };
    }
    getChangedRecords() {
        return this.db
            .findAll({
            where: {
                __action: {
                    $in: [RecordActionType.ADDED, RecordActionType.MODIFIED, RecordActionType.REMOVED],
                },
            },
        })
            .value({ cleanMetadata: false });
    }
    updateMany(records) {
        records.forEach(recordData => {
            this.addOrUpdateRecordValue({ recordData });
        });
    }
    findAndUpdate({ recordData, level, shouldNotifySubscribers = true, upsert = false, toBeMarkedAsDirty, resetRecordAction = false, isOrganicChange = false, changedColumnId, isUncommitted = false, }) {
        if (!recordData._id && !upsert) {
            throw new Error(`Cannot update a record without an '_id' property: ${{ recordData }}`);
        }
        let rec = recordData._id
            ? this.db.findOne({
                id: recordData._id,
                level,
                cleanMetadata: false,
                includeUnloaded: true,
                isUncommitted,
            })
            : null;
        // If a record is found and there is an uncommitted version of it, we should update that instead.
        const uncommittedRecord = this.db.findOne({
            id: recordData._id,
            level,
            cleanMetadata: false,
            includeUnloaded: true,
            isUncommitted: true,
        });
        if (uncommittedRecord) {
            rec = uncommittedRecord;
        }
        if (!rec) {
            if (!upsert) {
                throw new Error(`A record with ID '${recordData._id}' could not be found therefore it could not be updated. Are sure you want to update an existing record?`);
            }
            else {
                return this.db.addRecord({
                    data: recordData,
                    level,
                    beforeInsert: record => this.formatRecordValue(this.repackRecordData(record)),
                    afterInsert: ({ action, record }) => {
                        if (shouldNotifySubscribers) {
                            this.notifyValueChangeSubscribers(action, record, false);
                        }
                    },
                });
            }
        }
        else {
            const updatedData = this.getUpdatedRecord({
                previous: rec,
                newData: recordData,
                toBeMarkedAsDirty,
            });
            if (resetRecordAction) {
                delete updatedData.__action;
            }
            return this.db.update({
                data: updatedData,
                beforeUpdate: record => this.formatRecordValue(this.repackRecordData(record), undefined, !isOrganicChange, level),
                afterUpdate: () => {
                    if (updatedData.__phantom) {
                        updatedData.__forceRowUpdate = true;
                    }
                    if (shouldNotifySubscribers) {
                        this.notifyValueChangeSubscribers(RecordActionType.MODIFIED, updatedData, rec?.__uncommitted || false);
                    }
                    if (isOrganicChange) {
                        this.runValidationOnRecord({ recordData: updatedData, changedColumnId });
                    }
                },
            });
        }
    }
    addQueryResultToCollection({ result, level, markAsAdded = false, parentId, group, }) {
        if (result?.data) {
            this.hasNextPage = Boolean(result.pageInfo?.hasNextPage);
            return result.data.map((rec) => {
                const existingRecord = this.db.findOne({
                    id: rec._id,
                    level,
                    parentId,
                    cleanMetadata: false,
                    includeUnloaded: true,
                    where: {
                        __isGroup: { $eq: rec.__isGroup },
                        __groupKey: { $eq: rec.__groupKey },
                    },
                });
                /**
                 * If the record was already in the database for some reason we update it with the data coming from the server, but letting local changes to take priority, EXCEPT the __groupCount & __cursor (which can change with groups)
                 *  */
                if (existingRecord) {
                    const data = {
                        ...rec,
                        ...existingRecord,
                        ...(rec.__groupCount && { __groupCount: rec.__groupCount }),
                        ...(rec.__cursor && { __cursor: rec.__cursor }),
                        ...(group?.aggFunc && { __aggFunc: group.aggFunc }),
                    };
                    // We do not merge deeply the incoming record and the group key property could be lost so we need to ensure that we preserve it
                    if (group?.key) {
                        const groupKeyPath = getGroupKey({ groupKey: group.key, type: group.type });
                        if (!get(data, groupKeyPath)) {
                            set(data, groupKeyPath, get(rec, groupKeyPath));
                        }
                    }
                    // It is important to unset the __unloaded flag.
                    delete data.__unloaded;
                    return this.db.update({
                        data,
                        beforeUpdate: record => this.formatRecordValue(this.repackRecordData(record), undefined, true, level),
                    });
                }
                return this.db.addRecord({
                    data: { ...rec, ...(group?.aggFunc && { __aggFunc: group.aggFunc }) },
                    dirty: false,
                    level,
                    parentId,
                    cleanMetadata: true,
                    ...(!markAsAdded && { action: 'none' }),
                    beforeInsert: record => this.formatRecordValue(this.repackRecordData(record), undefined, true, level),
                });
            });
        }
        return [];
    }
    /** For data to be used in control objects without lokiJs metadata */
    getRawRecords() {
        return this.db.findAll({ level: null, parentId: null }).value();
    }
    getRawRecord({ cleanMetadata = true, id, includeUnloaded = false, level, temporaryRecords = [], isUncommitted = false, }) {
        return this.db.withTemporaryRecords(db => {
            return db.findOne({ id, level, cleanMetadata, includeUnloaded, isUncommitted });
        }, temporaryRecords);
    }
    /** Find a record based on a property's value */
    getRawRecordByFieldValue({ fieldName, fieldValue, level = 0, includeUnloaded = false, }) {
        const record = this.db.findOne({
            level,
            where: {
                [fieldName]: { $eq: fieldValue },
            },
            includeUnloaded,
        });
        if (!record) {
            return null;
        }
        if (this.levelMap?.[level]) {
            record[this.levelMap?.[level]] = this.getNestedRecordsRecursive({
                parentId: record._id,
                level: level + 1,
            });
        }
        return splitValueToMergedValue(cleanMetadataFromRecord(record));
    }
    getAncestorIds(args) {
        return this.db.getAncestorIds(args);
    }
    getNestedRecordsRecursive({ parentId, level, cb = (record) => record, }) {
        if (this.levelMap === undefined) {
            throw new Error('Missing level information!');
        }
        const records = this.db
            .findAll({ parentId, level })
            .sort({ orderBy: { _id: -1 }, columnDefinitions: this.getColumnDefinitions(level) })
            .value({ cleanMetadata: false });
        if (this.levelMap[level]) {
            records.forEach(record => {
                set(record, this.levelMap[level], this.getNestedRecordsRecursive({
                    parentId: record._id,
                    level: level + 1,
                    cb,
                }));
            });
        }
        return records.map(record => splitValueToMergedValue(cleanMetadataFromRecord(cb(record))));
    }
    getRecordWithChildren({ id, level = 0, isUncommitted = false, cleanMetadata = true, cb = (record) => record, }) {
        const foundRecord = this.db.findOne({ id, level, cleanMetadata: false, isUncommitted });
        if (!foundRecord) {
            return null;
        }
        if (this.levelMap?.[level]) {
            foundRecord[this.levelMap?.[level]] = this.getNestedRecordsRecursive({
                parentId: foundRecord._id,
                level: level + 1,
            });
        }
        return cleanMetadata ? splitValueToMergedValue(cleanMetadataFromRecord(cb(foundRecord))) : cb(foundRecord);
    }
    getRecordByIdAndLevel(args) {
        return this.db.findOne({ ...args, cleanMetadata: false });
    }
    cleanCollectionData() {
        this.db.clear();
        setWith(this.lastFetchedPage, [0, 0], 0, Object);
        this.filter = [];
        this.orderBy = [];
        this.forceRefetch = true;
    }
    async fetchNestedGrid({ rootNode, selectedRecordId, level, levelProps, queryArguments, childProperty, }) {
        const pageDefinition = getPageDefinitionFromState(this.screenId);
        const graphqlQueryResult = await fetchCollectionData({
            screenDefinition: pageDefinition,
            rootNode,
            rootNodeId: selectedRecordId,
            elementId: childProperty,
            nestedFields: levelProps.columns,
            queryArguments,
        });
        const newLevelData = graphqlQueryResult
            ? {
                ...graphqlQueryResult,
                data: (graphqlQueryResult.data || []).map((d) => ({
                    ...d,
                    __level: level,
                    __parentId: selectedRecordId,
                })),
            }
            : graphqlQueryResult;
        return this.addQueryResultToCollection({
            result: newLevelData,
            level,
            markAsAdded: false,
            parentId: selectedRecordId,
        });
    }
    getNestedGrid({ level, parentId }) {
        return this.db
            .findAll({
            level,
            parentId,
            where: { __action: { $ne: RecordActionType.REMOVED } },
        })
            .sort({
            orderBy: this.getOrderBy({ level, parentId }),
            columnDefinitions: this.getColumnDefinitions(level),
        })
            .value({ cleanMetadata: false });
    }
    /**
     * Checks if a record has child records at the next level.
     * First checks the local cache, and if not found, makes a minimal server request.
     */
    async hasChildRecords({ level, parentId, tableFieldProperties, }) {
        // First check if we already have data in the local cache
        const cachedData = this.getData({
            level: level + 1,
            parentId,
            limit: 1,
            cleanMetadata: false,
        });
        if (cachedData.length > 0) {
            return true;
        }
        // If no cached data, make a minimal server request (fetch only 1 record without caching)
        try {
            const pageDefinition = getPageDefinitionFromState(this.screenId);
            const childProperty = this.levelMap?.[level];
            if (!childProperty) {
                return false;
            }
            // Get the current filter for this level
            const filter = this.getFilter({ level: level + 1, parentId });
            const graphqlQueryResult = await fetchCollectionData({
                screenDefinition: pageDefinition,
                rootNode: this.getNode(level),
                rootNodeId: parentId,
                elementId: childProperty,
                nestedFields: this.columnDefinitions[level + 1] || [],
                queryArguments: {
                    first: 1,
                    filter: filter ? JSON.stringify(filter) : undefined,
                },
            });
            return graphqlQueryResult?.data?.length > 0;
        }
        catch (error) {
            // In case of error, assume no children (fail gracefully)
            xtremConsole.warn(`Error checking for child records: ${error}`);
            return false;
        }
    }
    /**
     * Adds __hasChildRecords metadata to records by checking if they have children at the next level.
     * This is used when checkForChildRecords is enabled to control chevron visibility in nested grids.
     */
    async addChildRecordsMetadata({ level, parentId, tableFieldProperties, }) {
        // Check if this level can have children
        const hasChildProperty = this.levelMap?.[level] !== undefined;
        if (!hasChildProperty) {
            return;
        }
        // Get all records at this level with database metadata preserved
        const records = this.db
            .findAll({
            level,
            parentId,
            where: { __action: { $ne: RecordActionType.REMOVED } },
        })
            .value({ cleanMetadata: false, cleanDatabaseMeta: false });
        // Check each record for children and update metadata
        await Promise.all(records.map(async (record) => {
            try {
                const hasChildren = await this.hasChildRecords({
                    level,
                    parentId: record._id,
                    tableFieldProperties,
                });
                // Update the record with __hasChildRecords metadata
                this.db.update({
                    data: {
                        ...record,
                        __hasChildRecords: hasChildren,
                    },
                    beforeUpdate: toBeUpdated => toBeUpdated,
                    afterUpdate: () => {
                        // Silent update, no notifications needed
                    },
                });
            }
            catch (error) {
                xtremConsole.warn(`Error checking child records for ${record._id}: ${error}`);
                // On error, set to false (no chevron)
                this.db.update({
                    data: {
                        ...record,
                        __hasChildRecords: false,
                    },
                    beforeUpdate: toBeUpdated => toBeUpdated,
                    afterUpdate: () => {
                        // Silent update, no notifications needed
                    },
                });
            }
        }));
    }
    getFilteredSortedRecords({ pageSize, pageNumber, group, limit = pageSize, level = 0, cleanMetadata = true, parentId, temporaryRecords = [], }) {
        return this.db.withTemporaryRecords(db => {
            const queryBuilder = db.findAll({
                where: { __action: { $ne: RecordActionType.REMOVED } },
                level,
                parentId,
            });
            const filter = this.getInternalFilter({ level, parentId });
            if (group !== undefined) {
                // If requesting top-level groups on a transient table for a Text column, generate client-side groups only if they don't exist yet
                if (group.value === undefined && this.isTransient) {
                    const normalizedGroupKey = getGroupKey({ groupKey: group.key, type: group.type });
                    const existingGroups = db
                        .findAll({
                        where: {
                            __isGroup: { $eq: true },
                            __groupKey: { $eq: normalizedGroupKey },
                            ...(group.aggFunc && { __aggFunc: { $eq: group.aggFunc } }),
                        },
                        level,
                        parentId,
                    })
                        .value({ cleanMetadata: false });
                    if (existingGroups.length === 0) {
                        // Build the dataset to aggregate, respecting current filter
                        const baseBuilder = db.findAll({
                            where: { __action: { $ne: RecordActionType.REMOVED }, __isGroup: { $ne: true } },
                            level,
                            parentId,
                        });
                        if (filter && !isEmpty(filter)) {
                            try {
                                baseBuilder.findAll({ where: transformToLokiJsFilter(filter), level, parentId });
                            }
                            catch {
                                // Ignore invalid filters here; grouping will simply result in no groups
                            }
                        }
                        const records = baseBuilder.value({ cleanMetadata: false });
                        const counts = records.reduce((acc, rec) => {
                            const value = get(rec, group.key);
                            const key = value == null ? '' : String(value);
                            acc[key] = (acc[key] || 0) + 1;
                            return acc;
                        }, {});
                        objectKeys(counts).forEach(groupValue => {
                            const groupRecord = {
                                _id: `__group-${groupValue}`,
                                __isGroup: true,
                                __groupKey: normalizedGroupKey,
                                __groupCount: counts[groupValue],
                                ...(group.aggFunc && { __aggFunc: group.aggFunc }),
                            };
                            // Set the original group key path for proper cell rendering
                            set(groupRecord, group.key, groupValue);
                            db.addRecord({
                                data: groupRecord,
                                dirty: false,
                                level,
                                parentId,
                                cleanMetadata: true,
                                action: 'none',
                                beforeInsert: record => this.formatRecordValue(this.repackRecordData(record), undefined, true, level),
                            });
                        });
                    }
                }
                const groupKey = getGroupKey({ groupKey: group.key, type: group.type });
                queryBuilder.findAll({
                    where: group.key && group.value !== undefined
                        ? {
                            $and: [
                                ...[this.sublevelProperty ? {} : { __isGroup: { $ne: true } }],
                                getGroupFilterValue({ group, mode: 'client' }),
                                { __aggFunc: { $eq: group.aggFunc } },
                            ],
                        }
                        : {
                            $and: [
                                ...[this.sublevelProperty ? {} : { __isGroup: { $eq: true } }],
                                { __groupKey: { $eq: groupKey } },
                                { __groupCount: { $gt: 0 } },
                                { __aggFunc: { $eq: group.aggFunc } },
                            ],
                        },
                    level,
                    parentId,
                });
            }
            else {
                queryBuilder.findAll({
                    where: this.sublevelProperty ? {} : { __isGroup: { $ne: true } },
                    level,
                    parentId,
                });
            }
            if (filter && !isEmpty(filter)) {
                const lokiJsFilter = transformToLokiJsFilter(filter);
                const where = group === undefined || group?.value !== undefined ? lokiJsFilter : {};
                try {
                    queryBuilder.findAll({
                        where,
                        level,
                        parentId,
                    });
                }
                catch (e) {
                    if (isDevMode()) {
                        xtremConsole.error(e);
                        xtremConsole.error(`Failed to filter dataset using in-memory database for ${this.screenId} - ${this.elementId}`);
                        xtremConsole.error('GraphQL filter:');
                        xtremConsole.error(JSON.stringify(filter, null, 4));
                        xtremConsole.error('LokiJS filter:');
                        xtremConsole.error(JSON.stringify(where, null, 4));
                    }
                }
            }
            let orderBy = this.getOrderBy({ level, parentId });
            if (group !== undefined && group.value === undefined) {
                // eslint-disable-next-line @sage/redos/no-vulnerable
                const nestedValueMatch = group.key.match(/(.*)\./);
                orderBy = deepMerge(nestedValueMatch ? set(set({}, group.key, 1), `${nestedValueMatch[1]}._id`, 1) : {}, objectKeys(flat(orderBy))
                    .filter(k => k.startsWith(group.key))
                    .reduce((acc, curr) => {
                    set(acc, curr, get(orderBy, curr));
                    return acc;
                }, {}));
            }
            queryBuilder
                .sort({
                orderBy,
                columnDefinitions: this.getColumnDefinitions(level),
            })
                .skip(pageNumber * pageSize)
                .take(limit);
            return queryBuilder.value({ cleanMetadata });
        }, temporaryRecords);
    }
    generateIndex() {
        return this.db.generateIndex();
    }
    getActiveOptionsMenuItem() {
        return this.activeOptionsMenuItem;
    }
    async getPageWithCurrentQueryArguments({ pageNumber, pageSize, tableFieldProperties, group, level = 0, parentId, cleanMetadata = true, cursor, totalCount = false, }) {
        const temporaryRecords = resolveByValue({
            skipHexFormat: true,
            screenId: this.screenId,
            propertyValue: tableFieldProperties.additionalLookupRecords,
            rowValue: null,
            fieldValue: null,
        });
        if (group === undefined && pageNumber <= get(this.lastFetchedPage, [level, parentId ?? 0], -1)) {
            return this.getFilteredSortedRecords({
                pageSize,
                pageNumber,
                group,
                cleanMetadata,
                level,
                parentId,
                temporaryRecords,
            });
        }
        const pageDefinition = getPageDefinitionFromState(this.screenId);
        await this.fetchServerRecords({
            pageDefinition,
            tableFieldProperties,
            pageSize,
            group,
            level,
            parentId,
            cursor,
            totalCount,
        });
        setWith(this.lastFetchedPage, [level ?? 0, parentId ?? 0], pageNumber, Object);
        // If checkForChildRecords is enabled, add __hasChildRecords metadata to records
        const hasCheckForChildRecords = 'checkForChildRecords' in tableFieldProperties &&
            tableFieldProperties.checkForChildRecords;
        if (hasCheckForChildRecords && level !== undefined) {
            await this.addChildRecordsMetadata({
                level,
                parentId,
                tableFieldProperties,
            });
        }
        return this.getFilteredSortedRecords({
            pageSize,
            pageNumber,
            group,
            cleanMetadata,
            level,
            parentId,
            temporaryRecords,
        });
    }
    async getRecordWithCurrentQueryArguments({ pageNumber, pageSize, tableFieldProperties, group, level, parentId, cursor, cleanMetadata = true, }) {
        const pageDefinition = getPageDefinitionFromState(this.screenId);
        const temporaryRecords = resolveByValue({
            skipHexFormat: true,
            screenId: this.screenId,
            propertyValue: tableFieldProperties.additionalLookupRecords,
            rowValue: null,
            fieldValue: null,
        });
        await this.fetchServerRecords({
            pageDefinition,
            tableFieldProperties,
            pageSize: 1,
            group,
            level,
            parentId,
            cursor,
        });
        // If checkForChildRecords is enabled, add __hasChildRecords metadata to records
        const hasCheckForChildRecords = 'checkForChildRecords' in tableFieldProperties && tableFieldProperties.checkForChildRecords;
        if (hasCheckForChildRecords && level !== undefined) {
            await this.addChildRecordsMetadata({
                level,
                parentId,
                tableFieldProperties,
            });
        }
        return this.getFilteredSortedRecords({
            pageSize,
            pageNumber,
            group,
            level,
            parentId,
            cleanMetadata,
            temporaryRecords,
        });
    }
    async getPage({ axiosCancelToken, cleanMetadata = true, cursor, fetchPageSize, filters = [], group, level = 0, orderBy: order, pageNumber = 0, pageSize = 20, parentId, rawFilter, searchText: text, selectedOptionsMenuItem, tableFieldProperties, totalCount, }) {
        const { pageDefinition, shouldRefetch } = this.getFetchServerRecordsOptions({
            filters,
            group,
            level,
            orderBy: order,
            parentId,
            rawFilter,
            searchText: text,
            selectedOptionsMenuItem,
            tableFieldProperties,
        });
        if (shouldRefetch) {
            await this.fetchServerRecords({
                pageDefinition,
                tableFieldProperties,
                pageSize: fetchPageSize || pageSize,
                group,
                axiosCancelToken,
                parentId,
                level,
                cursor,
                totalCount,
            });
            this.forceRefetch = false;
        }
        // If checkForChildRecords is enabled, add __hasChildRecords metadata to records
        const hasCheckForChildRecords = 'checkForChildRecords' in tableFieldProperties && tableFieldProperties.checkForChildRecords;
        if (hasCheckForChildRecords && level !== undefined) {
            await this.addChildRecordsMetadata({
                level,
                parentId,
                tableFieldProperties,
            });
        }
        return this.getFilteredSortedAndTemporaryRecords({
            cleanMetadata,
            group,
            level,
            pageNumber,
            pageSize,
            parentId,
            tableFieldProperties,
        });
    }
    // INFO: This function fetches the record data iteratively, page by page
    //       until the last page has been retrieved. For large data sets, this
    //       may cause performance issues. Unless all records are required
    //       (e.g. the excel/csv export), the getPage() function should be used
    //       instead.
    async getPagesIteratively({ axiosCancelToken, cleanMetadata = true, cursor, fetchPageSize, filters = [], group, level = 0, orderBy: order, pageSize = 20, parentId, rawFilter, searchText: text, selectedOptionsMenuItem, tableFieldProperties, totalCount, }) {
        const { pageDefinition, shouldRefetch } = this.getFetchServerRecordsOptions({
            filters,
            group,
            level,
            orderBy: order,
            parentId,
            rawFilter,
            searchText: text,
            selectedOptionsMenuItem,
            tableFieldProperties,
        });
        if (shouldRefetch) {
            let result;
            let currCursor = cursor;
            do {
                result = await this.fetchServerRecords({
                    pageDefinition,
                    tableFieldProperties,
                    pageSize: fetchPageSize || pageSize,
                    group,
                    axiosCancelToken,
                    parentId,
                    level,
                    cursor: currCursor,
                    totalCount,
                });
                currCursor = result?.pageInfo?.endCursor;
            } while (result?.pageInfo?.hasNextPage);
            this.forceRefetch = false;
        }
        return this.getFilteredSortedAndTemporaryRecords({
            cleanMetadata,
            group,
            level,
            pageNumber: 0,
            pageSize: Number.POSITIVE_INFINITY,
            parentId,
            tableFieldProperties,
        });
    }
    takeFilterSnapshot({ level = 0, parentId, snapshot, }) {
        setWith(this.filterSnapshot, [level, parentId ?? 0], snapshot, Object);
    }
    restoreFilterSnapshot() {
        if (isNil(this.filterSnapshot)) {
            return;
        }
        this.filter = this.filterSnapshot;
    }
    getFetchServerRecordsOptions({ filters = [], group, level = 0, orderBy: order, parentId, rawFilter, searchText: text, selectedOptionsMenuItem, tableFieldProperties, }) {
        const orderBy = order ?? this.getOrderBy({ level, parentId });
        const state = getStore().getState();
        const pageDefinition = getPageDefinitionFromState(this.screenId, state);
        const applicableFilters = [];
        // Resolve functional developer defined filters
        if (tableFieldProperties.filter) {
            const screenDefinition = getPageDefinitionFromState(this.screenId);
            // In nested context like reference lookup we only send rowValue as argument, but in normal fields like a table we send value and rowValue for filters, more details at lib/component/field/traits.ts interface.
            if (this.fieldType === CollectionFieldTypes.LOOKUP_DIALOG) {
                const filters = convertFilterDecoratorToGraphQLFilter(screenDefinition, tableFieldProperties.filter, this.recordContext);
                if (filters) {
                    applicableFilters.push(filters);
                }
            }
            else {
                applicableFilters.push(resolveByValue({
                    propertyValue: tableFieldProperties.filter,
                    rowValue: null,
                    fieldValue: null,
                    screenId: this.screenId,
                    skipHexFormat: true,
                }));
            }
        }
        // Resolve user defined filers
        if (filters && filters.length > 0) {
            applicableFilters.push(getGraphQLFilter(filters, getTypedNestedFields(this.screenId, this.getNode(0), getNestedFieldsFromProperties(tableFieldProperties, level), this.nodeTypes)));
        }
        if (rawFilter) {
            applicableFilters.push(rawFilter);
        }
        // Always take a snapshot even with an empty text search
        this.takeFilterSnapshot({ level, parentId, snapshot: mergeGraphQLFilters(applicableFilters) || {} });
        if (text) {
            const fieldProperties = tableFieldProperties;
            applicableFilters.push(buildSearchBoxFilter(fieldProperties, state.nodeTypes, this.locale, this.fieldType, text));
        }
        let hasOptionMenuItemChanged = false;
        if (!isEqual(this.activeOptionsMenuItem, selectedOptionsMenuItem)) {
            this.activeOptionsMenuItem = selectedOptionsMenuItem;
            hasOptionMenuItemChanged = true;
        }
        if (this.activeOptionsMenuItem) {
            applicableFilters.push(this.activeOptionsMenuItem.graphQLFilter);
        }
        // Merge functional developer and end-user filters.
        const combinedFilters = (mergeGraphQLFilters(applicableFilters) || {});
        const hasSortingChanged = !isEqual(this.getOrderBy({ level, parentId }), orderBy);
        const hasFilterChanged = !isEqual(this.getFilter({ level, parentId }), combinedFilters);
        const shouldRefetch = this.forceRefetch ||
            group !== undefined ||
            hasOptionMenuItemChanged ||
            hasSortingChanged ||
            hasFilterChanged;
        setWith(this.orderBy, [level, parentId ?? 0], orderBy, Object);
        setWith(this.filter, [level, parentId ?? 0], combinedFilters, Object);
        /**
         * When sorting or filtering changes, we need to delete all groups because we will fetch new data from the server.
         * It's not enough to just merge server data with existing groups because all filters apply to their children,
         * which are lazy loaded. It would be then impossible for us to know which groups are relevant for the current
         * page. On the other hand we need to keep groups in case a new page is being fetched.
         */
        if (hasFilterChanged && group !== undefined) {
            this.db
                .findAll({ where: { __isGroup: { $eq: true } } })
                .value({ cleanMetadata: false, cleanDatabaseMeta: false })
                .forEach(r => {
                this.db.remove({ data: r });
            });
        }
        return {
            pageDefinition,
            shouldRefetch,
        };
    }
    getFilteredSortedAndTemporaryRecords({ cleanMetadata = true, group, level = 0, pageNumber = 0, pageSize = 20, parentId, tableFieldProperties, }) {
        setWith(this.lastFetchedPage, [level, parentId ?? 0], 0, Object);
        const temporaryRecords = resolveByValue({
            skipHexFormat: true,
            screenId: this.screenId,
            propertyValue: tableFieldProperties.additionalLookupRecords,
            rowValue: null,
            fieldValue: null,
        });
        return this.getFilteredSortedRecords({
            pageSize,
            pageNumber,
            group,
            parentId,
            level,
            cleanMetadata,
            temporaryRecords,
        });
    }
    subscribeForValueChanges(callback, isUncommitted = false) {
        const subscriptions = isUncommitted ? this.valueChangeUncommittedSubscribers : this.valueChangeSubscribers;
        subscriptions.push(callback);
        return () => {
            const index = subscriptions.indexOf(callback);
            if (index !== -1) {
                subscriptions.splice(index, 1);
            }
        };
    }
    subscribeForValidityChanges(callback, isUncommitted = false) {
        const subscriptions = isUncommitted
            ? this.validityChangeUncommittedSubscribers
            : this.validityChangeSubscribers;
        subscriptions.push(callback);
        return () => {
            const index = subscriptions.indexOf(callback);
            if (index !== -1) {
                subscriptions.splice(index, 1);
            }
        };
    }
    getIdPathToNestedRecord(id, level) {
        return this.db.getIdPathToRecordByIdAndLevel({ id, level });
    }
    /**
     *
     * @param forceValidate If true, all records will be validated, otherwise only dirty records
     * @returns
     */
    async validate(forceValidate) {
        const columnDefinition = this.getColumnDefinitions(0);
        const revalidateColumns = forceValidate ? columnDefinition.map(c => c.properties.bind) : undefined;
        const validationResult = await Promise.all(this.db
            .findAll({
            where: {
                ...(forceValidate ? {} : { __dirty: { $eq: true } }),
                __action: { $nin: [RecordActionType.REMOVED, RecordActionType.TEMPORARY] },
            },
        })
            .value({ cleanMetadata: false, cleanDatabaseMeta: false })
            .map(r => this.runValidationOnRecord({ recordData: r, columnsToRevalidate: revalidateColumns })));
        return validationResult
            .map(r => Object.values(r))
            .reduce((prevValue, value) => [...prevValue, ...value], []);
    }
    /**
     * This function adds external validation errors directly to the collection without running the client side rules. It is used to push server generated
     * errors to the records. Listener toast events are batched together and executed at the end to optimize performance and reduce the number of
     * rendering cycles.
     */
    async addValidationErrors({ validationErrors, shouldNotifySubscribers = false, }) {
        const remappedValidationErrors = validationErrors.reduce((prevValue, v) => {
            if (!v.recordId) {
                return prevValue;
            }
            const key = this.db.getCompositeKey({ _id: v.recordId, __level: v.level });
            if (prevValue[key]) {
                prevValue[key].push(v);
                return prevValue;
            }
            return { ...prevValue, [key]: [v] };
        }, {});
        // Get all records that we might need to check: records that currently invalid by server side validation errors and new incoming server side errors
        const recordCompositeKeysToCheck = uniq([
            ...objectKeys(remappedValidationErrors),
            ...this.db.getAllServerRecordWithServerSideValidationIssues(),
        ]);
        const records = recordCompositeKeysToCheck
            .map(k => ({ ...this.db.getIdAndLevelFromCompositeKey(k), compositeKey: k }))
            .map(({ _id: id, __level: level, compositeKey }) => ({
            record: this.getRawRecord({
                cleanMetadata: false,
                id,
                includeUnloaded: true,
                level,
            }),
            id,
            level,
            compositeKey,
        }));
        const [existingRecords, unloadedRecords] = partition(records, r => r.record !== null);
        unloadedRecords.forEach(({ compositeKey, id, level }) => {
            const incomingServerSideValidationError = remappedValidationErrors[compositeKey];
            /**
             * If no record data is set, a small "unloaded" placeholder record is created with the validation error
             *  */
            const validationState = incomingServerSideValidationError.reduce((prevRecord, v) => {
                prevRecord[v.columnId || `__rowError_${v.recordId}`] = v;
                return prevRecord;
            }, {});
            const newRecord = {
                _id: id,
                __unloaded: true,
                __validationState: validationState,
            };
            this.db.addRecord({
                action: 'none',
                data: newRecord,
                level,
                cleanMetadata: false,
            });
        });
        const existingInvalidRecords = existingRecords.reduce((acc, { record: recordData, compositeKey }) => {
            const incomingServerSideValidationError = remappedValidationErrors[compositeKey];
            let currentValidationState = { ...(recordData.__validationState || {}) };
            // Remove previous server side errors
            objectKeys(currentValidationState).forEach(s => {
                if (currentValidationState[s].validationRule.startsWith(SERVER_VALIDATION_RULE_PREFIX)) {
                    delete currentValidationState[s];
                }
            });
            // Check if there is any new error messages sent by the server and add it to the validation object
            if (incomingServerSideValidationError && incomingServerSideValidationError.length) {
                currentValidationState = incomingServerSideValidationError.reduce((prevRecord, v) => {
                    prevRecord[v.columnId || `__rowError_${v.recordId}`] = v;
                    return prevRecord;
                }, currentValidationState);
            }
            // If the validation result is an empty object, we normalize to to undefined
            const normalizedCurrentValidationState = isEmpty(currentValidationState)
                ? undefined
                : currentValidationState;
            const normalizedPreviousValidationState = isEmpty(recordData.__validationState)
                ? undefined
                : recordData.__validationState;
            // Only trigger event if the validation state of the record changed compared to its previous state.
            if (!isEqual(normalizedCurrentValidationState, normalizedPreviousValidationState)) {
                // Get a fresh copy of the record from the database because the values could have changed while the validation was running
                const updatedRecordData = {
                    ...recordData,
                    __validationState: normalizedCurrentValidationState,
                };
                acc.push(this.db.update({
                    data: updatedRecordData,
                    beforeUpdate: record => this.formatRecordValue(this.repackRecordData(record), undefined, true),
                    cleanMetadata: false,
                }));
            }
            return acc;
        }, []);
        // Notify subscribers about the updated state.
        if (shouldNotifySubscribers) {
            await this.validateField();
            this.notifyValidationSubscribers(existingInvalidRecords);
        }
    }
    getMergedValue({ recordId, columnId, value, level = 0, toBeMarkedAsDirty, addToDirty, isUncommitted, }) {
        const record = this.db.findOne({
            id: recordId,
            level,
            cleanMetadata: false,
            isUncommitted,
        });
        if (!record) {
            throw new Error(`Could not set cell value: ${JSON.stringify({ recordId, columnId, level, value })}`);
        }
        const delta = set({}, columnId, value);
        return this.getUpdatedRecord({ previous: record, newData: delta, toBeMarkedAsDirty, addToDirty });
    }
    async setCellValue({ recordId, columnId, value, level = 0, isOrganicChange = false, toBeMarkedAsDirty, addToDirty, isUncommitted, shouldFetchDefault = false, }) {
        let updatedData = this.getMergedValue({
            recordId,
            columnId,
            value,
            level,
            toBeMarkedAsDirty,
            addToDirty,
            isUncommitted,
        });
        this.db.update({
            data: updatedData,
            beforeUpdate: toBeUpdated => this.formatRecordValue(this.repackRecordData(toBeUpdated), undefined, !isOrganicChange, level),
            afterUpdate: () => {
                if (updatedData.__phantom) {
                    updatedData.__forceRowUpdate = true;
                }
                this.notifyValueChangeSubscribers(RecordActionType.MODIFIED, updatedData, isUncommitted);
                if (isOrganicChange) {
                    this.runValidationOnRecord({ recordData: updatedData, changedColumnId: columnId, isUncommitted });
                }
            },
        });
        // Only fetch if column has the fetchDefault and is a new record.
        if (shouldFetchDefault && Number(recordId) < 0) {
            const defaultValues = await fetchNestedDefaultValues({
                screenId: this.screenId,
                elementId: this.elementId,
                recordId,
                data: Array.from(updatedData.__dirtyColumns || []).reduce((acc, dirtyColumn) => {
                    acc[dirtyColumn] = updatedData[dirtyColumn];
                    return acc;
                }, {}),
                isNewRow: true,
            });
            if (defaultValues.nestedDefaults) {
                delete defaultValues.nestedDefaults._id;
                updatedData = { ...updatedData, ...defaultValues.nestedDefaults };
            }
            // We only update the DB if the record is still present in the dataset after the default value fetching
            if (this.db.findOne({ id: recordId, level, isUncommitted })) {
                this.db.update({
                    data: updatedData,
                    beforeUpdate: toBeUpdated => this.formatRecordValue(this.repackRecordData(toBeUpdated), undefined, !isOrganicChange, level),
                    afterUpdate: () => {
                        this.notifyValueChangeSubscribers(RecordActionType.MODIFIED, updatedData, isUncommitted);
                        if (isOrganicChange) {
                            this.runValidationOnRecord({
                                recordData: updatedData,
                                changedColumnId: columnId,
                                isUncommitted,
                            });
                        }
                    },
                });
            }
        }
        return updatedData;
    }
    addOrUpdateRecordValue({ recordData, shouldNotifySubscribers = true, level, parentId, includeUnloaded = false, }) {
        if (recordData._id && this.db.findOne({ id: String(recordData._id), level, includeUnloaded })) {
            this.setRecordValue({
                recordData: recordData,
                shouldNotifySubscribers,
                level,
                upsert: true,
            });
            return this.getRecordWithChildren({
                id: String(recordData._id),
                level,
            });
        }
        if (recordData._id &&
            this.db.findOne({ id: String(recordData._id), level, includeUnloaded, isUncommitted: true })) {
            this.setRecordValue({
                recordData: recordData,
                shouldNotifySubscribers,
                level,
                upsert: true,
                isUncommitted: true,
            });
            return this.getRecordWithChildren({
                id: String(recordData._id),
                level,
                isUncommitted: true,
            });
        }
        if (level !== undefined && level !== 0 && parentId === undefined) {
            throw new Error(`Cannot create a record of type '${this.getNode(level)}' without a 'parentId': ${{
                level,
                recordValue: recordData,
            }}`);
        }
        return this.addRecord({
            recordData,
            level,
            shouldNotifySubscribers,
            parentId,
        });
    }
    setRecordValue({ recordData, shouldNotifySubscribers = true, level = 0, upsert = false, toBeMarkedAsDirty, resetRecordAction = false, isOrganicChange = false, changedColumnId, isUncommitted, }) {
        const childProperty = this.levelMap?.[level] || '';
        const { [childProperty]: childData, ...updateData } = recordData;
        const { _id: parentId } = this.findAndUpdate({
            recordData: updateData,
            level,
            shouldNotifySubscribers,
            upsert,
            toBeMarkedAsDirty,
            resetRecordAction,
            changedColumnId,
            isOrganicChange,
            isUncommitted,
        });
        get(recordData, String(childProperty), []).forEach((child) => {
            this.setRecordValue({
                recordData: { ...child, __parentId: parentId },
                shouldNotifySubscribers,
                level: level + 1,
                upsert,
                toBeMarkedAsDirty,
                isOrganicChange,
                changedColumnId,
            });
        });
    }
    addNestedLevelRecursive({ recordData, action = RecordActionType.ADDED, dirty = true, level, parentId, shouldNotifySubscribers, }) {
        if (this.levelMap === undefined) {
            return;
        }
        const childProperty = this.levelMap[level];
        const children = recordData[childProperty] || [];
        children.forEach(childRecordData => {
            // eslint-disable-next-line @typescript-eslint/naming-convention
            const { [this.levelMap?.[level + 1] ?? '']: _, ...data } = childRecordData;
            const newChildRecord = this.db.addRecord({
                data: { ...data, __parentId: parentId },
                level: level + 1,
                beforeInsert: record => this.formatRecordValue(this.repackRecordData(record), undefined, true, level),
                afterInsert: ({ action, record }) => shouldNotifySubscribers && this.notifyValueChangeSubscribers(action, record),
                cleanMetadata: false,
                action,
                dirty,
            });
            if (this.levelMap &&
                level + 1 < objectKeys(this.levelMap).length &&
                this.levelMap[level + 1] !== undefined) {
                this.addNestedLevelRecursive({
                    recordData: childRecordData,
                    level: level + 1,
                    shouldNotifySubscribers,
                    parentId: newChildRecord._id,
                    dirty,
                    action,
                });
            }
        });
    }
    addRecord({ recordData, level, shouldNotifySubscribers = true, parentId, cleanMetadata = true, action = RecordActionType.ADDED, isUncommitted = false, dirty = true, }) {
        if (level === undefined || this.levelMap === undefined) {
            return this.db.addRecord({
                data: recordData,
                afterInsert: ({ action: dropdownAction, record }) => shouldNotifySubscribers && this.notifyValueChangeSubscribers(dropdownAction, record, isUncommitted),
                level,
                beforeInsert: record => this.formatRecordValue(this.repackRecordData(record)),
                cleanMetadata,
                action,
                isUncommitted,
                dirty,
            });
        }
        // eslint-disable-next-line @typescript-eslint/naming-convention
        const { [this.levelMap?.[level] ?? '']: _, ...data } = recordData;
        const newRecord = this.db.addRecord({
            data: data,
            level,
            beforeInsert: record => this.formatRecordValue(this.repackRecordData(record)),
            afterInsert: ({ action: insertAction, record }) => shouldNotifySubscribers && this.notifyValueChangeSubscribers(insertAction, record),
            cleanMetadata: false,
            parentId,
            isUncommitted,
            dirty,
        });
        this.addNestedLevelRecursive({
            recordData: recordData,
            level,
            shouldNotifySubscribers,
            parentId: newRecord._id,
        });
        return this.getRecordWithChildren({
            id: newRecord._id,
            level,
            isUncommitted,
            cleanMetadata,
        });
    }
    removeRecord({ recordId, shouldNotifySubscribers = true, level, includeUnloaded = false, }) {
        const record = this.db.findOne({ id: recordId, level, cleanMetadata: false, includeUnloaded });
        if (!record) {
            throw new Error(`No record found with ID '${recordId}' in the collection value, therefore we cannot execute the removal. Are you sure you use a valid ID?`);
        }
        if (record.__action === RecordActionType.ADDED) {
            // If this record has never been saved to the database, we don't need to keep it.
            this.db.remove({
                data: record,
                afterRemove: () => {
                    this.db
                        .findAll({ level: (level || 0) + 1, parentId: recordId })
                        .value({ cleanMetadata: false })
                        .forEach(r => this.removeRecord({ recordId: r._id, shouldNotifySubscribers: false, level: r.__level }));
                    if (shouldNotifySubscribers) {
                        this.notifyValueChangeSubscribers(RecordActionType.REMOVED, record);
                    }
                },
            });
        }
        else {
            const updatedData = {
                ...record,
                __dirty: true,
                __validationState: undefined,
                __action: RecordActionType.REMOVED,
            };
            this.db.update({
                data: updatedData,
                beforeUpdate: toBeUpdated => this.formatRecordValue(this.repackRecordData(toBeUpdated), undefined, true, level),
                afterUpdate: () => {
                    this.db
                        .findAll({ level: (level || 0) + 1, parentId: recordId })
                        .value({ cleanMetadata: false })
                        .forEach(r => this.removeRecord({ recordId: r._id, shouldNotifySubscribers: false, level: r.__level }));
                    if (shouldNotifySubscribers) {
                        this.notifyValueChangeSubscribers(RecordActionType.REMOVED, updatedData);
                    }
                },
            });
        }
        dispatchUpdateNestedFieldValidationErrorsForRecord({
            screenId: this.screenId,
            elementId: this.elementId,
            recordId,
            level,
        });
        dispatchCloseSideBar({ screenId: this.screenId, elementId: this.elementId, recordId });
        this.notifyValidationSubscribers([record]);
    }
    getData({ cleanMetadata = true, includePhantom, inGroup = false, isUncommitted, level = 0, limit = 20, noLimit = false, parentId, temporaryRecords, where = transformToLokiJsFilter(this.getInternalFilter({ level, parentId })), } = {}) {
        return this.db.withTemporaryRecords(db => {
            let data = db
                .findAll({
                where: {
                    __action: { $ne: RecordActionType.REMOVED },
                    ...(this.sublevelProperty ? {} : { __isGroup: { $ne: !inGroup } }),
                    ...where,
                },
                level,
                parentId,
                includePhantom,
                isUncommitted,
            })
                .sort({
                orderBy: this.getOrderBy({ level, parentId }),
                columnDefinitions: this.getColumnDefinitions(level),
            });
            if (!noLimit) {
                data = data.take(limit);
            }
            return data.value({ cleanMetadata });
        }, temporaryRecords);
    }
    getAllDataAsTree({ excludeEmptyChildCollections = false, cleanInputTypes = true, } = {}) {
        const records = this.db.findAll({ level: null, parentId: null }).value({ cleanMetadata: false });
        return this.getTreeFromNormalizedCollection({
            records,
            includeActions: false,
            excludeEmptyChildCollections,
            cleanInputTypes,
        });
    }
    /** Creates a change dataset understood by the server */
    getChangedRecordsAsTree({ excludeEmptyChildCollections = true, } = {}) {
        const records = this.db
            .findAll({ where: { __action: { $exists: true } }, level: null, parentId: null })
            .value({ cleanMetadata: false });
        return this.getTreeFromNormalizedCollection({
            records,
            excludeEmptyChildCollections,
            removeNegativeId: true,
        });
    }
    /** Creates a change dataset understood by the server */
    getNormalizedChangedRecords(isRequestingDefaults = false) {
        return this.db
            .findAll({ where: { __action: { $exists: true } }, level: null, parentId: null })
            .value({ cleanMetadata: false })
            .map(r => {
            return new CollectionDataRow(r, this.columnDefinitions, this.nodes, this.nodeTypes, {
                removeNegativeId: false,
                includeActions: true,
                isRequestingDefaults,
                isTree: !!this.sublevelProperty,
            }).getChangeset();
        });
    }
    /** Gets all new records */
    getNewRecords() {
        return this.db
            .findAll({ where: { __action: RecordActionType.ADDED }, level: null, parentId: null })
            .value({ cleanMetadata: false })
            .map(r => {
            return new CollectionDataRow(r, this.columnDefinitions, this.nodes, this.nodeTypes, {
                removeNegativeId: false,
                includeActions: false,
                isRequestingDefaults: false,
                isTree: !!this.sublevelProperty,
            }).getChangeset();
        });
    }
    async refreshRecord({ recordId, level = 0, skipUpdate = false, }) {
        if (this.isTransient) {
            throw new Error('Transient collections cannot be refreshed.');
        }
        const pageDefinition = getPageDefinitionFromState(this.screenId);
        const nodeName = this.getNode(level);
        const nestedFields = this.getColumnDefinitions(level);
        const result = await fetchCollectionRecord(pageDefinition, nestedFields, nodeName, recordId);
        if (!result) {
            return null;
        }
        const recordValue = splitValueToMergedValue(this.repackRecordData(result));
        if (!skipUpdate) {
            this.setRecordValue({
                recordData: recordValue,
                shouldNotifySubscribers: true,
                level,
                upsert: false,
                toBeMarkedAsDirty: [],
                resetRecordAction: true,
            });
        }
        return splitValueToMergedValue(recordValue);
    }
    startRecordTransaction({ recordId, recordLevel }) {
        this.db.startRecordTransaction({ recordId, recordLevel });
    }
    commitRecord({ recordId, recordLevel, }) {
        const isNewRecord = this.db.commitRecord({ recordId, recordLevel });
        const updatedData = this.getRecordByIdAndLevel({ id: recordId, level: recordLevel });
        this.notifyValueChangeSubscribers(isNewRecord ? RecordActionType.ADDED : RecordActionType.MODIFIED, updatedData);
        if (isNewRecord) {
            const fieldProperties = getStore().getState().screenDefinitions[this.screenId].metadata
                .uiComponentProperties[this.elementId];
            if (fieldProperties.canAddNewLine && !fieldProperties.isPhantomRowDisabled) {
                setTimeout(() => {
                    this.createNewPhantomRow({ level: recordLevel, parentId: updatedData.__parentId });
                }, 500);
            }
        }
        return updatedData;
    }
    cancelRecordTransaction({ recordId, recordLevel }) {
        this.db.cancelRecordTransaction({ recordId, recordLevel });
    }
    // This method will retrieve the previous or next record in the collection and the server if the index is major than our collection and we have next page
    async getRecordByOffset({ offset, recordId, recordLevel, tableProperties, }) {
        const record = this.getRawRecord({ id: recordId, level: recordLevel, cleanMetadata: false });
        if (!record) {
            return null;
        }
        const level = record.__level;
        const parentId = record.__parentId;
        const records = this.getData({ level, parentId, noLimit: true });
        const recordIndex = records.findIndex(r => r._id === recordId);
        if (recordIndex === -1) {
            return null;
        }
        const newRecordIndex = recordIndex + offset;
        if (newRecordIndex >= 0 && newRecordIndex < records.length) {
            return records[newRecordIndex];
        }
        if (newRecordIndex >= records.length && this.hasNextPage && tableProperties) {
            const nextPage = await this.getPageWithCurrentQueryArguments({
                tableFieldProperties: tableProperties,
                pageSize: 20,
                pageNumber: get(this.lastFetchedPage, [level ?? 0, parentId ?? 0], -1) + 1,
                cursor: record.__cursor,
                level: recordLevel,
            });
            if (!nextPage) {
                return null;
            }
            return nextPage[0];
        }
        return null;
    }
    //  This method will retrieve the next record in the collection if it's the last one locally it will call the server to get more pages until the last page
    async getNextRecord({ recordId, recordLevel, tableProperties, }) {
        return this.getRecordByOffset({ offset: 1, recordId, recordLevel, tableProperties });
    }
    // This method will retrieve the previous record from the collection
    async getPreviousRecord({ recordId, recordLevel, }) {
        return this.getRecordByOffset({ offset: -1, recordId, recordLevel });
    }
    // This method will retrieve the parent node, parent id and current level of the provided node.
    getParentNodeAndParentIdFromChildNode({ node }) {
        const level = this.nodes.indexOf(node);
        if (level === -1 || level === 0) {
            return { childLevel: 0 };
        }
        const parentNode = this.getNode(level - 1);
        const parentRecord = this.db.findOne({
            level,
            cleanMetadata: false,
        });
        return {
            parentId: parentRecord.__parentId,
            parentNode,
            childLevel: level,
            parentBind: this.levelMap && this.levelMap[level - 1],
        };
    }
}
//# sourceMappingURL=collection-data-service.js.map