import { createUserClientSettings, updateUserClientSettings } from '../../service/metadata-service';
import { ActionType } from '../action-types';
import { navigationPanelId } from '../../component/container/navigation-panel/navigation-panel-types';
import { pageDialog } from '../../service/dialog-service';
import { fetchUserClientSettingsForElement, selectView, unselectView, } from '../../service/user-client-settings-service';
import { localize } from '../../service/i18n-service';
import { getNavigationPanelTablePropertiesFromPageDefinition, getPageDefinitionFromState, } from '../../utils/state-utils';
import { createNavigationPanelDefaultView } from '../../service/navigation-panel-service';
import { DEFAULT_VIEW_ID } from '../../utils/constants';
export const storeUserClientSettings = (screenId, elementId) => async (dispatch, getState) => {
    const state = getState();
    const view = state.screenDefinitions[screenId].userSettings[elementId]?.$current;
    await updateUserClientSettings(view);
    dispatch({
        type: ActionType.SetUserCustomizationSaved,
        value: {
            screenId,
            elementId,
            viewId: view._id,
        },
    });
};
export const saveTableUserClientSettingsForPageElement = (screenId, elementId) => async (dispatch, getState) => {
    // We do not want to autosave the navigation panel settings as it has its own view management system
    if (elementId === navigationPanelId) {
        return;
    }
    const state = getState();
    const view = state.screenDefinitions[screenId].userSettings[elementId]?.$current;
    let viewId = view._id;
    const viewWithoutFilters = {
        ...view,
        content: [
            {
                ...view.content[0],
                filter: undefined,
                optionsMenuItem: undefined,
                searchText: undefined,
                grouping: undefined,
            },
        ],
    };
    if (viewId && viewId !== DEFAULT_VIEW_ID) {
        await updateUserClientSettings(viewWithoutFilters);
    }
    else {
        viewId = await createUserClientSettings(screenId, elementId, viewWithoutFilters);
    }
    dispatch({
        type: ActionType.SetUserCustomizationSaved,
        value: {
            screenId,
            elementId,
            viewId,
        },
    });
};
export const setTableViewColumnHidden = (screenId, elementId, level, columnHidden) => (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.('tableVisibilityChange', {
        screenId,
        elementId,
    });
    dispatch({
        type: ActionType.SetTableViewColumnHidden,
        value: { screenId, elementId, level, columnHidden },
    });
    dispatch(saveTableUserClientSettingsForPageElement(screenId, elementId));
};
export const setTableViewColumnOrder = (screenId, elementId, level, columnOrder) => (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.('tableColumnOrderChange', {
        screenId,
        elementId,
    });
    dispatch({
        type: ActionType.SetTableViewColumnOrder,
        value: { screenId, elementId, level, columnOrder },
    });
    dispatch(saveTableUserClientSettingsForPageElement(screenId, elementId));
};
export const setTableViewFilter = (screenId, elementId, level, filter) => (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.('tableFiltered', {
        elementId,
        screenId,
    });
    dispatch({
        type: ActionType.SetTableViewFilter,
        value: { screenId, elementId, level, filter },
    });
};
export const setTableViewGrouping = (screenId, elementId, level, grouping) => (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.('tableGroupingChange', {
        elementId,
        screenId,
        item: grouping?.key,
    });
    dispatch({
        type: ActionType.SetTableViewGrouping,
        value: { screenId, elementId, level, grouping },
    });
};
export const setTableViewOptionMenuItem = (screenId, elementId, level, optionsMenuItem) => (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.('tableOptionsMenuChange', {
        elementId,
        item: optionsMenuItem?.title,
        screenId,
    });
    dispatch({
        type: ActionType.SetTableViewOptionsMenuItem,
        value: { screenId, elementId, level, optionsMenuItem },
    });
};
export const setTableViewOptionsMenuItemAndViewFilter = (screenId, elementId, level, optionsMenuItem, filter) => (dispatch) => {
    dispatch({
        type: ActionType.SetTableViewOptionsMenuItem,
        value: { screenId, elementId, level, optionsMenuItem, filter },
    });
};
export const setTableViewSearchText = (screenId, elementId, level, searchText) => (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.('tableQuickTextFiltered', {
        elementId,
        screenId,
    });
    dispatch({
        type: ActionType.SetTableViewSearchText,
        value: { screenId, elementId, level, searchText },
    });
};
export const clearNavigationPanelSearchText = (screenId) => (dispatch) => {
    dispatch({
        type: ActionType.ClearNavigationPanelSearchText,
        value: { screenId },
    });
};
export const setTableViewSortOrder = (screenId, elementId, level, sortOrder) => (dispatch) => {
    dispatch({
        type: ActionType.SetTableViewSortOrder,
        value: { screenId, elementId, level, sortOrder },
    });
    dispatch(saveTableUserClientSettingsForPageElement(screenId, elementId));
};
export const selectMainListView = (screenId, view) => async (dispatch) => {
    if (view?._id) {
        await dispatch({
            type: ActionType.SetElementUserSettings,
            value: { screenId, elementId: navigationPanelId, userSettings: view },
        });
        await selectView(screenId, navigationPanelId, view._id);
    }
};
export const unselectMainListView = (screenId) => async (dispatch, getState) => {
    const state = getState();
    const pageDefinition = getPageDefinitionFromState(screenId, state);
    const defaultTableProperties = getNavigationPanelTablePropertiesFromPageDefinition(pageDefinition, true);
    await dispatch({
        type: ActionType.SetElementUserSettings,
        value: {
            screenId,
            elementId: navigationPanelId,
            userSettings: createNavigationPanelDefaultView({
                screenId,
                defaultTableProperties,
                accessBindings: pageDefinition.accessBindings,
                dataTypes: state.dataTypes,
                nodeTypes: state.nodeTypes,
            }),
        },
    });
    await unselectView(screenId, navigationPanelId);
};
export const openCreateNewViewDialog = (screenId, elementId) => async (dispatch, getState) => {
    const state = getState();
    if (!state.clientUserSettingsEditPage) {
        return null;
    }
    const view = state.screenDefinitions[screenId]?.userSettings[navigationPanelId]?.$current;
    const isDefaultView = !view || view._id === DEFAULT_VIEW_ID;
    const result = await pageDialog({ dispatch, getState }, state.clientUserSettingsEditPage, { screenId, elementId }, {
        hasGreyBackground: true,
        resolveOnCancel: true,
        title: !isDefaultView
            ? localize('@sage/xtrem-ui/save-new-view-as', 'Save view as')
            : localize('@sage/xtrem-ui/create-a-view', 'Create a view'),
        subtitle: !isDefaultView
            ? localize('@sage/xtrem-ui/create-a-new-view', 'Create a new view')
            : localize('@sage/xtrem-ui/default-view-cannot-be-edited', 'Default view cannot be updated.'),
        values: {
            content: view?.content ? JSON.stringify(view.content) : '{}',
            screenId,
            elementId,
        },
    });
    if (!result?._id) {
        return null;
    }
    await selectView(screenId, elementId, result._id);
    return result._id;
};
export const openManageViewsDialog = (screenId, elementId) => async (dispatch, getState) => {
    const state = getState();
    if (state.clientUserSettingsListPage) {
        await pageDialog({ dispatch, getState }, state.clientUserSettingsListPage, { screenId, elementId }, {
            resolveOnCancel: true,
            hasGreyBackground: true,
        });
    }
    return fetchUserClientSettingsForElement(screenId, navigationPanelId);
};
//# sourceMappingURL=user-client-settings-actions.js.map