import { objectKeys } from '@sage/xtrem-shared';
import { cloneDeepWith, isEmpty, uniq } from 'lodash';
import { getImageUrlFromValue } from '../../component/field/image/image-utils';
import { GENERIC_PLATFORM_WIDGET_PREFIX, getGenericWidgetDefinition } from '../../dashboard/generic-widgets';
import { WidgetType } from '../../dashboard/widgets/abstract-widget';
import * as dashboardService from '../../service/dashboard-service';
import { resolveWidgetProperty } from '../../service/dashboard-service';
import { fetchEnumTranslations } from '../../service/metadata-service';
import * as screenLoaderService from '../../service/screen-loader-service';
import { arePlatformLiteralsMissing } from '../../utils/state-utils';
import { getArtifactDescription } from '../../utils/transformers';
import { ActionType } from '../action-types';
import { addTranslations } from './common-actions';
/**
 * Constructs the widget object depending its type. It checks if the widget is generic user configured or functional code provided, then constructs it.
 */
export const createWidgetObject = ({ contextVariables, dashboardId, group, settings, type, widgetConstructors, widgetId, }) => {
    const isGenericWidget = type.startsWith(GENERIC_PLATFORM_WIDGET_PREFIX);
    if (isGenericWidget) {
        if (!settings) {
            throw new Error('Generic widgets must have a settings object');
        }
        return getGenericWidgetDefinition(settings, dashboardId, widgetId, group, contextVariables);
    }
    const WidgetConstructor = widgetConstructors[type];
    if (!WidgetConstructor) {
        throw new Error(`No constructor found for widget: ${type}`);
    }
    WidgetConstructor.prototype.__id = widgetId;
    WidgetConstructor.prototype.__group = group;
    WidgetConstructor.prototype.__dashboardId = dashboardId;
    WidgetConstructor.prototype.__contextVariables = contextVariables;
    return new WidgetConstructor();
};
const loadDashboard = async (dashboard, group, contextVariables, forceRefetchData, dispatch, getState) => {
    const loadedWidgets = objectKeys(getState().dashboard.dashboardGroups[group].widgets);
    const filterWidgetsToLoad = (c) => forceRefetchData || loadedWidgets.indexOf(c._id) === -1;
    // We should not load widgets that we already have and generic user defined widgets which have the `@sage/xtrem-ui` prefix
    const filterWidgetsToFetch = (c) => filterWidgetsToLoad(c) && !c.type.startsWith(GENERIC_PLATFORM_WIDGET_PREFIX);
    const widgetsNeedLoading = dashboard ? uniq(dashboard.children.filter(filterWidgetsToFetch).map(c => c.type)) : [];
    const widgetConstructors = await screenLoaderService.fetchWidgetDefinitions(getState, dispatch, widgetsNeedLoading);
    if (!widgetConstructors) {
        return;
    }
    const enumsToLoad = dashboard?.children?.reduce((prevValue, currentValue) => {
        return uniq([...prevValue, ...(currentValue.settings?.usedEnums || [])]);
    }, []) || [];
    if (!isEmpty(enumsToLoad)) {
        const locale = getState().applicationContext?.locale || 'en-US';
        const enumTranslations = await fetchEnumTranslations(enumsToLoad, locale);
        dispatch(addTranslations(locale, enumTranslations));
    }
    const widgets = dashboard?.children.filter(filterWidgetsToLoad).map((item) => {
        const widgetId = String(item._id);
        const dashboardId = String(dashboard._id);
        const widgetObject = createWidgetObject({
            widgetConstructors,
            type: item.type,
            widgetId,
            dashboardId,
            group,
            contextVariables,
            settings: item.settings,
        });
        return {
            artifactName: item.type,
            _id: String(item._id),
            widgetObject,
            properties: widgetObject.constructor.prototype.__properties,
            widgetType: widgetObject.constructor.prototype.__type,
            runtimeSettings: {},
        };
    }) || [];
    dispatch({ type: ActionType.AddWidgets, value: { widgets, group } });
    if (dashboard) {
        dispatch({ type: ActionType.AddDashboard, value: { dashboard, group } });
    }
    const widgetPromises = widgets.map(w => {
        return dispatch(loadWidgetData({ widgetId: w._id, group }));
    });
    Promise.all(widgetPromises).then(() => {
        dispatch({ type: ActionType.SetLoadingDashboards, value: { isLoading: false, group } });
    });
};
export const refreshCurrentDashboard = (group, contextVariables) => async (dispatch, getState) => {
    const state = getState();
    const dashboard = Object.values(state.dashboard.dashboardGroups[group]?.dashboards || {}).find(d => d.isSelected);
    if (dashboard) {
        await loadDashboard(dashboard, group, contextVariables, true, dispatch, getState);
    }
};
export const fetchUserDashboardDefinition = (group, contextVariables) => async (dispatch, getState) => {
    dispatch({ type: ActionType.SetLoadingDashboards, value: { isLoading: true, group } });
    const state = getState();
    const locale = state.applicationContext?.locale || 'en-US';
    const shouldFetchPlatformLiterals = arePlatformLiteralsMissing(getState(), locale);
    const { dashboard, stringLiterals, availableDashboards, canEditDashboards } = await dashboardService.fetchCurrentDashboardDefinition(group, shouldFetchPlatformLiterals);
    if (shouldFetchPlatformLiterals) {
        dispatch(addTranslations(locale, stringLiterals));
    }
    await loadDashboard(dashboard, group, contextVariables, false, dispatch, getState);
    dispatch({
        type: ActionType.SetInitialDashboardInformation,
        value: { availableDashboards: availableDashboards || [], canEditDashboards, group },
    });
};
export const setSelectedDashboard = (id, group, contextVariables) => async (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.('dashboardTabSelected', { group });
    dispatch({ type: ActionType.SetLoadingDashboards, value: { isLoading: true, group } });
    const dashboard = await dashboardService.setSelectedDashboard(id, group);
    await loadDashboard(dashboard, group, contextVariables, false, dispatch, getState);
};
/**
 * Clone a dashboard based on its ID, then marks it automatically selected
 * @param idToClone
 * @returns
 */
export const cloneDashboard = (idToClone, group, contextVariables) => async (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.('dashboardCloned', { group });
    const { _id } = await dashboardService.cloneDashboard(idToClone);
    const dashboard = await dashboardService.setSelectedDashboard(_id, group);
    await loadDashboard(dashboard, group, contextVariables, false, dispatch, getState);
};
/**
 * Creates an empty dashboard, then marks it automatically selected
 * @param idToClone
 * @returns
 */
export const createEmptyDashboard = (group, contextVariables) => async (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.('emptyDashboardCreated', { group });
    const { _id } = await dashboardService.createEmptyDashboard(group);
    const dashboard = await dashboardService.setSelectedDashboard(_id, group);
    await loadDashboard(dashboard, group, contextVariables, false, dispatch, getState);
};
export const deleteDashboard = (dashboardId, group, contextVariables) => async (dispatch, getState) => {
    getState().applicationContext?.onTelemetryEvent?.('dashboardDeleted', { group });
    await dashboardService.deleteDashboard(dashboardId);
    dispatch({ type: ActionType.RemoveDashboard, value: { dashboardId, group } });
    dispatch(fetchUserDashboardDefinition(group, contextVariables));
};
export function getNewTableData({ widgetDefinition, data, }) {
    const tableProperties = widgetDefinition.properties;
    const widgetObject = cloneDeepWith(widgetDefinition.widgetObject, (_value, key, object) => {
        // Here we override the developer API so the new data can be accessed
        if (key === '_$' && object === widgetDefinition.widgetObject) {
            return { ...widgetDefinition.widgetObject.$, data };
        }
        return undefined;
    });
    const tableContent = typeof tableProperties.content === 'function'
        ? tableProperties.content.apply(widgetObject)
        : tableProperties.content;
    return (tableContent ?? []).map((r) => ({
        ...r,
        i: r._id,
        image: r.image ? getImageUrlFromValue(r.image) : undefined,
    }));
}
export const loadWidgetData = ({ widgetId, group, forceRefetch = false, queryArgs, dataMode = 'SET', skipUnset = false, }) => async (dispatch, getState) => {
    const state = getState();
    const widgetDefinition = state.dashboard.dashboardGroups[group].widgets[widgetId];
    const isTableWidget = widgetDefinition.widgetType === WidgetType.table;
    if (!skipUnset && (dataMode === 'SET' || !isTableWidget)) {
        dispatch({ type: ActionType.SetWidgetData, value: { widgetId, data: null, group } });
    }
    if (!widgetDefinition) {
        throw new Error(`Widget ${widgetId} is not in the state.`);
    }
    const artifactDescription = getArtifactDescription(widgetDefinition.artifactName);
    const pck = `${artifactDescription.vendor}/${artifactDescription.package}`;
    const version = state.applicationPackages[pck];
    if (dataMode !== 'ADD' && !skipUnset) {
        dispatch(SetWidgetLoading(widgetId, true));
    }
    const data = await dashboardService.fetchWidgetData({
        widgetDefinition,
        version,
        locale: state.applicationContext?.locale,
        forceRefetch,
        queryArgs,
        group,
    });
    if (dataMode !== 'ADD' && !skipUnset) {
        dispatch(SetWidgetLoading(widgetId, false));
    }
    if (widgetDefinition.widgetType === WidgetType.table) {
        const businessIcon = resolveWidgetProperty(widgetDefinition.properties.businessIcon, group, widgetDefinition.widgetObject || widgetDefinition._id);
        dispatch({
            type: ActionType.SetWidgetProperties,
            value: { widgetId, properties: { businessIcon }, group },
        });
    }
    if (dataMode === 'SET' || !isTableWidget) {
        dispatch({ type: ActionType.SetWidgetData, value: { widgetId, data, group } });
    }
    else {
        dispatch({ type: ActionType.AddWidgetData, value: { widgetId, data, group } });
    }
    if (isTableWidget) {
        return getNewTableData({ widgetDefinition, data, group });
    }
    return data;
};
export const setWidgetOptions = (dashboardId, widgetId, group, options) => ({
    type: ActionType.SetWidgetOptions,
    value: {
        dashboardId,
        widgetId,
        options,
        group,
    },
});
export const SetWidgetLoading = (widgetId, isLoading) => ({
    type: ActionType.SetWidgetLoading,
    value: { widgetId, isLoading },
});
export const setWidgetLoadingTimeoutId = (widgetId, timeoutId) => ({
    type: ActionType.SetWidgetLoadingTimeoutId,
    value: { widgetId, timeoutId },
});
export const resetWidgetLoadingTimeout = (widgetId) => ({
    type: ActionType.ResetWidgetLoadingTimeout,
    value: { widgetId },
});
export const removeDashboardGroup = (screenId) => ({
    type: ActionType.RemoveDashboardGroup,
    value: screenId,
});
//# sourceMappingURL=dashboard-actions.js.map