import { FilterableSelect, Option } from 'carbon-react/esm/components/select';
import AdvancedColorPicker from 'carbon-react/esm/components/advanced-color-picker';
import Box from 'carbon-react/esm/components/box';
import NumberComponent from 'carbon-react/esm/components/number';
import Typography from 'carbon-react/esm/components/typography';
import * as React from 'react';
import { fetchWidgetData, getWidgetsFromDashboardItems } from '../../service/dashboard-service';
import { localize } from '../../service/i18n-service';
import AsyncDragDropCanvasWrapper from '../async-drag-drop-canvas-wrapper';
import localeText from '../dashboard-strings';
import { getGenericWidgetArtifactName, getGenericWidgetDefinition } from '../generic-widgets';
import { WidgetType } from '../widgets/abstract-widget';
import { indicatorTileIconsSortedArray } from './indicator-tile-icons';
import { isChart, DEFAULT_CHART_COLORS } from './widget-editor-utils';
import Textbox from 'carbon-react/esm/components/textbox';
import { isEmpty, isNil, memoize } from 'lodash';
import { useDebounce } from 'usehooks-ts';
import { WidgetActionConfiguration } from './widget-action-configuration';
import { deepMerge, objectKeys } from '@sage/xtrem-shared';
import { queryPagesByNodeType } from '../../service/metadata-service';
import { PREVIEW_DASHBOARD_ID, PREVIEW_WIDGET_ID } from '../../utils/constants';
import { getNewTableData } from '../../redux/actions/dashboard-actions';
import { DASHBOARD_BREAKPOINTS } from '../dashboard-utils';
function PreviewDashboardConfigurationContent({ widgetDefinition, previewDashboardConfiguration, onBreakpointChange, }) {
    const json = React.useMemo(() => ({
        type: 'responsive',
        isHeaderHidden: true,
        isEditMode: false,
        locale: 'en-US',
        layout: {
            cols: { xxs: 1, xs: 1, sm: 1, md: 1, lg: 1 },
            breakpoints: DASHBOARD_BREAKPOINTS,
            rowHeight: widgetDefinition.type === 'INDICATOR_TILE' ? 120 : 400,
            width: 400,
            isDraggable: false,
        },
        widgets: previewDashboardConfiguration,
        stringLiterals: localeText(),
        onBreakpointChange,
    }), [onBreakpointChange, previewDashboardConfiguration, widgetDefinition.type]);
    return (React.createElement("div", { className: "e-layout-preview-dashboard-wrapper" },
        React.createElement(AsyncDragDropCanvasWrapper, { json: json })));
}
export function LayoutStep({ contextVariables, group, onWidgetDefinitionUpdated, stepIndex, widgetDefinition, }) {
    const [previewDashboardConfiguration, setPreviewDashboardConfiguration] = React.useState(null);
    const [maxNumAllowedValue, setMaxNumAllowedValue] = React.useState(String(widgetDefinition.aggregationsCount ?? 20));
    const [breakpoint, setBreakpoint] = React.useState('md');
    const [arePagesAvailableForNode, setPagesAvailableForNode] = React.useState(false);
    const [verticalAxisLabel, setVerticalAxisLabel] = React.useState(widgetDefinition.verticalAxisLabel);
    const [horizontalAxisLabel, setHorizontalAxisLabel] = React.useState(widgetDefinition.horizontalAxisLabel);
    const [maxNumAllowed, setMaxNumAllowed] = React.useState(widgetDefinition.aggregationsCount ?? 20);
    const first = useDebounce(maxNumAllowed, 500);
    const vAxisLabel = useDebounce(verticalAxisLabel, 500);
    const hAxisLabel = useDebounce(horizontalAxisLabel, 500);
    const [maxNumError, setMaxNumError] = React.useState('');
    const [chartColors, setChartColors] = React.useState((widgetDefinition.aggregations ?? []).reduce((acc, aggregation, index) => {
        acc[`${aggregation.id}-${aggregation.groupingMethod}`] = {
            color: aggregation.color ?? DEFAULT_CHART_COLORS[index].value,
            isOpen: false,
        };
        return acc;
    }, {}));
    const [subtitle, setSubtitle] = React.useState(widgetDefinition.subtitle || '');
    const icons = indicatorTileIconsSortedArray();
    React.useEffect(() => {
        if (widgetDefinition.node) {
            queryPagesByNodeType(widgetDefinition.node).then(r => setPagesAvailableForNode(!isEmpty(r)));
        }
    }, [widgetDefinition.node]);
    React.useEffect(() => {
        if (isChart(widgetDefinition)) {
            onWidgetDefinitionUpdated({
                ...widgetDefinition,
                aggregationsCount: maxNumAllowed,
            });
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [first, widgetDefinition]);
    React.useEffect(() => {
        if (isChart(widgetDefinition)) {
            onWidgetDefinitionUpdated({
                ...widgetDefinition,
                horizontalAxisLabel: hAxisLabel,
            });
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [hAxisLabel, widgetDefinition]);
    React.useEffect(() => {
        if (isChart(widgetDefinition)) {
            onWidgetDefinitionUpdated({
                ...widgetDefinition,
                verticalAxisLabel: vAxisLabel,
            });
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [vAxisLabel, widgetDefinition]);
    const [decimalDigitsValidationMessage, setDecimalDigitsValidationMessage] = React.useState(undefined);
    React.useEffect(() => {
        const widgetObject = getGenericWidgetDefinition(widgetDefinition, PREVIEW_DASHBOARD_ID, PREVIEW_WIDGET_ID, group, contextVariables, true);
        const dashboardWidgetDefinition = {
            _id: PREVIEW_WIDGET_ID,
            artifactName: getGenericWidgetArtifactName(widgetDefinition),
            properties: widgetObject.constructor.prototype.__properties,
            widgetObject,
            widgetType: widgetObject.constructor.prototype.__type,
            options: {},
        };
        fetchWidgetData({
            group,
            widgetDefinition: dashboardWidgetDefinition,
            queryArgs: isChart(widgetDefinition) ? { first } : undefined,
        }).then(data => {
            dashboardWidgetDefinition.widgetObject.constructor.prototype.__data = data;
            const isTableWidget = dashboardWidgetDefinition.widgetType === WidgetType.table;
            const w = dashboardWidgetDefinition.widgetType === WidgetType.table
                ? Math.max(Math.floor(objectKeys(dashboardWidgetDefinition.properties.rowDefinition).length / 2), 1)
                : 1;
            const widgets = getWidgetsFromDashboardItems({
                dashboardId: PREVIEW_DASHBOARD_ID,
                group,
                children: [
                    {
                        _id: PREVIEW_WIDGET_ID,
                        settings: {},
                        positions: [
                            { x: 0, y: 0, breakpoint: 'xxs', h: 1, w },
                            { x: 0, y: 0, breakpoint: 'xs', h: 1, w },
                            { x: 0, y: 0, breakpoint: 'sm', h: 1, w },
                            { x: 0, y: 0, breakpoint: 'md', h: 1, w },
                            { x: 0, y: 0, breakpoint: 'lg', h: 1, w },
                        ],
                        type: getGenericWidgetArtifactName(widgetDefinition),
                    },
                ],
                widgets: {
                    [PREVIEW_WIDGET_ID]: { ...dashboardWidgetDefinition, data, isLoading: false },
                },
                breakpoint,
            });
            setPreviewDashboardConfiguration(isTableWidget
                ? widgets.map(w => ({
                    ...w,
                    data: {
                        ...w.data,
                        loadMoreRows: memoize(async (loadMoreParams = {
                            startIndex: 0,
                            stopIndex: 20,
                            pageCount: 1,
                            pageSize: 20,
                            first: 20,
                        }) => {
                            const newPage = await fetchWidgetData({
                                widgetDefinition: dashboardWidgetDefinition,
                                version: undefined,
                                queryArgs: loadMoreParams,
                                group,
                            });
                            const newData = getNewTableData({
                                widgetDefinition: dashboardWidgetDefinition,
                                data: newPage,
                                group,
                            });
                            const mergedData = deepMerge(dashboardWidgetDefinition.widgetObject.constructor.prototype.__data, newPage);
                            dashboardWidgetDefinition.widgetObject.constructor.prototype.__data = mergedData;
                            setPreviewDashboardConfiguration(currentConfig => {
                                return currentConfig
                                    ? [
                                        {
                                            ...currentConfig[0],
                                            data: {
                                                ...currentConfig[0].data,
                                                data: [...currentConfig[0].data.data, ...newData],
                                            },
                                        },
                                    ]
                                    : currentConfig;
                            });
                            return newData;
                        }),
                    },
                }))
                : widgets);
        });
    }, [widgetDefinition, setPreviewDashboardConfiguration, first, group, contextVariables, breakpoint]);
    const onIconChanged = React.useCallback((ev) => {
        onWidgetDefinitionUpdated({ ...widgetDefinition, icon: ev.target.value });
    }, [onWidgetDefinitionUpdated, widgetDefinition]);
    const onSubtitleChanged = React.useCallback((ev) => {
        setSubtitle(ev.target.value);
    }, [setSubtitle]);
    const onBreakpointChange = React.useCallback(newBreakpoint => {
        setBreakpoint(newBreakpoint);
    }, []);
    const onSubtitleBlur = React.useCallback(() => {
        onWidgetDefinitionUpdated({ ...widgetDefinition, subtitle });
    }, [subtitle, widgetDefinition, onWidgetDefinitionUpdated]);
    const onDecimalDigitsChanged = React.useCallback((ev) => {
        if (!ev.target.value) {
            setDecimalDigitsValidationMessage(localize('@sage/xtrem-ui/widget-editor-action-decimal-digits-mandatory', 'You need to enter a number.'));
        }
        else {
            setDecimalDigitsValidationMessage(undefined);
        }
        const decimalDigits = parseInt(ev.target.value, 10);
        onWidgetDefinitionUpdated({
            ...widgetDefinition,
            decimalDigits: !Number.isNaN(decimalDigits) ? decimalDigits : undefined,
        });
    }, [onWidgetDefinitionUpdated, widgetDefinition, setDecimalDigitsValidationMessage]);
    const shouldRenderConfigColumn = arePagesAvailableForNode || widgetDefinition.type !== 'TABLE';
    return (React.createElement("div", { className: "e-widget-editor-layout-screen" },
        React.createElement(Typography, { variant: "h2", "data-testid": "e-widget-editor-step-title" }, localize('@sage/xtrem-ui/widget-editor-layout-step-title', '{{stepIndex}}. Create your layout', {
            stepIndex,
        })),
        React.createElement("div", { className: "e-layout-step e-widget-editor-section" },
            shouldRenderConfigColumn && (React.createElement("div", { className: "e-layout-settings-container", "data-testid": "e-layout-settings-container" },
                isChart(widgetDefinition) && (React.createElement(React.Fragment, null,
                    React.createElement("div", { style: { paddingBottom: '24px' } },
                        React.createElement(Typography, { mb: "16px", variant: "h4", fontWeight: "600", fontSize: "16px", lineHeight: "150%", p: 0 }, localize('@sage/xtrem-ui/series-options', 'Series options')),
                        (widgetDefinition.aggregations ?? []).map((aggregation, index) => {
                            return (React.createElement(Box, { key: aggregation.label, display: "flex", alignItems: "center", justifyContent: "space-between", paddingY: 1, boxShadow: "boxShadow010" },
                                React.createElement(Typography, { variant: "p", fontWeight: "400", fontSize: "14px", lineHeight: "150%", m: 0, p: 0 }, aggregation.label),
                                React.createElement(AdvancedColorPicker, { margin: 0, name: aggregation.label, availableColors: DEFAULT_CHART_COLORS, defaultColor: DEFAULT_CHART_COLORS[index]?.value, selectedColor: chartColors[`${aggregation.id}-${aggregation.groupingMethod}`]
                                        .color, onChange: (evt) => {
                                        const color = evt.target.value;
                                        setChartColors(c => {
                                            return {
                                                ...c,
                                                [`${aggregation.id}-${aggregation.groupingMethod}`]: {
                                                    isOpen: false,
                                                    color,
                                                },
                                            };
                                        });
                                        onWidgetDefinitionUpdated({
                                            ...widgetDefinition,
                                            aggregations: (widgetDefinition.aggregations ?? []).map(a => {
                                                if (a.id === aggregation.id) {
                                                    return { ...a, color };
                                                }
                                                return a;
                                            }),
                                        });
                                    }, onOpen: () => {
                                        setChartColors(c => {
                                            return {
                                                ...c,
                                                [`${aggregation.id}-${aggregation.groupingMethod}`]: {
                                                    ...c[`${aggregation.id}-${aggregation.groupingMethod}`],
                                                    isOpen: true,
                                                },
                                            };
                                        });
                                    }, onClose: () => {
                                        setChartColors(c => {
                                            return {
                                                ...c,
                                                [`${aggregation.id}-${aggregation.groupingMethod}`]: {
                                                    ...c[`${aggregation.id}-${aggregation.groupingMethod}`],
                                                    isOpen: false,
                                                },
                                            };
                                        });
                                    }, open: chartColors[`${aggregation.id}-${aggregation.groupingMethod}`]
                                        .isOpen })));
                        }),
                        React.createElement(Textbox, { my: "16px", name: "max-num-allowed", "data-testid": "e-widget-editor-layout-max-num-allowed", label: localize('@sage/xtrem-ui/widget-editor-max-num-of-values', 'Maximum number of values'), value: maxNumAllowedValue, error: maxNumError || undefined, onChange: ({ target: { value } }) => {
                                setMaxNumAllowedValue(value);
                                if (value.match(/^-?\d+$/)) {
                                    const num = Number(value);
                                    if (num < 1 || num > 365) {
                                        setMaxNumError(localize('@sage/xtrem-ui/must-be-between-1-and-365', 'Enter a number between 1 and 365'));
                                    }
                                    else {
                                        setMaxNumAllowed(num);
                                        setMaxNumError('');
                                    }
                                }
                                else {
                                    setMaxNumError(localize('@sage/xtrem-ui/must-be-a-number', 'You need to enter a number'));
                                }
                            } })),
                    React.createElement("div", { style: { paddingBottom: '24px' } },
                        React.createElement(Typography, { variant: "h4", fontWeight: "600", fontSize: "16px", lineHeight: "150%", m: 0, p: 0 }, localize('@sage/xtrem-ui/axes', 'Axes')),
                        React.createElement(Textbox, { mt: "16px", maxLength: 50, name: "vertical-axis-label", "data-testid": "e-widget-editor-layout-vertical-axis-label", label: localize('@sage/xtrem-ui/widget-editor-vertical-axis-label', 'Vertical axis label'), value: verticalAxisLabel, onChange: ({ target: { value } }) => {
                                setVerticalAxisLabel(value);
                            } }),
                        React.createElement(Textbox, { mt: "16px", maxLength: 50, name: "horizontal-axis-label", "data-testid": "e-widget-editor-layout-horizontal-axis-label", label: localize('@sage/xtrem-ui/widget-editor-horizontal-axis-label', 'Horizontal axis label'), value: horizontalAxisLabel, onChange: ({ target: { value } }) => {
                                setHorizontalAxisLabel(value);
                            } })))),
                widgetDefinition.type === 'INDICATOR_TILE' && (React.createElement(React.Fragment, null,
                    React.createElement(FilterableSelect, { "data-testid": "e-widget-editor-layout-icon", required: false, validationOnLabel: true, name: "icon", id: "icon", label: localize('@sage/xtrem-ui/widget-editor-icon', 'Icon'), onChange: onIconChanged, value: widgetDefinition.icon || '' }, icons.map(k => (React.createElement(Option, { text: k.label, value: k.value, key: k.value })))),
                    React.createElement(NumberComponent, { mt: "16px", "data-testid": "e-widget-editor-layout-decimal-digits", required: true, validationOnLabel: true, name: "decimal-digits", id: "decimal-digits", disabled: widgetDefinition.groupBy?.property.data.type === 'String' ||
                            widgetDefinition.groupBy?.method === 'distinctCount', label: localize('@sage/xtrem-ui/widget-editor-content-formatting', 'Decimal places'), onChange: onDecimalDigitsChanged, value: !isNil(widgetDefinition.decimalDigits)
                            ? String(widgetDefinition.decimalDigits)
                            : '', error: decimalDigitsValidationMessage }),
                    React.createElement(Textbox, { mt: "16px", name: "subtitle", id: "subtitle", "data-testid": "e-widget-editor-layout-subtitle", label: localize('@sage/xtrem-ui/widget-editor-subtitle', 'Subtitle'), value: subtitle, onChange: onSubtitleChanged, onBlur: onSubtitleBlur }),
                    arePagesAvailableForNode && React.createElement("div", { className: "e-layout-separator" }))),
                arePagesAvailableForNode && (React.createElement(React.Fragment, null,
                    React.createElement("div", { className: "e-widget-editor-label" }, localize('@sage/xtrem-ui/widget-actions', 'Actions')),
                    React.createElement(WidgetActionConfiguration, { actionKey: "seeAllAction", actionName: localize('@sage/xtrem-ui/widget-action-see-all', 'See all'), helperText: localize('@sage/xtrem-ui/widget-action-see-all-help', 'View records based on your data selection'), widgetDefinition: widgetDefinition, onWidgetDefinitionUpdated: onWidgetDefinitionUpdated }),
                    React.createElement(WidgetActionConfiguration, { actionKey: "createAction", actionName: localize('@sage/xtrem-ui/widget-action-create', 'Create'), helperText: localize('@sage/xtrem-ui/widget-action-create-help', 'Add a shortcut to create a new record'), widgetDefinition: widgetDefinition, onWidgetDefinitionUpdated: onWidgetDefinitionUpdated }))))),
            React.createElement("div", { className: "e-layout-preview-container" },
                React.createElement("div", { className: "e-layout-preview-preview-label" }, localize('@sage/xtrem-ui/widget-preview-label', 'Preview')),
                previewDashboardConfiguration && (React.createElement(PreviewDashboardConfigurationContent, { widgetDefinition: widgetDefinition, previewDashboardConfiguration: previewDashboardConfiguration, onBreakpointChange: onBreakpointChange }))))));
}
//# sourceMappingURL=layout-step.js.map