import { FilterableSelect, Option, OptionRow } from 'carbon-react/esm/components/select';
import * as React from 'react';
import { aggregatableGraphqlTypes } from '../../../component/types';
import { GridColumn, GridRow } from '@sage/xtrem-ui-components';
import { localize } from '../../../service/i18n-service';
import { getGutterSize } from '../../../utils/responsive-utils';
import { get, includes, memoize } from 'lodash';
import { PropertyTableHeader, getPropertyParentNode } from '../widget-editor-utils';
import Numeric from 'carbon-react/esm/components/number';
import Textbox from 'carbon-react/esm/components/textbox';
import { Aggregations, aggregationsGraphqlMapping, numericFields } from '@sage/xtrem-shared';
function indicatorWidgetContentReducer(state, action) {
    switch (action.type) {
        case 'SET_GROUPING_METHOD':
            return {
                ...state,
                groupingMethod: action.groupingMethod,
            };
        case 'SET_DIVISOR':
            return {
                ...state,
                divisor: action.divisor,
            };
        case 'SET_GROUPING_PROPERTY':
            if (action.groupingProperty === state.groupingProperty) {
                return state;
            }
            if (action.groupingProperty === undefined ||
                action.groupingProperty.data.type === undefined ||
                !includes(aggregatableGraphqlTypes, action.groupingProperty.data.type)) {
                return {
                    ...state,
                    groupingMethods: [],
                    groupingMethod: undefined,
                    groupingProperty: undefined,
                    divisor: '1',
                };
            }
            const groupingMethods = get(aggregationsGraphqlMapping, action.groupingProperty.data.type) ?? [];
            return {
                ...state,
                groupingProperty: action.groupingProperty,
                groupingMethods,
                groupingMethod: groupingMethods.length === 1 ? groupingMethods[0] : undefined,
                divisor: '1',
            };
        default:
            return state;
    }
}
export const aggregationTranslations = memoize(() => ({
    [Aggregations.min]: localize('@sage/xtrem-ui/minimum', 'Minimum'),
    [Aggregations.max]: localize('@sage/xtrem-ui/maximum', 'Maximum'),
    [Aggregations.sum]: localize('@sage/xtrem-ui/sum', 'Sum'),
    [Aggregations.avg]: localize('@sage/xtrem-ui/average', 'Average'),
    [Aggregations.distinctCount]: localize('@sage/xtrem-ui/distinct-count', 'Distinct count'),
}));
export function IndicatorWidgetContent({ browserIs, onWidgetDefinitionUpdated, widgetDefinition, nodeNames, }) {
    const [state, dispatch] = React.useReducer(indicatorWidgetContentReducer, {
        groupingMethod: widgetDefinition.groupBy?.method,
        groupingMethods: widgetDefinition.groupBy?.property && widgetDefinition.groupBy.property.data.type
            ? (get(aggregationsGraphqlMapping, widgetDefinition.groupBy.property.data.type) ?? [])
            : [],
        groupingProperties: Object.values(widgetDefinition.selectedProperties ?? {}),
        groupingProperty: widgetDefinition.groupBy?.property,
        divisor: widgetDefinition.groupBy?.divisor ? String(widgetDefinition.groupBy.divisor) : '1',
    });
    React.useEffect(() => {
        if (state.groupingMethod !== undefined &&
            state.groupingProperty !== undefined &&
            state.divisor !== undefined &&
            !Number.isNaN(Number(state.divisor))) {
            onWidgetDefinitionUpdated({
                ...widgetDefinition,
                groupBy: {
                    method: state.groupingMethod,
                    property: state.groupingProperty,
                    divisor: Number(state.divisor),
                },
            });
        }
        else {
            onWidgetDefinitionUpdated({ ...widgetDefinition, groupBy: undefined });
        }
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [state.groupingMethod, state.groupingProperty, state.divisor]);
    const onGroupingMethodChanged = React.useCallback((evt) => {
        dispatch({
            type: 'SET_GROUPING_METHOD',
            groupingMethod: (evt.target.value || undefined),
        });
    }, []);
    const onDivisorChanged = React.useCallback((evt) => {
        dispatch({
            type: 'SET_DIVISOR',
            divisor: evt.target.value,
        });
    }, []);
    const divisorError = React.useMemo(() => {
        const value = Number(state.divisor);
        if (Number.isNaN(value)) {
            return localize('@sage/xtrem-ui/must-be-a-number', 'You need to enter a number');
        }
        if (value <= 0) {
            return localize('@sage/xtrem-ui/must-be-greater-than-zero', 'This value needs to be greater than 0.');
        }
        return undefined;
    }, [state.divisor]);
    const onGroupingPropertyChanged = React.useCallback(({ target: { value: groupingProperty }, }) => {
        dispatch({
            type: 'SET_GROUPING_PROPERTY',
            groupingProperty: groupingProperty || undefined,
        });
    }, []);
    const gridGutter = getGutterSize(browserIs);
    const canBeAggregated = (canFilter = false, canSort = false) => canFilter && canSort;
    return (React.createElement(GridRow, { columns: 8, gutter: gridGutter, margin: 0, verticalMargin: 0 },
        React.createElement(GridColumn, { columnSpan: 4 },
            React.createElement(FilterableSelect, { required: true, multiColumn: true, tableHeader: React.createElement(PropertyTableHeader, null), openOnFocus: true, "data-testid": "e-widget-editor-grouping-property", label: localize('@sage/xtrem-ui/widget-editor-grouping-property', 'Grouping property'), 
                // @ts-expect-error "onChange" is actually triggered with { target: { value: Property } }
                onChange: onGroupingPropertyChanged, 
                // @ts-expect-error value is of type Property
                value: state.groupingProperty ?? {} }, state.groupingProperties
                .filter(p => p.data.type &&
                includes(aggregatableGraphqlTypes, p.data.type) &&
                canBeAggregated(p.data.canFilter, p.data.canSort))
                .map(p => (
            // @ts-expect-error value is of type Property
            React.createElement(OptionRow, { text: p.label, value: p, key: p.id },
                React.createElement("td", { width: "50%", style: {
                        overflow: 'hidden',
                        whiteSpace: 'pre-line',
                        maxWidth: 0,
                    } }, p.label),
                React.createElement("td", { width: "50%", style: {
                        overflow: 'hidden',
                        whiteSpace: 'pre-line',
                        maxWidth: 0,
                        textAlign: 'end',
                    } }, getPropertyParentNode({ nodeNames, property: p, widgetDefinition }))))))),
        React.createElement(GridColumn, { columnSpan: 4 },
            state.groupingMethods.length > 1 && (React.createElement(FilterableSelect, { required: true, openOnFocus: true, "data-testid": "e-widget-editor-grouping-method", label: localize('@sage/xtrem-ui/widget-editor-grouping-method', 'Grouping method'), onChange: onGroupingMethodChanged, value: state.groupingMethod ?? '', disabled: state.groupingProperty == null }, state.groupingMethods.map(aggregationMethod => (React.createElement(Option, { text: aggregationTranslations()[aggregationMethod], value: aggregationMethod, key: aggregationMethod }))))),
            state.groupingMethods.length <= 1 && (React.createElement(Textbox, { "data-testid": "e-widget-editor-grouping-method", required: true, disabled: state.groupingProperty == null, readOnly: true, label: localize('@sage/xtrem-ui/widget-editor-grouping-method', 'Grouping method'), value: aggregationTranslations()[state.groupingMethods[0]] }))),
        state.groupingProperty?.data?.type && includes(numericFields, state.groupingProperty.data.type) && (React.createElement(GridColumn, { columnSpan: 4 },
            React.createElement(Numeric, { "data-testid": "e-widget-editor-content-divisor", label: localize('@sage/xtrem-ui/divisor', 'Divisor'), inputMode: "numeric", tooltipPosition: "top", onChange: onDivisorChanged, value: state.divisor ?? '', disabled: state.groupingProperty == null, error: divisorError, fieldHelp: localize('@sage/xtrem-ui/divisor-helper-text', 'Reduce large numbers by entering a divisor.') })))));
}
//# sourceMappingURL=indicator-tile-widget-content.js.map