import * as React from 'react';
import { connect } from 'react-redux';
import * as xtremRedux from '../redux';
import { objectKeys } from '@sage/xtrem-shared';
import DialogFullScreen from 'carbon-react/esm/components/dialog-full-screen';
import { getWidgetsFromDashboardItems } from '../service/dashboard-service';
import { DASHBOARD_SCREEN_ID } from '../utils/constants';
import localeText from './dashboard-strings';
import AsyncDragDropCanvasWrapper from './async-drag-drop-canvas-wrapper';
import { localize } from '../service/i18n-service';
import Button from 'carbon-react/esm/components/button';
import Form from 'carbon-react/esm/components/form';
import { DashboardTitleEditor } from './dashboard-title-editor';
import { openDirtyPageConfirmationDialog } from '../service/dirty-state-service';
import { showToast } from '../service/toast-service';
import { ConnectedDashboardSelectionTabs } from './dashboard-selection-tabs';
import { DashboardPlaceholder } from './dashboard-placeholder';
import NavigationPanelComponent from '../component/container/navigation-panel/navigation-panel-component';
import { getNavigationPanelState } from '../utils/state-utils';
import ToggleNavigationPanelButton from '../component/container/navigation-panel/toggle-navigation-panel-button';
import { useElementSize } from 'usehooks-ts';
import { AsyncConnectedWidgetEditorDialog } from './widget-editor/async-widget-editor-dialog';
import { DASHBOARD_BREAKPOINTS, DASHBOARD_COLS, DASHBOARD_EDITOR_SIDEBAR_WIDTH, DASHBOARD_EDITOR_WITH_SIDEBAR_BREAKPOINTS, getWidgetsWithLoadingState, } from './dashboard-utils';
import { isEqual, isNil } from 'lodash';
import { ContextType } from '../types';
export function DashboardEditorDialog({ canEditDashboards, contextVariables, currentDashboardDefinition, dashboardDefinition, group, isDirty, isNavigationPanelOpened, isOpen, isRedoDisabled, isUndoDisabled, loadingDashboards, locale, onClose, redoDashboardEditor, removeDashboardEditorWidget, saveDashboardEditorState, setNavigationPanelIsOpened, undoDashboardEditor, updateDashboardEditorWidgetPositions, updateDashboardTitleInEditor, widgets, isMobileEditor, }) {
    const [dashboardCanvasContainerRef, { width = 0 }] = useElementSize();
    const [breakpoint, setBreakpoint] = React.useState(null);
    const currentWidgets = React.useMemo(() => isNil(breakpoint)
        ? []
        : getWidgetsFromDashboardItems({
            dashboardId: String(dashboardDefinition._id),
            group,
            children: currentDashboardDefinition.children,
            widgets,
            isEditing: true,
            onCloseCallback: removeDashboardEditorWidget,
            canEditDashboards,
            breakpoint,
        }), [
        breakpoint,
        canEditDashboards,
        currentDashboardDefinition.children,
        dashboardDefinition._id,
        group,
        removeDashboardEditorWidget,
        widgets,
    ]);
    const onDataChange = React.useCallback(({ widgets: updatedWidgets, breakpoint: b }) => {
        if (!isEqual(currentWidgets, updatedWidgets)) {
            updateDashboardEditorWidgetPositions(updatedWidgets, b);
        }
    }, [updateDashboardEditorWidgetPositions, currentWidgets]);
    const onDialogClose = React.useCallback(() => {
        // If the dashboard is dirty, first the confirmation dialog needs to be displayed
        if (isDirty) {
            openDirtyPageConfirmationDialog(DASHBOARD_SCREEN_ID)
                .then(onClose)
                .catch(() => {
                // Intentionally left empty, the user decided not to leave.
            });
        }
        else {
            onClose();
        }
    }, [onClose, isDirty]);
    const onChangeTabs = React.useCallback(async () => {
        // If the dashboard is dirty, first the confirmation dialog needs to be displayed
        if (isDirty) {
            await openDirtyPageConfirmationDialog(DASHBOARD_SCREEN_ID);
        }
    }, [isDirty]);
    const onSave = React.useCallback(async (ev) => {
        ev.preventDefault();
        try {
            await saveDashboardEditorState();
            showToast(localize('@sage/xtrem-ui/dashboard-editor-saved-successfully', 'Dashboard saved.'), {
                type: 'success',
            });
            onClose();
        }
        catch (err) {
            showToast(localize('@sage/xtrem-ui/dashboard-editor-save-error', 'Failed to save dashboard.'), {
                type: 'error',
            });
        }
    }, [saveDashboardEditorState, onClose]);
    const onBreakpointChange = React.useCallback((newBreakpoint) => {
        setBreakpoint(newBreakpoint);
    }, []);
    const args = React.useMemo(() => ({
        type: 'responsive',
        isHeaderHidden: true,
        locale,
        isEditMode: true,
        onBreakpointChange,
        layout: {
            cols: DASHBOARD_COLS,
            breakpoints: isNavigationPanelOpened
                ? DASHBOARD_EDITOR_WITH_SIDEBAR_BREAKPOINTS
                : DASHBOARD_BREAKPOINTS,
            rowHeight: 120,
            width: isNavigationPanelOpened ? width - DASHBOARD_EDITOR_SIDEBAR_WIDTH : width,
            isDraggable: true,
        },
        widgets: currentWidgets,
        onDataChange,
        stringLiterals: localeText(),
    }), [currentWidgets, isNavigationPanelOpened, locale, onBreakpointChange, onDataChange, width]);
    return (React.createElement(DialogFullScreen, { className: "e-dashboard-editor-dialog", "data-testid": "e-dashboard-editor-dialog", title: localize('@sage/xtrem-ui/dashboard-editor-title', 'Dashboard editor'), open: !!isOpen, onCancel: onDialogClose },
        React.createElement(Form, { stickyFooter: true, leftSideButtons: React.createElement(Button, { "data-testid": "e-dashboard-editor-dialog-cancel", onClick: onDialogClose }, localize('@sage/xtrem-ui/dashboard-editor-cancel-edit', 'Cancel')), saveButton: React.createElement(Button, { buttonType: "primary", "data-testid": "e-dashboard-editor-dialog-save", disabled: !isDirty, onClick: onSave, type: "submit" }, localize('@sage/xtrem-ui/dashboard-editor-save', 'Save')), onSubmit: ev => {
                ev.preventDefault();
            } },
            React.createElement("div", { className: "e-dashboard-editor" },
                !isNavigationPanelOpened && !isMobileEditor && (React.createElement(ToggleNavigationPanelButton, { setNavigationPanelIsOpened: setNavigationPanelIsOpened, isNavigationPanelOpened: isNavigationPanelOpened, screenId: DASHBOARD_SCREEN_ID })),
                isNavigationPanelOpened && (React.createElement(NavigationPanelComponent, { contextType: ContextType.navigationPanel, screenId: DASHBOARD_SCREEN_ID, selectedRecordId: "-1" })),
                React.createElement("div", { className: "e-dashboard-editor-main" },
                    React.createElement("div", { className: "e-dashboard-editor-tab-container" },
                        React.createElement(ConnectedDashboardSelectionTabs, { canChangeTabs: onChangeTabs, group: group })),
                    loadingDashboards && React.createElement(DashboardPlaceholder, null),
                    !loadingDashboards && (React.createElement(React.Fragment, null,
                        React.createElement("div", { className: "e-dashboard-editor-title-line" },
                            React.createElement(DashboardTitleEditor, { title: currentDashboardDefinition.title, onChange: updateDashboardTitleInEditor }),
                            React.createElement("div", { className: "e-dashboard-editor-history" },
                                React.createElement(Button, { "data-testid": "e-dashboard-editor-undo", iconType: "u_turn_left", buttonType: "tertiary", onClick: undoDashboardEditor, disabled: isUndoDisabled }, localize('@sage/xtrem-ui/dashboard-editor-undo', 'Undo')),
                                React.createElement(Button, { "data-testid": "e-dashboard-editor-redo", iconType: "u_turn_right", buttonType: "tertiary", onClick: redoDashboardEditor, disabled: isRedoDisabled }, localize('@sage/xtrem-ui/dashboard-editor-redo', 'Redo')))),
                        React.createElement("div", { className: "e-dashboard-editor-body", "data-testid": "e-dashboard-editor-body", ref: dashboardCanvasContainerRef },
                            React.createElement(AsyncDragDropCanvasWrapper, { json: args }))))))),
        isOpen && React.createElement(AsyncConnectedWidgetEditorDialog, { group: group, contextVariables: contextVariables })));
}
const mapStateToProps = (state, props) => {
    const dashboardGroup = state.dashboard.dashboardGroups[props.group];
    const { dashboardEditor, dashboards, widgets } = dashboardGroup;
    const activeDashboard = objectKeys(dashboards).find(d => dashboards[d].isSelected);
    const navigationPanelState = getNavigationPanelState(DASHBOARD_SCREEN_ID, state);
    if (!activeDashboard) {
        throw new Error('The editor cannot be used if no active dashboard definition is loaded.');
    }
    return {
        ...props,
        canEditDashboards: state.dashboard?.canEditDashboards || false,
        currentDashboardDefinition: dashboardEditor.currentDashboardDefinition,
        dashboardDefinition: dashboards[activeDashboard],
        isDirty: dashboardEditor.isDirty,
        isNavigationPanelOpened: navigationPanelState?.isOpened || false,
        isOpen: dashboardEditor.isOpen,
        isRedoDisabled: dashboardEditor.currentHistoryIndex === dashboardEditor.history.length - 1,
        isUndoDisabled: dashboardEditor.currentHistoryIndex === 0,
        loadingDashboards: state.loading.loadingDashboards,
        locale: (state.applicationContext?.locale || 'en-US'),
        onClose: xtremRedux.actions.actionStub,
        redoDashboardEditor: xtremRedux.actions.actionStub,
        removeDashboardEditorWidget: xtremRedux.actions.actionStub,
        saveDashboardEditorState: xtremRedux.actions.actionStub,
        setNavigationPanelIsOpened: xtremRedux.actions.actionStub,
        undoDashboardEditor: xtremRedux.actions.actionStub,
        updateDashboardEditorWidgetPositions: xtremRedux.actions.actionStub,
        updateDashboardTitleInEditor: xtremRedux.actions.actionStub,
        widgets: getWidgetsWithLoadingState({ widgetState: state.loading.widgets, widgets: widgets ?? {} }),
        isMobileEditor: state.browser.is.xs,
    };
};
const mapDispatchToProps = (dispatch, props) => {
    return {
        saveDashboardEditorState: () => dispatch(xtremRedux.actions.saveDashboardEditorState(props.group, props.contextVariables)),
        onClose: () => {
            dispatch(xtremRedux.actions.closeDashboardEditorDialog(props.group));
        },
        setNavigationPanelIsOpened: (isOpened) => dispatch(xtremRedux.actions.setNavigationPanelIsOpened(isOpened, DASHBOARD_SCREEN_ID)),
        undoDashboardEditor: () => {
            dispatch(xtremRedux.actions.undoDashboardEditor(props.group));
        },
        redoDashboardEditor: () => {
            dispatch(xtremRedux.actions.redoDashboardEditor(props.group));
        },
        removeDashboardEditorWidget: (widgetId) => {
            dispatch(xtremRedux.actions.removeDashboardEditorWidget(widgetId, props.group));
        },
        updateDashboardEditorWidgetPositions: (widgets, breakpoint) => {
            dispatch(xtremRedux.actions.updateDashboardEditorWidgetPositions({
                widgets,
                group: props.group,
                contextVariables: props.contextVariables,
                breakpoint,
            }));
        },
        updateDashboardTitleInEditor: (newTitle) => {
            dispatch(xtremRedux.actions.updateDashboardTitleInEditor(newTitle, props.group));
        },
    };
};
export const ConnectedDashboardEditorDialog = connect(mapStateToProps, mapDispatchToProps)(DashboardEditorDialog);
//# sourceMappingURL=dashboard-editor-dialog.js.map