import * as React from 'react';
import { connect } from 'react-redux';
import * as xtremRedux from '../redux';
import { objectKeys } from '@sage/xtrem-shared';
import { getWidgetsFromDashboardItems } from '../service/dashboard-service';
import { DASHBOARD_SCREEN_ID } from '../utils/constants';
import Typography from 'carbon-react/esm/components/typography';
import localeText from './dashboard-strings';
import AsyncDragDropCanvasWrapper from './async-drag-drop-canvas-wrapper';
import Heading from 'carbon-react/esm/components/heading';
import Button from 'carbon-react/esm/components/button';
import { ActionPopover, ActionPopoverDivider, ActionPopoverItem } from 'carbon-react/esm/components/action-popover';
import { localize } from '../service/i18n-service';
import { ConnectedDashboardEditorDialog } from './dashboard-editor-dialog';
import { ConnectedDashboardSelectionTabs } from './dashboard-selection-tabs';
import { ConnectedDashboardCreateDialog } from './dashboard-create-dialog';
import { confirmationDialog } from '../service/dialog-service';
import { xtremConsole } from '../utils/console';
import { showToast } from '../service/toast-service';
import { DashboardPlaceholder } from './dashboard-placeholder';
import { isEqual } from 'lodash';
import { DASHBOARD_BREAKPOINTS, DASHBOARD_COLS, getWidgetsWithLoadingState } from './dashboard-utils';
import Icon from 'carbon-react/esm/components/icon';
import * as tokens from '@sage/design-tokens/js/base/common';
export class DashboardComponent extends React.Component {
    constructor(props) {
        super(props);
        this.fetchDashboards = () => {
            if (!this.props.dashboardDefinition && this.props.hasInstalledPackagesLoaded && !this.state.startedFetching) {
                this.setState({ startedFetching: true }, this.props.fetchUserDashboardDefinition);
            }
        };
        this.onCreateDialogOpen = () => this.setState({ dialogMode: 'CREATE_DASHBOARD' });
        this.onCreateDialogClose = (success = false) => {
            this.setState({ dialogMode: 'NONE' });
            if (success) {
                this.props.openDashboardEditorDialog();
            }
        };
        this.onDeleteDashboard = async () => {
            if (this.props.dashboardDefinition) {
                try {
                    await confirmationDialog(DASHBOARD_SCREEN_ID, 'warn', localize('@sage/xtrem-ui/dashboard-delete-dashboard-dialog-title', 'Delete dashboard'), localize('@sage/xtrem-ui/dashboard-delete-dashboard-dialog-message', 'Are you sure you want to delete this dashboard?'));
                    try {
                        await this.props.deleteDashboard(String(this.props.dashboardDefinition._id));
                        showToast(localize('@sage/xtrem-ui/dashboard-deleted', 'Dashboard deleted.'), {
                            type: 'success',
                        });
                    }
                    catch (err) {
                        xtremConsole.error(err);
                        showToast(localize('@sage/xtrem-ui/dashboard-failed-to-delete', 'The dashboard could not be deleted.'), { type: 'error' });
                    }
                }
                catch {
                    // Do nothing
                }
            }
        };
        this.onDuplicateDashboard = async () => {
            if (this.props.dashboardDefinition) {
                try {
                    await this.props.cloneDashboard(String(this.props.dashboardDefinition._id));
                    showToast(localize('@sage/xtrem-ui/dashboard-duplicated', 'Dashboard duplicated.'), {
                        type: 'success',
                    });
                }
                catch (err) {
                    xtremConsole.error(err);
                    showToast(localize('@sage/xtrem-ui/dashboard-failed-to-duplicate', 'The dashboard could not be duplicated.'), { type: 'error' });
                }
            }
        };
        this.renderNoDashboards = () => {
            return (React.createElement("div", { className: "e-dashboard-empty", "data-testid": "e-dashboard-empty-no-dashboard" },
                React.createElement(Heading, { "data-testid": "e-dashboard-empty-no-dashboard-title", divider: false, title: localize('@sage/xtrem-ui/no-dashboard-title', 'Drive your business') }),
                React.createElement("div", { className: "e-dashboard-empty-center-wrapper" },
                    React.createElement("img", { width: 300, height: 300, src: "/images/dashboard-create.svg", alt: "Create dashboard logo" }),
                    React.createElement("div", { className: "e-dashboard-empty-right-wrapper" },
                        React.createElement(Heading, { "data-testid": "e-dashboard-empty-no-dashboard-heading", divider: false, title: localize('@sage/xtrem-ui/no-dashboard-heading', 'This page has no dashboard.') }),
                        React.createElement("span", { className: "e-dashboard-empty-right-subtitle", "data-testid": "e-dashboard-empty-no-dashboard-subtitle" }, localize('@sage/xtrem-ui/no-dashboard-subtitle', 'Create a dashboard to get started.')),
                        this.props.canEditDashboards && (React.createElement(Button, { "data-testid": "e-dashboard-create", iconType: "plus", onClick: this.onCreateDialogOpen, buttonType: "primary" }, localize('@sage/xtrem-ui/dashboard-create-dashboard', 'Create dashboard')))))));
        };
        this.renderEmptyDashboard = () => {
            return (React.createElement("div", { className: "e-dashboard-empty", "data-testid": "e-dashboard-empty-no-widgets" },
                React.createElement("div", { className: "e-dashboard-empty-center-wrapper" },
                    React.createElement("img", { width: 300, height: 300, src: "/images/dashboard-create.svg", alt: "Create dashboard logo" }),
                    React.createElement("div", { className: "e-dashboard-empty-right-wrapper" },
                        React.createElement(Heading, { "data-testid": "e-dashboard-empty-no-widgets-heading", divider: false, title: localize('@sage/xtrem-ui/dashboard-empty-heading', 'This dashboard is empty.') }),
                        React.createElement("span", { className: "e-dashboard-empty-right-subtitle", "data-testid": "e-dashboard-empty-no-widgets-subtitle" }, localize('@sage/xtrem-ui/dashboard-empty-subtitle', 'Add a widget to customize your dashboard.')),
                        this.props.canEditDashboards && (React.createElement(Button, { "data-testid": "e-dashboard-create", iconType: "plus", onClick: this.props.openDashboardEditorDialog, buttonType: "primary" }, localize('@sage/xtrem-ui/dashboard-add-widget', 'Add widget')))))));
        };
        this.renderActionButton = () => {
            return (React.createElement("div", { "data-component": "action-popover-button", "data-element": "action-popover-button" },
                React.createElement(Icon, { type: "ellipsis_vertical", tooltipMessage: localize('@sage/xtrem-ui/dashboard-more_options', 'More options'), color: tokens.colorsUtilityMajor450 })));
        };
        this.renderKebabMenu = () => {
            if (!this.props.dashboardDefinition) {
                return null;
            }
            const destructiveIconClassName = {
                className: 'e-action-popover-item-destructive',
            };
            return (React.createElement(Typography, { variant: "h1", mt: 3, mr: 3, ml: 3, mb: 0 },
                React.createElement("span", { className: "e-dashboard-title", "data-testid": "e-dashboard-title" }, this.props.dashboardDefinition.title),
                this.props.canEditDashboards && (React.createElement("span", { className: "e-dashboard-title-menu" },
                    React.createElement(ActionPopover, { "data-testid": "e-dashboard-actions", renderButton: this.renderActionButton },
                        React.createElement(ActionPopoverItem, { "data-pendoid": "dashboardActionEdit", icon: "edit", onClick: this.props.openDashboardEditorDialog, "data-testid": "e-dashboard-action-edit" }, localize('@sage/xtrem-ui/dashboard-edit', 'Edit')),
                        React.createElement(ActionPopoverItem, { "data-pendoid": "dashboardActionDuplicate", icon: "duplicate", onClick: this.onDuplicateDashboard, "data-testid": "e-dashboard-action-duplication" }, localize('@sage/xtrem-ui/dashboard-duplicate', 'Duplicate')),
                        React.createElement(ActionPopoverItem, { "data-pendoid": "dashboardActionCreate", icon: "plus_large", onClick: this.onCreateDialogOpen, "data-testid": "e-dashboard-action-create" }, localize('@sage/xtrem-ui/dashboard-create', 'Create')),
                        !this.props.dashboardDefinition?.isTemplate && React.createElement(ActionPopoverDivider, null),
                        !this.props.dashboardDefinition?.isTemplate && (React.createElement(ActionPopoverItem, { "data-pendoid": "dashboardActionDelete", icon: "delete", onClick: this.onDeleteDashboard, "data-testid": "e-dashboard-action-delete", ...destructiveIconClassName }, localize('@sage/xtrem-ui/dashboard-delete', 'Delete'))))))));
        };
        this.getWidgets = () => {
            return this.props.dashboardDefinition
                ? getWidgetsFromDashboardItems({
                    dashboardId: String(this.props.dashboardDefinition._id),
                    group: this.props.group,
                    children: this.props.dashboardDefinition.children,
                    widgets: this.props.widgets,
                    isEditing: undefined,
                    onCloseCallback: undefined,
                    canEditDashboards: this.props.canEditDashboards,
                    breakpoint: this.state.breakpoint,
                    onTelemetryEvent: this.props.onTelemetryEvent,
                })
                : [];
        };
        this.onBreakpointChange = newBreakpoint => {
            this.setState({ breakpoint: newBreakpoint });
        };
        this.renderDashboard = () => {
            const widgets = this.getWidgets();
            const isLoading = !this.state.startedFetching || !this.state.verticalSpaceReady;
            const isEmptyDashboard = !isLoading &&
                !this.props.loadingDashboards &&
                !this.props.globalLoading &&
                this.state.verticalSpaceReady &&
                this.props.dashboardDefinition !== undefined &&
                widgets.length === 0;
            return (React.createElement(React.Fragment, null,
                !this.props.loadingDashboards && this.renderKebabMenu(),
                !this.props.loadingDashboards &&
                    (isEmptyDashboard ? (this.renderEmptyDashboard()) : (React.createElement(AsyncDragDropCanvasWrapper, { json: {
                            onBreakpointChange: this.onBreakpointChange,
                            type: 'responsive',
                            isHeaderHidden: true,
                            isEditMode: false,
                            locale: this.props.locale,
                            layout: {
                                cols: DASHBOARD_COLS,
                                breakpoints: DASHBOARD_BREAKPOINTS,
                                rowHeight: 120,
                                width: 1200,
                                isDraggable: false,
                            },
                            widgets,
                            stringLiterals: localeText(),
                        } }))),
                this.props.loadingDashboards && React.createElement(DashboardPlaceholder, null),
                React.createElement(ConnectedDashboardEditorDialog, { group: this.props.group, contextVariables: this.props.dashboardContextVariables })));
        };
        this.state = {
            dialogMode: 'NONE',
            startedFetching: this.props.dashboardDefinition !== undefined,
            verticalSpaceReady: false,
            breakpoint: 'md',
        };
    }
    componentDidMount() {
        this.fetchDashboards();
        setTimeout(() => {
            this.setState({ verticalSpaceReady: true });
        }, 100);
    }
    componentDidUpdate(prevProps) {
        this.fetchDashboards();
        if (prevProps.group !== this.props.group ||
            !isEqual(prevProps.dashboardContextVariables, this.props.dashboardContextVariables)) {
            this.props.refreshCurrentDashboard();
        }
    }
    render() {
        const isLoading = !this.state.startedFetching || !this.state.verticalSpaceReady;
        const hasNoDashboards = !isLoading &&
            !this.props.loadingDashboards &&
            !this.props.globalLoading &&
            this.state.verticalSpaceReady &&
            this.props.dashboardDefinition === undefined;
        return (React.createElement("main", { className: "e-dashboard" },
            this.props.hasTabs && (React.createElement(ConnectedDashboardSelectionTabs, { onCreateDialogOpen: this.onCreateDialogOpen, group: this.props.group, contextVariables: this.props.dashboardContextVariables })),
            hasNoDashboards && this.renderNoDashboards(),
            this.props.dashboardDefinition !== undefined &&
                this.state.verticalSpaceReady &&
                this.renderDashboard(),
            this.props.canEditDashboards && (React.createElement(ConnectedDashboardCreateDialog, { group: this.props.group, isOpen: this.state.dialogMode === 'CREATE_DASHBOARD', onClose: this.onCreateDialogClose, onNextClicked: this.onCreateDialogClose, contextVariables: this.props.dashboardContextVariables }))));
    }
}
const mapStateToProps = (state, props) => {
    const dashboard = state.dashboard.dashboardGroups[props.group];
    const { dashboards, widgets } = dashboard || {};
    const activeDashboard = objectKeys(dashboards || {}).find(d => dashboards[d].isSelected);
    return {
        ...props,
        canEditDashboards: state.dashboard.canEditDashboards && !state.browser.is.xs,
        cloneDashboard: xtremRedux.actions.actionStub,
        dashboardDefinition: activeDashboard ? dashboards[activeDashboard] : undefined,
        deleteDashboard: xtremRedux.actions.actionStub,
        fetchUserDashboardDefinition: xtremRedux.actions.actionStub,
        globalLoading: state.loading.globalLoading,
        hasInstalledPackagesLoaded: !!state.applicationPackages,
        loadingDashboards: state.loading.loadingDashboards,
        locale: (state.applicationContext?.locale || 'en-US'),
        openDashboardEditorDialog: xtremRedux.actions.actionStub,
        refreshCurrentDashboard: xtremRedux.actions.actionStub,
        widgets: getWidgetsWithLoadingState({ widgetState: state.loading.widgets, widgets: widgets ?? {} }),
        hasTabs: (dashboard?.availableDashboards ?? []).length > 1,
        onTelemetryEvent: state.applicationContext?.onTelemetryEvent,
    };
};
const mapDispatchToProps = (dispatch, props) => {
    return {
        fetchUserDashboardDefinition: () => {
            dispatch(xtremRedux.actions.fetchUserDashboardDefinition(props.group, props.dashboardContextVariables));
        },
        cloneDashboard: (dashboardId) => dispatch(xtremRedux.actions.cloneDashboard(dashboardId, props.group, props.dashboardContextVariables)),
        deleteDashboard: (dashboardId) => dispatch(xtremRedux.actions.deleteDashboard(dashboardId, props.group, props.dashboardContextVariables)),
        openDashboardEditorDialog: () => {
            dispatch(xtremRedux.actions.openDashboardEditorDialog(props.group, props.dashboardContextVariables));
        },
        refreshCurrentDashboard: () => {
            dispatch(xtremRedux.actions.refreshCurrentDashboard(props.group, props.dashboardContextVariables));
        },
    };
};
export const ConnectedDashboardComponent = connect(mapStateToProps, mapDispatchToProps)(DashboardComponent);
//# sourceMappingURL=dashboard-component.js.map