import Icon from 'carbon-react/esm/components/icon';
import IconButton from 'carbon-react/esm/components/icon-button';
import { LoaderSpinner } from 'carbon-react/esm/components/loader-spinner';
import * as React from 'react';
import { localize } from '../service/i18n-service';
import { xtremConsole } from '../utils/console';
import { sendTranscribeStreamingRequest } from './copilot-service';
const SAMPLE_RATE = 16000;
const formatDuration = (seconds) => {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins}:${secs.toString().padStart(2, '0')}`;
};
function mergeBuffers(buffers) {
    const totalLength = buffers.reduce((acc, b) => acc + b.length, 0);
    const merged = new Int16Array(totalLength);
    let offset = 0;
    for (const b of buffers) {
        merged.set(b, offset);
        offset += b.length;
    }
    return new Blob([merged.buffer], { type: 'audio/pcm' });
}
export function RecordAudioButton({ applicationContext, onTelemetryEvent, onTranscriptionFinished, onTranscriptionStarted, onTranscriptionUpdated, }) {
    const { path, locale } = applicationContext;
    const [isRecording, setIsRecording] = React.useState(false);
    const [isProcessing, setIsProcessing] = React.useState(false);
    // eslint-disable-next-line @typescript-eslint/naming-convention
    const [_error, setError] = React.useState(null);
    const [recordingDuration, setRecordingDuration] = React.useState(0);
    const mediaRecorderRef = React.useRef(null);
    const streamRef = React.useRef(null);
    const audioContextRef = React.useRef(null);
    const processorRef = React.useRef(null);
    const durationIntervalRef = React.useRef(null);
    const recordingStartTimeRef = React.useRef(0);
    const processRecording = React.useCallback(async (blob, durationInMilliseconds = 0) => {
        onTelemetryEvent?.('copilotVoiceInputUsed', {
            recordingDuration: Math.round(durationInMilliseconds),
            recordingSize: blob.size,
        });
        try {
            setIsProcessing(true);
            setError(null);
            await sendTranscribeStreamingRequest({
                blob,
                sampleRate: SAMPLE_RATE,
                path,
                locale,
                onStreamEvent: event => {
                    switch (event.type) {
                        case 'connected':
                            xtremConsole.log('Streaming connected');
                            onTranscriptionStarted();
                            break;
                        case 'error':
                            xtremConsole.error('Streaming error:', event.error);
                            setError(`Streaming failed: ${event.error}`);
                            onTranscriptionFinished('');
                            break;
                        case 'text_chunk':
                            onTranscriptionUpdated(event.text);
                            break;
                        case 'completed':
                            onTranscriptionFinished(event.finalText);
                            break;
                        default:
                            throw new Error(`Unknown event type: ${event.type}`);
                    }
                },
            });
        }
        catch (err) {
            xtremConsole.error('Transcription error:', err);
            setError(`Transcription failed: ${err instanceof Error ? err.message : 'Unknown error'}`);
        }
        finally {
            setIsProcessing(false);
        }
    }, [locale, onTelemetryEvent, onTranscriptionFinished, onTranscriptionStarted, onTranscriptionUpdated, path]);
    const startRecording = React.useCallback(async () => {
        try {
            setError(null);
            setRecordingDuration(0);
            const stream = await navigator.mediaDevices.getUserMedia({
                audio: {
                    channelCount: 1,
                    sampleRate: SAMPLE_RATE,
                    echoCancellation: true,
                    noiseSuppression: true,
                    autoGainControl: true,
                },
            });
            const audioContext = new AudioContext({ sampleRate: SAMPLE_RATE });
            const source = audioContext.createMediaStreamSource(stream);
            await audioContext.audioWorklet.addModule(`${path ?? ''}/pcm-processor.js`);
            const workletNode = new AudioWorkletNode(audioContext, 'pcm-processor');
            source.connect(workletNode);
            workletNode.connect(audioContext.destination); // optional (so you can hear yourself)
            const recordedBuffers = [];
            workletNode.port.onmessage = (event) => {
                const pcmSamples = event.data; // Float32Array
                const pcm16 = new Int16Array(pcmSamples.length);
                for (let i = 0; i < pcmSamples.length; i++) {
                    const s = Math.max(-1, Math.min(1, pcmSamples[i]));
                    pcm16[i] = s < 0 ? s * 0x8000 : s * 0x7fff;
                }
                recordedBuffers.push(pcm16);
            };
            streamRef.current = stream;
            audioContextRef.current = audioContext;
            const mediaRecorder = new MediaRecorder(stream, {
                mimeType: 'audio/webm;codecs=opus',
                audioBitsPerSecond: SAMPLE_RATE,
            });
            mediaRecorderRef.current = mediaRecorder;
            mediaRecorder.onstop = () => {
                const blob = mergeBuffers(recordedBuffers);
                // Calculate actual recording duration with high precision
                const actualDuration = performance.now() - recordingStartTimeRef.current;
                // Automatically start processing with the recording duration
                processRecording(blob, actualDuration).catch(err => {
                    xtremConsole.error('Processing error:', err);
                    setError(`Processing failed: ${err instanceof Error ? err.message : 'Unknown error'}`);
                });
            };
            mediaRecorder.start();
            setIsRecording(true);
            setRecordingDuration(0);
            recordingStartTimeRef.current = performance.now();
            // Update duration timer
            durationIntervalRef.current = setInterval(() => {
                setRecordingDuration(prev => prev + 1);
            }, 1000);
        }
        catch (err) {
            setError('Failed to access microphone. Please check permissions.');
            xtremConsole.error('Recording error:', err);
        }
    }, [path, processRecording]);
    const stopRecording = React.useCallback(() => {
        if (mediaRecorderRef.current && isRecording) {
            mediaRecorderRef.current.stop();
            setIsRecording(false);
            if (streamRef.current) {
                streamRef.current.getTracks().forEach(track => track.stop());
            }
            if (processorRef.current) {
                processorRef.current.disconnect();
            }
            if (audioContextRef.current) {
                audioContextRef.current.close();
            }
            if (durationIntervalRef.current) {
                clearInterval(durationIntervalRef.current);
            }
        }
    }, [isRecording]);
    const onButtonClick = React.useCallback((event) => {
        if (isRecording) {
            stopRecording();
        }
        else {
            startRecording();
        }
    }, [isRecording, startRecording, stopRecording]);
    const tooltipMessage = isRecording
        ? localize('@sage/xtrem-ui/recording-voice', 'Listening... {{recordingDuration}}', {
            recordingDuration: formatDuration(recordingDuration),
        })
        : localize('@sage/xtrem-ui/record-voice', 'Start dictation');
    if (isProcessing && !isRecording) {
        return React.createElement(LoaderSpinner, { size: "small", showSpinnerLabel: false, marginRight: "8px", marginTop: "3px" });
    }
    return (React.createElement(IconButton, { "aria-label": tooltipMessage, onClick: onButtonClick, marginRight: "8px", "data-pendoid": "copilot-record-audio" },
        React.createElement(Icon, { tooltipVisible: isRecording, tooltipMessage: tooltipMessage, tooltipBgColor: isRecording ? 'var(--colorsSemanticNegative500)' : undefined, type: isRecording ? 'stop_circle' : 'microphone', color: isRecording ? 'var(--colorsSemanticNegative500)' : 'var(--colorsUtilityMajor300)' })));
}
//# sourceMappingURL=record-audio-button.js.map