import { isArray, isEmpty } from 'lodash';
import uid from 'uid';
export async function fetchConversationHistory(locale, path = '') {
    const response = await fetch(`${path}/chat/conversation-history`, {
        method: 'GET',
        headers: {
            'Accept-Language': locale,
        },
    });
    if (!response.ok) {
        throw new Error(`HTTP error, status: ${response.status}`);
    }
    const history = await response.json();
    return history.map(item => ({
        ...item,
        messages: item.messages.map(message => ({
            timestamp: message.timestamp,
            timeToFirstTokenMs: message.timeToFirstTokenMs,
            timeToLastTokenMs: message.timeToLastTokenMs,
            role: message.role,
            content: isArray(message.content) && message.content[0]?.type === 'text'
                ? message.content[0]?.text
                : message.content,
        })),
    }));
}
export async function fetchConversationStarterQuestions(locale, path = '') {
    const response = await fetch(`${path}/chat/conversation-starter`, {
        method: 'GET',
        headers: {
            'Accept-Language': locale,
        },
    });
    if (!response.ok) {
        throw new Error(`HTTP error, status: ${response.status}`);
    }
    return response.json();
}
export async function fetchConversationAccess(locale, path = '') {
    const response = await fetch(`${path}/chat/access`, {
        method: 'GET',
        headers: {
            'Accept-Language': locale,
        },
    });
    return response.json();
}
export async function sendTranscribeStreamingRequest({ blob, sampleRate, path = '', locale = 'en-US', onStreamEvent, }) {
    const formData = new FormData();
    formData.append('audio', blob, `recording-${Date.now()}.pcm`);
    formData.append('sampleRate', String(sampleRate));
    try {
        const response = await fetch(`${path}/chat/transcribe`, {
            method: 'POST',
            body: formData,
            headers: {
                'Accept-Language': locale,
                Accept: 'text/event-stream',
            },
        });
        if (!response.ok) {
            throw new Error(`HTTP error, status: ${response.status}`);
        }
        if (!response.body) {
            throw new Error('No response body available for streaming');
        }
        await processStreamResponse(response.body, onStreamEvent);
    }
    catch (error) {
        const errorMessage = error instanceof Error ? error.message : 'Unknown streaming error';
        onStreamEvent({ type: 'error', error: errorMessage });
    }
}
/**
 * Send a copilot message with streaming support using fetch API
 * This provides more control over the streaming process and better browser support
 */
export async function sendCopilotMessageStreaming(conversation, onStreamEvent, locale, conversationId, pageContext, abortSignal, path = '') {
    const updatedConversation = conversation.reduce((prevValue, c) => {
        // Ensure all messages are properly formatted
        const toolMap = c.toolEvents?.reduce((acc, toolEvent) => {
            const id = uid();
            return {
                ...acc,
                [id]: {
                    use: {
                        id,
                        type: 'tool_use',
                        name: toolEvent.toolName,
                        input: toolEvent.input,
                    },
                    result: {
                        tool_use_id: id,
                        type: 'tool_result',
                        content: toolEvent.result,
                    },
                },
            };
        }, {});
        return [
            ...prevValue,
            {
                ...c,
                toolEvents: undefined,
                content: c.content,
            },
            ...(isEmpty(toolMap)
                ? []
                : Object.keys(toolMap).reduce((prevValue, id) => {
                    return [
                        ...prevValue,
                        {
                            role: 'assistant',
                            content: [toolMap[id].use],
                        },
                        {
                            role: 'user',
                            content: [toolMap[id].result],
                        },
                    ];
                }, [])),
        ];
    }, []);
    try {
        const response = await fetch(`${path}/chat/stream`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                Accept: 'text/event-stream',
                'Accept-Language': locale,
            },
            body: JSON.stringify({
                conversationId,
                messages: updatedConversation,
                context: pageContext
                    ? {
                        nodeName: pageContext?.nodeName,
                        screenTitle: pageContext?.screenTitle,
                        recordId: pageContext?.recordFilter?._id,
                    }
                    : null,
            }),
            signal: abortSignal,
        });
        if (!response.ok) {
            throw new Error(`HTTP error, status: ${response.status}`);
        }
        if (!response.body) {
            throw new Error('No response body available for streaming');
        }
        return await processStreamResponse(response.body, onStreamEvent, updatedConversation);
    }
    catch (error) {
        const errorMessage = error instanceof Error ? error.message : 'Unknown streaming error';
        onStreamEvent({ type: 'error', error: errorMessage });
        throw error;
    }
}
/**
 * Process the streaming response from the server
 */
async function processStreamResponse(body, onStreamEvent, updatedConversation = []) {
    const reader = body.getReader();
    const decoder = new TextDecoder();
    let assistantMessage = '';
    try {
        let buffer = '';
        // eslint-disable-next-line no-constant-condition
        while (true) {
            const { done, value } = await reader.read();
            if (done) {
                break;
            }
            // Decode the chunk and add to buffer
            buffer += decoder.decode(value, { stream: true });
            // Process complete lines from buffer
            const lines = buffer.split('\n');
            buffer = lines.pop() || ''; // Keep incomplete line in buffer
            // Process each line and update assistant message
            const processedMessages = lines.map(line => processStreamLine(line, onStreamEvent));
            const completedMessage = processedMessages.find(msg => msg !== null);
            if (completedMessage !== undefined) {
                assistantMessage = completedMessage;
            }
        }
        return [...updatedConversation, { role: 'assistant', content: assistantMessage }];
    }
    finally {
        reader.releaseLock();
    }
}
/**
 * Process a single line from the stream
 * @returns The final message content if completed, null otherwise
 */
function processStreamLine(line, onStreamEvent) {
    if (!line.startsWith('data: ')) {
        return null;
    }
    const eventData = line.slice(6); // Remove 'data: ' prefix
    if (eventData === '[DONE]') {
        return null;
    }
    try {
        const streamEvent = JSON.parse(eventData);
        // Forward the event to the caller
        onStreamEvent(streamEvent);
        switch (streamEvent.type) {
            case 'text_chunk':
                return null; // Continue streaming
            case 'completed':
                return streamEvent.finalText;
            case 'error':
                throw new Error(streamEvent.error);
            case 'connected':
            case 'thinking':
            case 'tool_use':
            case 'tool_result':
                return null; // Continue streaming
            default:
                return null; // Unknown event type, continue
        }
    }
    catch (parseError) {
        // Silently ignore parse errors for malformed events
        return null;
    }
}
/**
 * Simple streaming function that accumulates text chunks and returns the final message
 * Use this when you want streaming but don't need to handle individual events
 */
export async function sendCopilotMessageStreamingSimple(conversation, locale, conversationId, onTextChunk, pageContext, abortSignal) {
    return sendCopilotMessageStreaming(conversation, (event) => {
        switch (event.type) {
            case 'text_chunk':
                if (onTextChunk) {
                    onTextChunk(event.text);
                }
                break;
            case 'error':
                throw new Error(event.error);
            // Handle other events silently
            default:
                break;
        }
    }, locale, conversationId, pageContext, abortSignal);
}
/**
 * Check if streaming is supported in the current environment
 */
export function isStreamingSupported() {
    return typeof fetch !== 'undefined' && typeof ReadableStream !== 'undefined';
}
//# sourceMappingURL=copilot-service.js.map