import { isEmpty, isString, trimEnd, trimStart } from 'lodash';
import * as React from 'react';
import { CollapsibleToolEvents } from './tool-events-component';
import * as showdown from 'showdown';
import { CopilotChart } from './copilot-chart';
import { useResizeObserver } from 'usehooks-ts';
import Profile from 'carbon-react/esm/components/profile';
import Loader from 'carbon-react/esm/components/loader';
import { getDateTimeString } from '../utils/date-utils';
const converter = new showdown.Converter({
    tables: true,
    tasklists: true,
    disableForced4SpacesIndentedSublists: true,
    emoji: true,
});
function MessageBody({ messageContent, isFullScreen, onClick, locale, isStreaming, timestamp, role, applicationContext, }) {
    const { displayName, profilePictureUrl } = applicationContext;
    const messageRef = React.useRef(null);
    const { width } = useResizeObserver({ ref: messageRef });
    const childrenElements = React.useMemo(() => {
        const chartComponents = messageContent.match(/```chart-json((\n|.)*?)```/gm) ?? [];
        const children = [];
        if (chartComponents.length > 0) {
            let key = 1;
            let text = messageContent;
            chartComponents.reverse().forEach((chartInputConfig, index) => {
                const after = messageContent.substring(text.lastIndexOf(chartInputConfig) + chartInputConfig.length);
                text = messageContent.substring(0, text.lastIndexOf(chartInputConfig));
                children.push(React.createElement("span", { key: `text-${key}`, 
                    // eslint-disable-next-line react/no-danger
                    dangerouslySetInnerHTML: {
                        __html: converter.makeHtml(after),
                    } }));
                key += 1;
                const trimmedChartInputConfig = trimEnd(trimStart(chartInputConfig, '```chart-json'), '```');
                const chartProps = JSON.parse(trimmedChartInputConfig);
                children.push(
                // eslint-disable-next-line react/jsx-props-no-spreading, react/no-array-index-key
                React.createElement(CopilotChart, { key: `chart-${key}`, ...chartProps, width: width, isFullScreen: isFullScreen }));
                key += 1;
            });
            children.push(React.createElement("span", { 
                // eslint-disable-next-line react/no-danger
                dangerouslySetInnerHTML: {
                    __html: converter.makeHtml(text),
                } }));
            children.reverse();
        }
        else {
            const chartPosition = messageContent.indexOf('```chart-json');
            if (chartPosition !== -1) {
                const textBefore = messageContent.substring(0, chartPosition);
                if (!isEmpty(textBefore)) {
                    children.push(React.createElement("span", { key: "text-before", 
                        // eslint-disable-next-line react/no-danger
                        dangerouslySetInnerHTML: {
                            __html: converter.makeHtml(textBefore),
                        } }));
                }
            }
            else {
                children.push(React.createElement("span", { 
                    // eslint-disable-next-line react/no-danger
                    dangerouslySetInnerHTML: {
                        __html: converter.makeHtml(messageContent),
                    } }));
            }
        }
        return children;
    }, [isFullScreen, messageContent, width]);
    const formattedDateTime = getDateTimeString(locale, timestamp);
    const copilotLogoSvg = React.useMemo(() => {
        return `${applicationContext.path ?? ''}/images/copilot-logo.svg`;
    }, [applicationContext.path]);
    return (React.createElement("div", { className: `e-copilot-message e-copilot-message-type-${role}`, ref: messageRef, onClick: onClick },
        React.createElement("div", { className: "e-copilot-message-container" },
            role === 'assistant' && (React.createElement("div", { className: "e-copilot-message-author e-copilot-message-author-type-assistant" },
                React.createElement(Profile, { size: "XS", name: "Sage Copilot", marginBottom: "8px", src: copilotLogoSvg }))),
            role === 'user' && (React.createElement("div", { className: "e-copilot-message-author e-copilot-message-author-type-user" },
                React.createElement(Profile, { size: "XS", name: displayName, marginBottom: "8px", src: profilePictureUrl }))),
            React.createElement("div", { className: "e-copilot-message-body" }, childrenElements),
            isStreaming && (React.createElement("span", { className: "e-copilot-streaming-cursor" },
                React.createElement(Loader, { variant: "gradient", size: "small" }))),
            formattedDateTime && React.createElement("div", { className: "e-copilot-message-timestamp" }, formattedDateTime))));
}
export function CopilotMessageBubble({ areToolsDisplayed, isFullScreen, onClick, message, locale, applicationContext, }) {
    const messageContent = message.content;
    if (isString(messageContent) && isEmpty(messageContent.trim())) {
        return null;
    }
    return (React.createElement(React.Fragment, null,
        areToolsDisplayed && message.toolEvents && message.toolEvents.length > 0 && (React.createElement("div", { className: "e-copilot-message e-copilot-message-type-assistant e-copilot-message-type-tools" },
            React.createElement(CollapsibleToolEvents, { toolEvents: message.toolEvents }))),
        isString(messageContent) && (React.createElement(MessageBody, { locale: locale, timestamp: message.timestamp, messageContent: messageContent, role: message.role, isFullScreen: isFullScreen, isStreaming: message.isStreaming, onClick: onClick, applicationContext: applicationContext }))));
}
//# sourceMappingURL=copilot-message-bubble.js.map