/* eslint-disable react/no-array-index-key */
import * as React from 'react';
import Textbox from 'carbon-react/esm/components/textbox';
import Button from 'carbon-react/esm/components/button';
import Message from 'carbon-react/esm/components/message';
import Loader from 'carbon-react/esm/components/loader';
import ButtonMinor from 'carbon-react/esm/components/button-minor';
import Icon from 'carbon-react/esm/components/icon';
import Link from 'carbon-react/esm/components/link';
import IconButton from 'carbon-react/esm/components/icon-button';
import * as tokens from '@sage/design-tokens/js/base/common';
import { xtremConsole } from '../utils/console';
import { CopilotIcon } from './copilot-icon';
import { sendCopilotMessageStreaming, isStreamingSupported, fetchConversationStarterQuestions, fetchConversationAccess, fetchConversationHistory, } from './copilot-service';
import { localize } from '../service/i18n-service';
import Switch from 'carbon-react/esm/components/switch';
import { isEmpty, trimStart } from 'lodash';
import { RecordAudioButton } from './record-audio-button';
import { useDeepCompareEffect } from '@sage/xtrem-ui-components';
import Pill from 'carbon-react/esm/components/pill';
import { toRelative } from '../component/field/relative-date/relative-date-utils';
import { CopilotMessageBubble } from './copilot-message-bubble';
import Tooltip from 'carbon-react/esm/components/tooltip';
import { getDateTimeString } from '../utils/date-utils';
import { Datetime, DateValue, Time } from '@sage/xtrem-date-time';
const TOOLS_DISPLAYED_STORAGE_KEY = 'e-copilot-tools-displayed';
const emptyAccess = {
    isAllowed: true,
    usage: { percent: 0 },
};
const getUsageColor = (percent) => {
    if (percent < 50)
        return tokens.colorsSemanticPositive500;
    if (percent < 80)
        return tokens.colorsSemanticFocus500;
    if (percent < 95)
        return tokens.colorsSemanticCaution500;
    return tokens.colorsSemanticNegative500;
};
// More aggressive coloring for days left in month. We want to start coloring as soon as only 80% of the remaining days are left for use.
const getDaysLeftColor = (percent) => {
    if (percent > 80)
        return tokens.colorsSemanticPositive500;
    if (percent > 40)
        return tokens.colorsSemanticFocus500;
    if (percent > 20)
        return tokens.colorsSemanticCaution500;
    return tokens.colorsSemanticNegative500;
};
const getDaysLeftInMonth = (usagePercent) => {
    const time0 = Time.make(0, 0, 0);
    const begOfMonth = DateValue.today().begOfMonth().at(time0);
    const endOfMonth = DateValue.today().endOfMonth().addDays(1).at(time0);
    const now = Datetime.now();
    const allMillisLeftInMonth = endOfMonth.millisDiff(now);
    const allMillisInMonth = endOfMonth.millisDiff(begOfMonth);
    const usageMillisLeftInMonth = (allMillisInMonth * (100 - usagePercent)) / 100;
    const allDays = Math.floor(allMillisLeftInMonth / (1000 * 60 * 60 * 24));
    const usageDays = Math.floor(usageMillisLeftInMonth / (1000 * 60 * 60 * 24));
    return { allDays, usageDays };
};
const defaultUsageDaysLeft = () => getDaysLeftInMonth(100);
export function CopilotComponent({ applicationContext, onClose, pageContext, }) {
    const locale = React.useMemo(() => applicationContext.locale ?? 'en-US', [applicationContext.locale]);
    const onTelemetryEvent = applicationContext.onTelemetryEvent;
    const bodyRef = React.useRef(null);
    const containerRef = React.useRef(null);
    const footerRef = React.useRef(null);
    const textboxRef = React.useRef(null);
    const [draftMessage, setDraftMessage] = React.useState('');
    const [conversationStarters, setConversationStarters] = React.useState([]);
    const [isTranscribingProcessing, setTranscribingProcessing] = React.useState(false);
    const [isFullScreen, setFullScreen] = React.useState(false);
    const [isConversationHistoryOpen, setConversationHistoryOpen] = React.useState(false);
    const [conversationHistory, setConversationHistory] = React.useState(null);
    const [conversation, setConversation] = React.useState([]);
    const [conversationId, setConversationId] = React.useState(undefined);
    const [hasConversationFailed, setConversationFailed] = React.useState(false);
    const [conversationAccess, setConversationAccess] = React.useState(emptyAccess);
    const [usageDaysLeft, setUsageDaysLeft] = React.useState(defaultUsageDaysLeft());
    const [abortController, setAbortController] = React.useState(null);
    const [areToolsDisplayed, setAreToolsDisplayed] = React.useState(() => localStorage.getItem(TOOLS_DISPLAYED_STORAGE_KEY) === 'true');
    const isProcessing = React.useMemo(() => {
        return !!conversation.slice(-1)[0]?.isStreaming;
    }, [conversation]);
    const processingGifPath = React.useMemo(() => {
        return `${applicationContext.path ?? ''}/images/copilot-processing.gif`;
    }, [applicationContext.path]);
    const copilotCssClasses = React.useMemo(() => {
        let classes = 'e-copilot';
        if (isFullScreen) {
            classes += ' e-copilot-full-screen';
        }
        return classes;
    }, [isFullScreen]);
    const onInputChange = React.useCallback((event) => {
        const value = event.target.value;
        setDraftMessage(value);
    }, []);
    const onToggleFullScreen = React.useCallback(() => {
        onTelemetryEvent?.('copilotToggleFullScreen', { isFullScreen });
        setFullScreen(prev => !prev);
    }, [isFullScreen, onTelemetryEvent]);
    const onOpenHistory = React.useCallback(() => {
        setConversationHistoryOpen(true);
        fetchConversationHistory(locale, applicationContext.path).then(history => {
            setConversationHistory(history);
        });
    }, [applicationContext.path, locale]);
    const onNewChat = React.useCallback(async () => {
        onTelemetryEvent?.('copilotNewConversation', { isFullScreen });
        // Clear conversation state
        setConversationFailed(false);
        setConversationAccess(emptyAccess);
        setUsageDaysLeft(defaultUsageDaysLeft());
        setConversation([]);
        setConversationId(undefined);
        setDraftMessage('');
        setTranscribingProcessing(false);
        setConversationHistoryOpen(false);
        // Scroll to top
        if (bodyRef.current) {
            bodyRef.current.scrollTop = 0;
        }
    }, [onTelemetryEvent, isFullScreen]);
    const onSendMessage = React.useCallback(async (starterMessage) => {
        const userMessage = starterMessage ?? draftMessage.trim();
        if (!userMessage || isProcessing || conversation.slice(-1)[0]?.isStreaming)
            return;
        onTelemetryEvent?.('copilotMessageSent', {});
        setDraftMessage(''); // Clear input immediately
        // Check if streaming is supported
        if (isStreamingSupported()) {
            // Create abort controller for cancellation
            const controller = new AbortController();
            setAbortController(controller);
            try {
                const newConversation = [
                    ...conversation,
                    { role: 'user', content: userMessage, timestamp: new Date().toISOString() },
                ];
                setConversation(newConversation);
                const onStreamEvent = (event) => {
                    switch (event.type) {
                        case 'text_chunk':
                            setConversation(prevConversation => {
                                const lastMessage = prevConversation[prevConversation.length - 1];
                                return [
                                    ...prevConversation.slice(0, -1),
                                    {
                                        ...lastMessage,
                                        content: `${lastMessage.content ?? ''}${event.text}`,
                                    },
                                ];
                            });
                            break;
                        case 'tool_use':
                            // Add tool use to the tracking array
                            const newToolEvent = {
                                type: 'tool_use',
                                toolName: event.toolName,
                                input: event.input,
                                status: 'running',
                            };
                            setConversation(prevConversation => {
                                const lastMessage = prevConversation[prevConversation.length - 1];
                                return [
                                    ...prevConversation.slice(0, -1),
                                    {
                                        ...lastMessage,
                                        toolEvents: [...(lastMessage.toolEvents ?? []), newToolEvent],
                                    },
                                ];
                            });
                            xtremConsole.log('Tool use:', event.toolName, event.input);
                            break;
                        case 'tool_result':
                            setConversation(prevConversation => {
                                const lastMessage = prevConversation[prevConversation.length - 1];
                                const toolEvents = lastMessage.toolEvents ?? [];
                                const updatedToolEvents = toolEvents.map((tool, index) => index === toolEvents.length - 1 && tool.toolName === event.toolName
                                    ? {
                                        ...tool,
                                        type: 'tool_result',
                                        result: event.result,
                                        status: event.status,
                                    }
                                    : tool);
                                return [
                                    ...prevConversation.slice(0, -1),
                                    {
                                        ...lastMessage,
                                        toolEvents: updatedToolEvents,
                                    },
                                ];
                            });
                            xtremConsole.log('Tool result:', event.toolName, event.result);
                            break;
                        case 'thinking':
                            xtremConsole.log('Thinking:', event.message);
                            break;
                        case 'access':
                            setConversationAccess(event.access);
                            setUsageDaysLeft(getDaysLeftInMonth(event.access.usage?.percent ?? 100));
                            break;
                        case 'error':
                            setConversation(prevConversation => {
                                const lastMessage = prevConversation[prevConversation.length - 1];
                                const toolEvents = lastMessage.toolEvents ?? [];
                                if (toolEvents.length === 0) {
                                    return prevConversation;
                                }
                                const updatedToolEvents = toolEvents.map((tool, index) => index === toolEvents.length - 1 ? { ...tool, status: 'error' } : tool);
                                return [
                                    ...prevConversation.slice(0, -1),
                                    {
                                        ...lastMessage,
                                        toolEvents: updatedToolEvents,
                                    },
                                ];
                            });
                            setConversationFailed(true);
                            xtremConsole.error('Stream error:', event.error);
                            break;
                        case 'completed':
                            setConversation(prevConversation => {
                                const lastMessage = prevConversation[prevConversation.length - 1];
                                return [
                                    ...prevConversation.slice(0, -1),
                                    {
                                        role: 'assistant',
                                        content: lastMessage.content,
                                        toolEvents: lastMessage.toolEvents,
                                        timestamp: event.timestamp,
                                        timeToFirstTokenMs: event.timeToFirstTokenMs,
                                        timeToLastTokenMs: event.timeToLastTokenMs,
                                    },
                                ];
                            });
                            setConversationId(event.conversationId);
                            setAbortController(null); // Clear abort controller on completion
                            break;
                        case 'connected':
                            setConversation(prevConversation => [
                                ...prevConversation,
                                { role: 'assistant', content: '', isStreaming: true },
                            ]);
                            xtremConsole.log('Stream connected');
                            break;
                        default:
                            // Handle any unknown event types
                            break;
                    }
                };
                await sendCopilotMessageStreaming(newConversation, onStreamEvent, locale, conversationId, pageContext, controller.signal, applicationContext.path);
            }
            catch (error) {
                if (error instanceof Error && error.name === 'AbortError') {
                    xtremConsole.log('Streaming request was cancelled');
                }
                else {
                    xtremConsole.error('Error in streaming message:', error);
                }
                setConversation(prevConversation => [...prevConversation].slice(0, -1));
            }
            finally {
                setAbortController(null);
            }
        }
    }, [
        draftMessage,
        isProcessing,
        conversation,
        onTelemetryEvent,
        locale,
        conversationId,
        pageContext,
        applicationContext.path,
    ]);
    const onTranscriptionFinished = React.useCallback((text) => {
        setTranscribingProcessing(false);
        onSendMessage(text);
    }, [onSendMessage]);
    const onTranscriptionStarted = React.useCallback(() => {
        setTranscribingProcessing(true);
    }, []);
    const onTranscriptionUpdated = React.useCallback((text) => {
        setDraftMessage(text);
    }, []);
    const onToolsToggle = React.useCallback((e) => {
        const checked = e.target.checked;
        onTelemetryEvent?.('copilotToggleToolsDisplayed', { isDisplayed: checked });
        setAreToolsDisplayed(checked);
        localStorage.setItem(TOOLS_DISPLAYED_STORAGE_KEY, String(checked));
    }, [onTelemetryEvent]);
    // Handle canceling/stopping the current processing operation
    const onCancelProcessing = React.useCallback(() => {
        if (abortController) {
            abortController.abort();
            setAbortController(null);
        }
        // Reset processing states
        if (isProcessing) {
            setConversation(prevConversation => [...prevConversation].slice(0, -1));
        }
        xtremConsole.log('Processing cancelled by user');
    }, [abortController, isProcessing]);
    // Handle canceling/stopping the current processing operation
    const onKeyDown = React.useCallback((event) => {
        if (event.key === 'Enter' && !(event.ctrlKey || event.metaKey)) {
            event.preventDefault(); // Prevent default Enter behavior (e.g., form submission)
            onSendMessage(); // Call the send message function
        }
    }, [onSendMessage]);
    const onConversationHistoryEntrySelected = React.useCallback((item) => () => {
        const conversationAgeInSeconds = (new Date().getTime() - new Date(item._createStamp).getTime()) / 1000;
        onTelemetryEvent?.('copilotConversationHistoryEntrySelected', { conversationAgeInSeconds });
        setConversationHistoryOpen(false);
        setConversation(item.messages);
        setConversationId(item.id);
        setConversationFailed(item.isFailed);
    }, [onTelemetryEvent]);
    const fullScreenButtonTooltip = React.useMemo(() => isFullScreen
        ? localize('@sage/xtrem-ui/copilot-minimize', 'Minimize chat window')
        : localize('@sage/xtrem-ui/copilot-maximize', 'Maximize chat window'), [isFullScreen]);
    const newConversationButtonTooltip = React.useMemo(() => localize('@sage/xtrem-ui/copilot-new-chat', 'New conversation'), []);
    const conversationHistoryButtonTooltip = React.useMemo(() => localize('@sage/xtrem-ui/copilot-conversation-history', 'Conversation history'), []);
    React.useEffect(() => {
        setTimeout(() => {
            if (bodyRef.current) {
                bodyRef.current.scrollTop = bodyRef.current.scrollHeight;
            }
        }, 100);
    }, [conversation, isProcessing, conversation]);
    // Save conversation to localStorage when it changes
    useDeepCompareEffect(() => {
        if (conversation.length === 0) {
            fetchConversationStarterQuestions(locale, applicationContext.path).then(response => {
                setConversationStarters(response);
            });
        }
        else {
            setConversationStarters([]);
        }
    }, [applicationContext.path, conversation, locale]);
    React.useEffect(() => {
        fetchConversationAccess(locale, applicationContext.path).then(access => {
            setConversationAccess(access);
            setUsageDaysLeft(getDaysLeftInMonth(access.usage?.percent ?? 100));
        });
    }, [applicationContext.path, locale]);
    const onChatbotLinkClicked = React.useCallback((event) => {
        const chatbotPrefix = '/hyperchat/';
        const downloadLinkPrefix = '/download';
        const anchor = event.target;
        if (!(anchor instanceof HTMLAnchorElement))
            return;
        const href = anchor.href;
        const isInternalLink = href.startsWith(window.location.origin);
        if (isInternalLink) {
            const internalPath = href.substring(window.location.origin.length);
            if (internalPath.startsWith(chatbotPrefix)) {
                applicationContext?.onTelemetryEvent?.('copilotResponseLinkClick', { type: 'hyperchat' });
                event.preventDefault();
                onSendMessage(atob(internalPath.substring(chatbotPrefix.length)).trim());
            }
            else if (internalPath.startsWith(downloadLinkPrefix)) {
                applicationContext?.onTelemetryEvent?.('copilotResponseLinkClick', { type: 'download' });
            }
            else {
                applicationContext?.onTelemetryEvent?.('copilotResponseLinkClick', { type: 'navigation' });
                event.preventDefault();
                applicationContext.handleNavigation(trimStart(internalPath, '/'), {});
            }
        }
    }, [applicationContext, onSendMessage]);
    const copilotTokenConsumptionMessage = localize('@sage/xtrem-ui/copilot-token-consumption', 'Sage Copilot consumption uses tokens that are found in your messages and responses. A token is typically a word or part of a word. Once you run out of tokens, you need to wait for you regain some tokens.');
    return (React.createElement("div", { ref: containerRef, className: copilotCssClasses },
        React.createElement("div", { className: "e-copilot-header", id: "copilot-window", role: "dialog", "aria-modal": "false", "aria-label": "Sage Copilot" },
            React.createElement("div", { className: "e-copilot-header-logo" },
                React.createElement(CopilotIcon, { height: 24 })),
            React.createElement("div", { className: "e-copilot-header-title" }, "Sage Copilot"),
            React.createElement("div", { className: "e-copilot-header-actions" },
                conversation.length > 0 && (React.createElement(IconButton, { "aria-label": newConversationButtonTooltip, onClick: onNewChat },
                    React.createElement(Icon, { tooltipMessage: newConversationButtonTooltip, type: "plus", color: "var(--colorsYang100)" }))),
                React.createElement(IconButton, { "aria-label": conversationHistoryButtonTooltip, onClick: onOpenHistory },
                    React.createElement(Icon, { tooltipMessage: conversationHistoryButtonTooltip, type: "refresh_clock", color: "var(--colorsYang100)" })),
                React.createElement(IconButton, { "aria-label": fullScreenButtonTooltip, onClick: onToggleFullScreen },
                    React.createElement(Icon, { tooltipMessage: fullScreenButtonTooltip, type: isFullScreen ? 'normalscreen' : 'fullscreen', color: "var(--colorsYang100)" })),
                React.createElement(IconButton, { "aria-label": localize('@sage/xtrem-ui/action-close', 'Close'), onClick: onClose },
                    React.createElement(Icon, { tooltipMessage: localize('@sage/xtrem-ui/action-close', 'Close'), type: "close", color: "var(--colorsYang100)" })))),
        !isConversationHistoryOpen && (React.createElement(React.Fragment, null,
            React.createElement("div", { className: "e-copilot-body", ref: bodyRef, role: "log", "aria-live": "polite" },
                isEmpty(conversationStarters) && isEmpty(conversation) && (React.createElement(Loader, { variant: "gradient", size: "small" })),
                conversationStarters.map(q => (React.createElement(ButtonMinor, { buttonType: "tertiary", key: q, marginBottom: "24px", onClick: () => onSendMessage(q), size: "small" }, q))),
                conversation.map((message, index) => (React.createElement(CopilotMessageBubble, { areToolsDisplayed: areToolsDisplayed, isFullScreen: isFullScreen, key: `message-${index}`, locale: locale, message: message, onClick: onChatbotLinkClicked, applicationContext: applicationContext }))),
                hasConversationFailed && (React.createElement(Message, { variant: "error" }, localize('@sage/xtrem-ui/copilot-conversation-failed', 'Conversation failed. Try starting a new one.'))),
                !conversationAccess.isAllowed && (React.createElement(Message, { variant: "error" }, conversationAccess.reason || 'Unknown reason')),
                conversationAccess.usage?.blockedUntil && (React.createElement(Message, { variant: "error" }, localize('@sage/xtrem-ui/blocked-until', 'You have run out of tokens. You are blocked until {{0}}', [`${getDateTimeString(locale, conversationAccess.usage.blockedUntil)}`]))),
                !hasConversationFailed && isProcessing && (React.createElement("div", { className: "e-copilot-processing" },
                    React.createElement("img", { className: "e-copilot-processing-image", src: processingGifPath, alt: localize('@sage/xtrem-ui/copilot-processing', 'Processing...') }),
                    React.createElement("span", null, localize('@sage/xtrem-ui/copilot-processing', 'Processing...')),
                    React.createElement(ButtonMinor, { buttonType: "tertiary", className: "e-copilot-cancel-button", iconType: "cross", marginLeft: "12px", onClick: onCancelProcessing, size: "small" }, localize('@sage/xtrem-ui/cancel', 'Cancel'))))),
            React.createElement("div", { className: "e-copilot-footer", ref: footerRef },
                React.createElement("div", { ref: textboxRef },
                    React.createElement(Textbox, { onChange: onInputChange, value: draftMessage, marginX: "24px", placeholder: localize('@sage/xtrem-ui/copilot-placeholder-type-your-message', 'Type your message...'), disabled: hasConversationFailed || isTranscribingProcessing, onKeyDown: onKeyDown },
                        React.createElement(RecordAudioButton, { applicationContext: applicationContext, onTelemetryEvent: onTelemetryEvent, onTranscriptionFinished: onTranscriptionFinished, onTranscriptionStarted: onTranscriptionStarted, onTranscriptionUpdated: onTranscriptionUpdated }))),
                React.createElement("div", { className: "e-copilot-footer-actions" },
                    React.createElement("span", { className: "e-copilot-usage" },
                        React.createElement(Tooltip, { message: copilotTokenConsumptionMessage },
                            React.createElement("span", null,
                                localize('@sage/xtrem-ui/copilot-usage', 'Usage:'),
                                React.createElement("span", { className: "e-copilot-usage-amount", style: { color: getUsageColor(conversationAccess.usage?.percent || 0) } }, localize('@sage/xtrem-ui/copilot-usage-amount', '{{0}}%', [
                                    Math.floor((conversationAccess.usage?.percent || 0) * 100) / 100,
                                ])),
                                React.createElement("span", { className: "e-copilot-usage-amount", style: {
                                        color: getDaysLeftColor((usageDaysLeft.usageDays / usageDaysLeft.allDays) * 100),
                                    } }, usageDaysLeft.usageDays < usageDaysLeft.allDays
                                    ? localize('@sage/xtrem-ui/copilot-est-days-left', '{{usageDays}}/{{allDays}} est. days left', usageDaysLeft)
                                    : '')))),
                    React.createElement(Switch, { label: localize('@sage/xtrem-ui/copilot-tools-displayed', 'Tools displayed:'), labelInline: true, checked: areToolsDisplayed, onChange: onToolsToggle, marginRight: "12px" }),
                    React.createElement(Button, { buttonType: "primary", "data-pendoid": "copilot-send-message", disabled: hasConversationFailed || !draftMessage.trim() || isProcessing, iconPosition: "after", iconType: "send", onClick: () => onSendMessage(), size: "small" }, localize('@sage/xtrem-ui/copilot-send-message', 'Send'))),
                React.createElement("div", { className: "e-copilot-disclaimer" },
                    React.createElement("span", null, localize('@sage/xtrem-ui/copilot-disclaimer', 'Check AI content for inaccuracies. This content is for information purposes and is not professional advice.')),
                    ' ',
                    React.createElement(Link, { href: "#", target: "_blank", rel: "noopener", iconAlign: "right", icon: "link" }, localize('@sage/xtrem-ui/copilot-privacy notice', 'Privacy notice')))))),
        isConversationHistoryOpen && (React.createElement("div", { className: "e-copilot-conversation-history" },
            React.createElement("ul", { className: "e-copilot-conversation-history-list" }, conversationHistory?.map(item => (React.createElement("li", { key: item.id, className: "e-copilot-conversation-history-item" },
                React.createElement("button", { className: "e-copilot-conversation-history-item-button", type: "button", onClick: onConversationHistoryEntrySelected(item) },
                    React.createElement("span", { className: "e-copilot-conversation-history-item-title" },
                        item.title,
                        item.isFailed && (React.createElement(Icon, { marginLeft: "16px", tooltipMessage: localize('@sage/xtrem-ui/chatbot-conversation-failed', 'Conversation failed'), type: "error", color: tokens.colorsSemanticNegative500 }))),
                    React.createElement("span", { className: "e-copilot-conversation-history-item-date" },
                        React.createElement(Pill, { borderColor: tokens.colorsYang100, fill: true, size: "S" }, toRelative('datetime', new Date(item._createStamp)) || '')))))))))));
}
//# sourceMappingURL=copilot-component.js.map