import { useDeepCompareMemo } from '@sage/xtrem-ui-components';
import Icon from 'carbon-react/esm/components/icon';
import IconButton from 'carbon-react/esm/components/icon-button';
import { isNil } from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import * as xtremRedux from '../../../../redux';
import { calculateActionMenuWithSeparator } from '../../../../utils/action-menu-utils';
import { INTERNAL_COLUMN_IDS } from '../../../../utils/ag-grid/ag-grid-column-config';
import { callGridMethod, tryToCommitPhantomRow } from '../../../../utils/ag-grid/ag-grid-table-utils';
import { triggerHandledEvent } from '../../../../utils/events';
import { getPageDefinitionFromState } from '../../../../utils/state-utils';
import { splitValueToMergedValue } from '../../../../utils/transformers';
import { navigationPanelId } from '../../../container/navigation-panel/navigation-panel-types';
import XtremActionPopover from '../../xtrem-action-popover';
import { TABLE_INLINE_ACTION_CSS_CLASS } from '../../../../utils/constants';
import { localize } from '../../../../service/i18n-service';
import { TAB } from '../../../../utils/keyboard-event-utils';
export function TableDropdownActions({ accessBindings, api, column, fieldId, fieldProperties, hasInlineActions, isDisabled, isPhantomRow, level, recordId, rowIndex, rowValue, screenId, tableData, onTelemetryEvent, }) {
    const actionType = 'table-dropdown-action';
    const isNestedGrid = React.useMemo(() => Object.prototype.hasOwnProperty.call(fieldProperties, 'levels'), [fieldProperties]);
    // rowValue is an object so "useDeepCompareMemo" is needed
    const onClickMenu = useDeepCompareMemo(() => (context, onClick, onError) => {
        if (onTelemetryEvent) {
            onTelemetryEvent(`tableDropdownActionTriggered-${context.uniqueId}`, {
                screenId,
                elementId: fieldId,
                recordId,
                id: context.id,
                uniqueId: context.uniqueId,
            });
        }
        if (!onClick) {
            return;
        }
        const ancestorIds = tableData.getAncestorIds({
            id: recordId,
            level,
        });
        if (isNestedGrid) {
            triggerHandledEvent(screenId, fieldId, { onClick, onError: onError || fieldProperties.onError }, recordId, splitValueToMergedValue(rowValue), level, ancestorIds);
        }
        else {
            triggerHandledEvent(screenId, fieldId, { onClick, onError: onError || fieldProperties.onError }, recordId, splitValueToMergedValue(rowValue));
        }
    }, [fieldId, fieldProperties.onError, isNestedGrid, level, recordId, rowValue, screenId, tableData]);
    const menuItems = React.useMemo(() => {
        const actions = isNestedGrid && fieldProperties.levels
            ? fieldProperties.levels[level].dropdownActions
            : fieldProperties.dropdownActions;
        if (actions) {
            const rowData = splitValueToMergedValue(rowValue);
            if (!rowData) {
                return [];
            }
            return calculateActionMenuWithSeparator({
                accessBindings,
                actions,
                actionType,
                onTriggerMenuItem: onClickMenu,
                rowValue: rowData,
                screenId,
                pendoId: `tableDropdownActionsButton-${screenId}-${fieldId}-${level}`,
            });
        }
        return [];
    }, [
        isNestedGrid,
        fieldProperties.levels,
        fieldProperties.dropdownActions,
        level,
        rowValue,
        accessBindings,
        onClickMenu,
        screenId,
        fieldId,
    ]);
    // Helper function to get platform-specific key combination for tooltips
    const getCommitKeyboardShortcut = React.useCallback(() => {
        const isOSX = navigator?.userAgentData?.platform === 'macOS' || navigator.platform?.startsWith('Mac');
        return isOSX ? 'Command+Enter' : 'Ctrl+Enter';
    }, []);
    const renderPhantomRowActionButtons = React.useCallback(() => {
        const row = tableData.getRawRecord({
            id: recordId,
            level,
            cleanMetadata: false,
        });
        const handleCommitPhantomRow = async (event) => {
            if (isNil(api)) {
                return;
            }
            // Prevent event propagation to avoid interference
            event?.preventDefault();
            event?.stopPropagation();
            // Stop editing to ensure current cell data is saved before commit
            callGridMethod(api, 'stopEditing');
            onTelemetryEvent?.(`tablePhantomRowCommittedByButton-${fieldId}`, {
                screenId,
                elementId: fieldId,
            });
            await tryToCommitPhantomRow({
                api,
                screenId,
                elementId: fieldId,
                value: xtremRedux.getStore().getState().screenDefinitions[screenId].values[fieldId],
            }).then(() => {
                // jump to first available column excluding selection column
                const firstColumn = (callGridMethod(api, 'getAllDisplayedColumns') ?? []).find(c => !INTERNAL_COLUMN_IDS.includes(c.getColId()));
                if (isNil(firstColumn)) {
                    return;
                }
                setTimeout(() => {
                    callGridMethod(api, 'startEditingCell', {
                        colKey: firstColumn.getColId(),
                        rowIndex: 0,
                        key: undefined,
                        rowPinned: 'top',
                    });
                }, 100);
            });
        };
        const handleClearPhantomRow = async (event) => {
            // Prevent event propagation to avoid interference
            event?.preventDefault();
            event?.stopPropagation();
            await tableData.resetRowToDefaults({
                id: row._id,
                level: row.__level,
                parentId: row.__parentId,
                isOrganicChange: true,
                resetDirtiness: true,
            });
            // Keep focus on minus button after clearing
            setTimeout(() => {
                if (api) {
                    callGridMethod(api, 'setFocusedCell', 0, column?.getColId() || 'actions', 'top');
                }
            }, 100);
        };
        const commitTooltip = localize('@sage/xtrem-ui/phantom-row-confirm-tooltip', 'Press {{keyboardShortcut}} to add.', {
            keyboardShortcut: getCommitKeyboardShortcut(),
        });
        const clearTooltip = localize('@sage/xtrem-ui/phantom-row-clear-tooltip', 'Clear line');
        return (React.createElement(React.Fragment, null,
            React.createElement(IconButton
            // @ts-expect-error className not included in the carbon IconButton typings
            , { 
                // @ts-expect-error className not included in the carbon IconButton typings
                className: TABLE_INLINE_ACTION_CSS_CLASS, onClick: handleCommitPhantomRow, disabled: false, "aria-label": "Confirm", "data-testid": "e-confirm-phantom-row" },
                React.createElement(Icon, { tooltipMessage: commitTooltip, type: "plus", fontSize: "small", "data-element": "plus", color: "--colorsActionMinor500" })),
            React.createElement(IconButton
            // @ts-expect-error className not included in the carbon IconButton typings
            , { 
                // @ts-expect-error className not included in the carbon IconButton typings
                className: TABLE_INLINE_ACTION_CSS_CLASS, onClick: handleClearPhantomRow, disabled: false, "aria-label": "Clear", "data-testid": "e-clear-phantom-row", onKeyDown: e => {
                    if (e.code !== TAB) {
                        return;
                    }
                    if (isNil(api) || isNil(column) || isNil(rowIndex)) {
                        return;
                    }
                    if (!e.shiftKey) {
                        const phantomRow = api.getPinnedTopRow(0);
                        if (phantomRow?.data?.__dirty) {
                            onTelemetryEvent?.(`tablePhantomRowCommittedByBlur-${fieldId}`, {
                                screenId,
                                elementId: fieldId,
                            });
                            tryToCommitPhantomRow({
                                api,
                                screenId,
                                elementId: fieldId,
                                value: xtremRedux.getStore().getState().screenDefinitions[screenId].values[fieldId],
                            }).then(() => {
                                // jump to first available column excluding selection column
                                const firstColumn = (callGridMethod(api, 'getAllDisplayedColumns') ?? []).find(c => !INTERNAL_COLUMN_IDS.includes(c.getColId()));
                                if (isNil(firstColumn)) {
                                    return;
                                }
                                setTimeout(() => {
                                    callGridMethod(api, 'startEditingCell', {
                                        colKey: firstColumn.getColId(),
                                        rowIndex: 0,
                                        key: undefined,
                                        rowPinned: 'top',
                                    });
                                }, 100);
                            });
                        }
                    }
                } },
                React.createElement(Icon, { tooltipMessage: clearTooltip, type: "minus", fontSize: "small", "data-element": "minus", color: "--colorsActionMinor500" }))));
    }, [
        api,
        column,
        fieldId,
        level,
        onTelemetryEvent,
        recordId,
        rowIndex,
        screenId,
        tableData,
        getCommitKeyboardShortcut,
    ]);
    if (tableData) {
        if (isPhantomRow) {
            return renderPhantomRowActionButtons();
        }
        return (React.createElement(XtremActionPopover, { items: menuItems, isDisabled: isDisabled, api: api, column: column, rowIndex: rowIndex, hasInlineActions: hasInlineActions, pendoId: `tableDropdownActionsButton-${screenId}-${fieldId}-${level}` }));
    }
    return null;
}
const mapStateToProps = (state, props) => {
    const screenDefinition = getPageDefinitionFromState(props.screenId, state);
    const fieldProperties = screenDefinition.metadata.uiComponentProperties[props.fieldId];
    const tableData = props.fieldId === navigationPanelId
        ? screenDefinition.navigationPanel?.value
        : screenDefinition.values[props.fieldId];
    return {
        ...props,
        accessBindings: screenDefinition.accessBindings,
        fieldProperties,
        onTelemetryEvent: state.applicationContext?.onTelemetryEvent,
        tableData,
    };
};
export default connect(mapStateToProps)(TableDropdownActions);
//# sourceMappingURL=table-dropdown-actions.js.map