import { RANGE } from '@sage/xtrem-shared';
import React, { useEffect, useMemo, useRef } from 'react';
import { includes } from 'lodash';
import { getModelFromDateState, getDateStateFromModel, shouldTriggerModelChange, dateRanges, } from './date-filter-utils';
import { DateFilterSelect } from './date-filter-select';
import { Time } from '@sage/xtrem-date-time';
import { DatetimeInputComponent } from '../../../datetime/datetime-input-component';
import { makeDatetime } from '../../../datetime/datetime-utils';
import { localize } from '../../../../../service/i18n-service';
export default function DatetimeFilter({ model, onModelChange, locale }) {
    const dateTimeRef = useRef(null);
    const dateTimeToRef = useRef(null);
    const [dateState, setDateState] = React.useState(() => getDateStateFromModel(model));
    const [time, setTime] = React.useState(dateState.formattedValue || null);
    const [timeTo, setTimeTo] = React.useState(dateState.formattedValueTo || null);
    const [date, setDate] = React.useState(null);
    const [dateTo, setDateTo] = React.useState(null);
    const [isPopoverOpen, setIsPopoverOpen] = React.useState(false);
    const [isPopoverToOpen, setIsPopoverToOpen] = React.useState(false);
    const isPredefinedRange = useMemo(() => includes(dateRanges, dateState.type), [dateState.type]);
    const isRange = useMemo(() => dateState.type === RANGE || isPredefinedRange, [dateState.type, isPredefinedRange]);
    const onChange = React.useCallback((newValue) => {
        if (newValue.rawValue?.toString() === dateState.rawValue?.toString()) {
            return;
        }
        setDateState(current => {
            // On type change we reset everything.
            if (newValue.type && newValue.type !== current.type) {
                newValue.rawValue = '';
                newValue.rawValueTo = '';
                newValue.formattedValue = '';
                newValue.formattedValueTo = '';
                onModelChange(null);
            }
            const updatedState = { ...current, ...newValue };
            if (shouldTriggerModelChange(newValue, updatedState)) {
                // Just check if the updated state is essentially empty
                const isEmpty = !updatedState.rawValue && !updatedState.rawValueTo && !updatedState.formattedValue;
                if (isEmpty && !includes(dateRanges, updatedState.type)) {
                    onModelChange(null);
                }
                else {
                    onModelChange(getModelFromDateState({ dateState: updatedState, locale, isDatetime: true }));
                }
            }
            return updatedState;
        });
    }, [onModelChange, locale, dateState.rawValue]);
    useEffect(() => {
        if (time && date && !isRange) {
            const newValue = makeDatetime(date, Time.parse(time));
            onChange({
                rawValue: newValue.toString(),
            });
        }
        else if (isRange && time && date && timeTo && dateTo) {
            const newValue = makeDatetime(date, Time.parse(time));
            const newValueTo = makeDatetime(dateTo, Time.parse(timeTo));
            onChange({
                rawValue: newValue.toString(),
                rawValueTo: newValueTo.toString(),
            });
        }
        else if ((!time || !date) && dateState.rawValue) {
            onChange({ rawValue: '' });
        }
    }, [time, date, onChange, timeTo, dateTo, isRange, dateState.rawValue]);
    const handleDateChange = (date) => {
        setDate(date);
    };
    const handleTimeChange = (time) => {
        setTime(time);
    };
    const handleDateToChange = (date) => {
        setDateTo(date);
    };
    const handleTimeToChange = (time) => {
        setTimeTo(time);
    };
    const onFilterTypeChanged = React.useCallback((item) => {
        const newValue = item.value;
        setDateState(current => ({ ...current, type: newValue }));
        onChange({ type: newValue });
        setTime(null);
        setTimeTo(null);
        setDate(null);
        setDateTo(null);
    }, [onChange]);
    return (React.createElement("div", { className: "e-filter-date-container" },
        React.createElement(DateFilterSelect, { onChange: onFilterTypeChanged, value: dateState.type }),
        React.createElement("div", { className: "e-date-input-wrapper", ref: dateTimeRef },
            React.createElement(DatetimeInputComponent, { "aria-label": localize('@sage/xtrem-ui/date-time-component-aria-label', 'Select date and time'), inputRef: dateTimeRef, date: date, time: time, locale: locale, onDateChange: handleDateChange, onTimeChange: handleTimeChange, isReadOnly: isPredefinedRange, size: "small", isPopoverOpen: isPopoverOpen, onPopperOpenChange: isOpen => setIsPopoverOpen(isOpen), rangeStartDate: null, type: "single" })),
        isRange && (React.createElement("div", { className: "e-date-input-wrapper", ref: dateTimeToRef },
            React.createElement(DatetimeInputComponent, { "aria-label": localize('@sage/xtrem-ui/date-time-component-aria-label', 'Select date and time'), inputRef: dateTimeToRef, date: dateTo, time: timeTo, locale: locale, onDateChange: handleDateToChange, onTimeChange: handleTimeToChange, isReadOnly: isPredefinedRange, size: "small", isPopoverOpen: isPopoverToOpen, onPopperOpenChange: isOpen => setIsPopoverToOpen(isOpen), rangeStartDate: null, type: "single" })))));
}
//# sourceMappingURL=datetime-filter.js.map