import { RANGE } from '@sage/xtrem-shared';
import CarbonDate from 'carbon-react/esm/components/date';
import React from 'react';
import { isEqual, includes } from 'lodash';
import { getModelFromDateState, getDateStateFromModel, agGridDateFilterTypes, shouldTriggerModelChange, dateRanges, } from './date-filter-utils';
import { useDeepCompareEffect } from '@sage/xtrem-ui-components';
import { DateFilterSelect } from './date-filter-select';
export default function DateFilter({ model, onModelChange, locale }) {
    const [dateState, setDateState] = React.useState(() => getDateStateFromModel(model));
    useDeepCompareEffect(() => {
        const newDateState = getDateStateFromModel(model);
        if (!isEqual(dateState, newDateState)) {
            setDateState(current => agGridDateFilterTypes.includes(current.type) ? newDateState : { ...newDateState, type: current.type });
        }
    }, [model]);
    const onChange = React.useCallback((newValue) => {
        setDateState(current => {
            // On type change we reset everything.
            if (newValue.type && newValue.type !== current.type) {
                newValue.rawValue = '';
                newValue.rawValueTo = '';
                newValue.formattedValue = '';
                newValue.formattedValueTo = '';
                onModelChange(null);
            }
            const updatedState = { ...current, ...newValue };
            if (shouldTriggerModelChange(newValue, updatedState)) {
                const isEmpty = !updatedState.rawValue && !updatedState.rawValueTo && !updatedState.formattedValue;
                if (isEmpty && !includes(dateRanges, updatedState.type)) {
                    onModelChange(null);
                }
                else {
                    onModelChange(getModelFromDateState({ dateState: updatedState, locale }));
                }
            }
            return updatedState;
        });
    }, [onModelChange, locale]);
    const handleChange = React.useCallback((event) => {
        const newValue = event.target.value.rawValue;
        const newFormattedValue = event.target.value.formattedValue;
        onChange({ rawValue: newValue, formattedValue: newFormattedValue });
    }, [onChange]);
    const handleChangeTo = React.useCallback((event) => {
        const newValue = event.target.value.rawValue;
        const newFormattedValue = event.target.value.formattedValue;
        onChange({ rawValueTo: newValue, formattedValueTo: newFormattedValue });
    }, [onChange]);
    const onFilterTypeChanged = React.useCallback((item) => {
        const newValue = item.value;
        onChange({ type: newValue });
    }, [onChange]);
    const isPredefinedRange = React.useMemo(() => includes(dateRanges, dateState.type), [dateState.type]);
    const isRange = React.useMemo(() => dateState.type === RANGE || isPredefinedRange, [dateState.type, isPredefinedRange]);
    return (React.createElement("div", { className: "e-filter-date-container" },
        React.createElement(DateFilterSelect, { onChange: onFilterTypeChanged, value: dateState.type }),
        React.createElement(CarbonDate, { readOnly: isPredefinedRange, mt: "6px", "data-testid": "e-date-filter", allowEmptyValue: true, onChange: handleChange, value: dateState.formattedValue, size: "small", autoComplete: "off", disablePortal: true, inputWidth: 100 }),
        isRange && (React.createElement(CarbonDate, { readOnly: isPredefinedRange, mt: "6px", allowEmptyValue: true, onChange: handleChangeTo, value: dateState.formattedValueTo ?? '', size: "small", autoComplete: "off", disablePortal: true, inputWidth: 100 }))));
}
//# sourceMappingURL=date-filter.js.map