import { getDateRange, isValidIsoDate } from '@sage/xtrem-date-time';
import { EQUALS, GREATER_THAN, GREATER_THAN_EQUAL, LESS_THAN, LESS_THAN_EQUAL, NOT_EQUALS, RANGE, } from '@sage/xtrem-shared';
import { includes, isNil } from 'lodash';
export const isSimpleDateFilter = (filter) => {
    return !Object.prototype.hasOwnProperty.call(filter, 'operator');
};
export const agGridDateFilterTypes = [
    EQUALS,
    NOT_EQUALS,
    LESS_THAN,
    GREATER_THAN,
    LESS_THAN_EQUAL,
    GREATER_THAN_EQUAL,
    RANGE,
];
export const defaultDateState = () => ({
    rawValue: '',
    formattedValue: '',
    rawValueTo: '',
    formattedValueTo: '',
    type: EQUALS,
});
export const dateRanges = [
    'last-30-days',
    'last-7-days',
    'next-day',
    'next-month',
    'next-week',
    'next-year',
    'previous-day',
    'previous-month',
    'previous-week',
    'previous-year',
    'same-day',
    'same-month',
    'same-week',
    'same-year',
];
export const getModelFromDateState = ({ dateState, locale, isDatetime = false, }) => {
    if (includes(dateRanges, dateState.type)) {
        const dateRange = getDateRange({
            date: new Date().toISOString().slice(0, 10),
            range: dateState.type,
            locale,
        });
        const dateFrom = isDatetime
            ? (dateRange.start?.toJsDate().toISOString() ?? '')
            : (dateRange.start?.format('YYYY-MM-DD') ?? '');
        const dateTo = isDatetime
            ? (dateRange.end?.toJsDate().toISOString() ?? '')
            : (dateRange.end?.format('YYYY-MM-DD') ?? '');
        return {
            filterType: 'date',
            dateFrom,
            dateTo,
            type: RANGE,
            formattedValue: dateRange.start?.toString() ?? '',
            formattedValueTo: dateRange.end?.toString() ?? '',
        };
    }
    return {
        filterType: 'date',
        dateFrom: dateState.rawValue,
        dateTo: dateState.rawValueTo,
        type: dateState.type,
        formattedValue: dateState.formattedValue,
        formattedValueTo: dateState.formattedValueTo,
    };
};
export function getDateStateFromModel(model) {
    const value = defaultDateState();
    if (!model || !isSimpleDateFilter(model)) {
        return value;
    }
    if (!isNil(model.dateFrom)) {
        value.rawValue = model.dateFrom;
    }
    if (!isNil(model.dateTo)) {
        value.rawValueTo = model.dateTo;
    }
    if (!isNil(model.formattedValue)) {
        value.formattedValue = model.formattedValue;
    }
    if (!isNil(model.formattedValueTo)) {
        value.formattedValueTo = model.formattedValueTo;
    }
    if (!isNil(model.type)) {
        value.type = model.type;
    }
    return value;
}
/**
 * Determines if a model change should be triggered based on the filter state.
 * Prevents premature triggering for range filters that don't have both values set.
 */
export const shouldTriggerModelChange = (newValue, updatedState) => {
    const values = Object.values(newValue ?? {});
    // Clear filter if no values
    if (values.filter(Boolean).length === 0) {
        return true; // Trigger to clear the filter
    }
    // Check if we should trigger the model change
    const hasValidValue = values.length > 0 &&
        values.some(v => isValidIsoDate(v) || (newValue.type && includes(dateRanges, newValue.type)));
    if (!hasValidValue) {
        return false;
    }
    // For range filters, ensure both values are present before triggering
    const isRangeType = updatedState.type === RANGE || includes(dateRanges, updatedState.type);
    if (isRangeType) {
        // Only trigger if both range values are present and valid, or if it's a predefined range
        const hasBothRangeValues = Boolean(updatedState.rawValue && updatedState.rawValueTo);
        const isPredefinedRange = includes(dateRanges, updatedState.type);
        return isPredefinedRange || hasBothRangeValues;
    }
    // For non-range filters, trigger immediately
    return true;
};
//# sourceMappingURL=date-filter-utils.js.map