import { objectKeys } from '@sage/xtrem-shared';
import { camelCase, isEmpty, noop } from 'lodash';
import * as React from 'react';
import { useSelector, useStore } from 'react-redux';
import { confirmationDialog, pageDialog } from '../../../service/dialog-service';
import { queryToGraphQuery } from '../../../service/graphql-utils';
import { localize } from '../../../service/i18n-service';
import { showToast } from '../../../service/toast-service';
import { getSelectionFilter } from '../../../utils/ag-grid/ag-grid-table-utils';
import { getBulkActionAccessStatus } from '../../../utils/access-utils';
import { BULK_ACTION_ASYNC_EXPORT, QUERY_PARAM_PRINTING_NODE_TYPE, QUERY_PARAM_PRINTING_SOURCE_PAGE, } from '../../../utils/constants';
import { getDataTestIdAttribute } from '../../../utils/dom';
import { useDeepEqualSelector } from '../../../utils/hooks/use-deep-equal-selector';
import { getNavigationPanelState, getNavigationPanelTablePropertiesFromPageDefinition, getPageDefinitionFromState, } from '../../../utils/state-utils';
import { ButtonComponent } from '../../field/button/button-component';
export function DesktopTableBulkActionBar({ bulkActions, gridApi, isSelectAllChecked, selectedRowCount, screenId, onClearSelection, groupByColumn, onTelemetryEvent, }) {
    const store = useStore();
    const contextPage = useSelector(s => getPageDefinitionFromState(screenId, s).path);
    const exportTemplatesByNode = useDeepEqualSelector(s => getPageDefinitionFromState(screenId, s).metadata.exportTemplatesByNode);
    const graphApi = useDeepEqualSelector(s => getPageDefinitionFromState(screenId, s).page.$.graph);
    const locale = useDeepEqualSelector(s => s.applicationContext?.locale);
    const tableFieldProperties = useDeepEqualSelector(s => getNavigationPanelTablePropertiesFromPageDefinition(getPageDefinitionFromState(screenId, s)));
    const activeOptionsMenuItem = useDeepEqualSelector(s => getNavigationPanelState(screenId, s)?.value?.getActiveOptionsMenuItem());
    const pageDefinition = useDeepEqualSelector(s => getPageDefinitionFromState(screenId, s));
    const node = React.useMemo(() => {
        const pageNode = tableFieldProperties.node?.toString();
        if (!pageNode) {
            throw new Error(`Page ${screenId} does not have a node defined.`);
        }
        return pageNode;
    }, [screenId, tableFieldProperties.node]);
    // Filter bulk actions based on access permissions
    const visibleBulkActions = React.useMemo(() => {
        if (!bulkActions || !pageDefinition) {
            return bulkActions || [];
        }
        return bulkActions.filter(bulkAction => {
            const bulkActionId = bulkAction.mutation;
            const bulkActionAccess = bulkAction.access;
            const accessRule = getBulkActionAccessStatus(pageDefinition.accessBindings, bulkActionId, node, bulkActionAccess);
            return accessRule === 'authorized';
        });
    }, [bulkActions, pageDefinition, node]);
    const getBulkActionParameters = React.useCallback(async (bulkAction) => {
        if (exportTemplatesByNode &&
            bulkAction.mutation === BULK_ACTION_ASYNC_EXPORT &&
            exportTemplatesByNode[node] &&
            objectKeys(exportTemplatesByNode[node]).length === 1) {
            // If the bulk action is an async export action and the page has exactly one template, then we skip the dialog
            return { id: objectKeys(exportTemplatesByNode[node])[0] };
        }
        if (bulkAction.configurationPage) {
            try {
                const result = await pageDialog(store, bulkAction.configurationPage, {
                    [QUERY_PARAM_PRINTING_SOURCE_PAGE]: screenId,
                    [QUERY_PARAM_PRINTING_NODE_TYPE]: node || '',
                    contextPage,
                    node,
                }, { size: 'medium', height: 250 });
                if (!isEmpty(result)) {
                    return result;
                }
                return false;
            }
            catch {
                // If the custom dialog is rejected we stop the mutation process gracefully
                return false;
            }
        }
        return {};
    }, [contextPage, exportTemplatesByNode, node, screenId, store]);
    const handleBulkAction = React.useCallback(async (bulkAction) => {
        try {
            if (selectedRowCount <= 0) {
                return;
            }
            onTelemetryEvent?.(`tableBulkActionTriggered-${bulkAction.id || camelCase(bulkAction.title)}`, {
                screenId,
                id: bulkAction.id || bulkAction.title,
            });
            const parameters = await getBulkActionParameters(bulkAction);
            if (parameters === false) {
                // The user rejected the dialog
                return;
            }
            await confirmationDialog(screenId, 'warn', bulkAction.title, localize('@sage/xtrem-ui/bulk-action-dialog-content', 'Perform this action on the selected items: {{itemCount}}', { itemCount: selectedRowCount }), {
                acceptButton: {
                    text: localize('@sage/xtrem-ui/ok', 'OK'),
                },
                cancelButton: {
                    text: localize('@sage/xtrem-ui/cancel', 'Cancel'),
                },
                size: 'small',
            });
            const filter = getSelectionFilter({
                groupByColumn,
                gridApi,
                screenId,
                isSelectAllChecked,
                tableFieldProperties,
                activeOptionsMenuItem,
            });
            const targetNode = graphApi.node(node);
            try {
                const mergeParameters = {
                    filter: JSON.stringify(filter),
                    ...parameters,
                };
                if (bulkAction.isGlobal) {
                    // TODO: Remove this when the global bulk actions are implemented in the Client Gen API
                    const stringQuery = queryToGraphQuery({
                        mutation: {
                            global: {
                                [bulkAction.mutation]: {
                                    start: {
                                        __args: mergeParameters,
                                        trackingId: true,
                                    },
                                },
                            },
                        },
                    });
                    await graphApi.raw(stringQuery);
                }
                else {
                    await targetNode.asyncOperations?.[bulkAction.mutation]
                        ?.start({ trackingId: true }, mergeParameters)
                        .execute();
                }
                onClearSelection();
                showToast(localize('@sage/xtrem-ui/bulk-action-started', 'Action started on the selected items.'), {
                    type: 'success',
                });
            }
            catch (_err) {
                showToast(localize('@sage/xtrem-ui/bulk-action-error', 'Action could not be started, please try again.'), { type: 'error' });
            }
        }
        catch (error) {
            // User dismissed dialog, intentionally left empty
        }
    }, [
        selectedRowCount,
        getBulkActionParameters,
        screenId,
        graphApi,
        node,
        onClearSelection,
        onTelemetryEvent,
        activeOptionsMenuItem,
        gridApi,
        groupByColumn,
        isSelectAllChecked,
        tableFieldProperties,
    ]);
    return (React.createElement("div", { className: "e-page-navigation-panel-bulk-actions-bar-wrapper" },
        React.createElement("div", { className: "e-page-navigation-panel-bulk-actions-bar" },
            React.createElement("span", { className: "e-page-navigation-panel-bulk-actions-bar-selected-items", "data-testid": "e-table-field-bulk-actions-bar-selected-items" }, localize('@sage/xtrem-ui/bulk-actions-bar-selected', 'Items selected: {{count}}', {
                count: selectedRowCount,
            })),
            visibleBulkActions?.map(bulkAction => {
                const handleButtonClick = () => {
                    handleBulkAction(bulkAction);
                };
                const { mutation, title, buttonType, icon, isDestructive } = bulkAction;
                return (React.createElement("div", { key: mutation, "data-testid": getDataTestIdAttribute('table', title, mutation, 'bulk-action') },
                    React.createElement(ButtonComponent, { screenId: screenId, elementId: mutation, locale: locale, onFocus: noop, fieldProperties: {
                            title,
                            isTitleHidden: true,
                        }, isNested: true, onClick: handleButtonClick, buttonType: buttonType, icon: icon, isDestructive: isDestructive, value: title })));
            }),
            React.createElement("button", { className: "e-page-navigation-panel-bulk-actions-bar-clear-selection", "data-testid": "e-table-field-bulk-actions-bar-clear-selection", type: "button", onClick: onClearSelection }, localize('@sage/xtrem-ui/clear-selection', 'Clear selection')))));
}
//# sourceMappingURL=desktop-table-bulk-action-bar.js.map