import * as React from 'react';
import { connect } from 'react-redux';
import ValidationIcon from 'carbon-react/esm/__internal__/validations';
import { ConnectedNestedFieldWrapper } from '../../render/nested-field-wrapper';
import { ContextType } from '../../types';
import { getDataTestIdAttribute, isHidden as isElementHidden } from '../../utils/dom';
import { convertDeepBindToPathNotNull, isNestedField } from '../../utils/nested-field-utils';
import { resolveByValue } from '../../utils/resolve-value-utils';
import { calculateContainerWidth, calculateFieldWidth, getGutterSize } from '../../utils/responsive-utils';
import { splitValueToMergedValue } from '../../utils/transformers';
import { withCollectionValueItemSubscription } from '../connected-collection';
import { generateFieldId } from '../field/carbon-helpers';
import IconButton from 'carbon-react/esm/components/icon-button';
import { withoutNestedTechnical } from '../nested-fields';
import { GridColumn, GridRow } from '@sage/xtrem-ui-components';
import XtremActionPopover from './xtrem-action-popover';
import Icon from 'carbon-react/esm/components/icon';
import { localize } from '../../service/i18n-service';
import * as tokens from '@sage/design-tokens/js/base/common';
import { getNestedFieldElementId, normalizeUnderscoreBind } from '../../utils/abstract-fields-utils';
import { get, isNil, isObject, isString } from 'lodash';
import { Checkbox } from 'carbon-react/esm/components/checkbox';
import { getBindAndSelectorFromDeepBinding } from '../../service/graphql-query-builder';
export class NestedBlock extends React.Component {
    constructor() {
        super(...arguments);
        this.onChange = async (bind, value) => {
            if (this.props.onChange) {
                await this.props.onChange(bind, value);
            }
        };
        this.renderField = (nestedField) => {
            const { properties, type } = nestedField;
            const columnProperties = {
                isReadOnly: undefined,
                ...properties,
            };
            const columnSpan = this.props.browser
                ? calculateFieldWidth(this.props.browser?.is, type, this.props.availableColumns, columnProperties.isFullWidth, columnProperties.width)
                : 2;
            columnProperties.isDisabled = this.props.isDisabled || columnProperties.isDisabled;
            const readOnlyOverride = resolveByValue({
                screenId: this.props.screenId,
                propertyValue: this.props.readOnlyOverride,
                fieldValue: columnProperties.bind,
                skipHexFormat: true,
                rowValue: null,
            });
            columnProperties.isReadOnly = readOnlyOverride === undefined ? columnProperties.isReadOnly : readOnlyOverride;
            const gridColumnClasses = ['e-field-grid-column'];
            const mergedRowValue = splitValueToMergedValue(this.props.recordValue || {});
            const bind = normalizeUnderscoreBind(getNestedFieldElementId(nestedField));
            let value = get(mergedRowValue, convertDeepBindToPathNotNull(bind));
            if (isNil(value) && this.props.contextNode && this.props.nodeTypes) {
                const { valuePath, jsonSelector } = getBindAndSelectorFromDeepBinding(bind, String(this.props.contextNode), this.props.nodeTypes);
                const rawFieldValue = get(mergedRowValue, valuePath, null);
                if (rawFieldValue && jsonSelector) {
                    let parsedJsonField;
                    if (isObject(rawFieldValue)) {
                        parsedJsonField = rawFieldValue;
                    }
                    else if (isString(rawFieldValue)) {
                        parsedJsonField = JSON.parse(rawFieldValue);
                    }
                    else {
                        parsedJsonField = null;
                    }
                    if (parsedJsonField) {
                        value = get(parsedJsonField, jsonSelector);
                    }
                }
            }
            if (resolveByValue({
                propertyValue: columnProperties.isHidden,
                rowValue: mergedRowValue,
                fieldValue: value,
                skipHexFormat: true,
                screenId: this.props.screenId,
            })) {
                gridColumnClasses.push('e-field-grid-column-hidden');
            }
            const { screenId, contextType, parentElementId } = this.props;
            const key = generateFieldId({
                screenId,
                elementId: convertDeepBindToPathNotNull(nestedField.properties.bind),
                contextType,
                fieldProperties: nestedField.properties,
                parentElementId,
                isNested: true,
            });
            const validationErrors = (this.props.validationErrors || []).filter(e => e.columnId === bind) || [];
            const getContextType = () => {
                // INFO: Currently NestedBlock is only used for pod components. If in future the
                //       NestedBlock is also to be used for non-pod components, this function should be
                //       used to figure out the correct context type, based on the parent component.
                return ContextType.pod;
            };
            return (React.createElement(GridColumn, { className: gridColumnClasses.join(' '), columnSpan: columnSpan, key: key },
                React.createElement(ConnectedNestedFieldWrapper, { _id: this.props.recordValue ? this.props.recordValue._id : '', columnDefinition: nestedField, columnName: bind, columnProperties: columnProperties, contextNode: this.props.contextNode, contextType: getContextType(), focusPosition: this.props.focusPosition, handlersArguments: {
                        rowValue: mergedRowValue,
                        onChange: [this.props.recordValue?._id, mergedRowValue],
                        onClick: [this.props.recordValue?._id, mergedRowValue],
                    }, nestedReadOnlyField: this.props.isReadOnly, parentElementId: this.props.parentElementId, recordContext: mergedRowValue, screenId: this.props.screenId, setFieldValue: this.onChange, shouldRenderLabelInNestedReadOnlyMode: this.props.isReadOnly, value: value, validationErrors: validationErrors, validate: this.props.validate, isParentReadOnly: this.props.isReadOnly })));
        };
        this.resolveTitle = () => !isNestedField(this.props.title)
            ? resolveByValue({
                screenId: this.props.screenId,
                propertyValue: this.props.title,
                skipHexFormat: true,
                rowValue: this.props.recordValue ? splitValueToMergedValue(this.props.recordValue) : undefined,
            })
            : null;
        this.getStatusIcon = (hideValidationSummary, errors, nestedFields, screenId, recordValue, warningMessage, infoMessage) => {
            if (errors && errors.length > 0 && !hideValidationSummary) {
                const message = errors
                    .map(e => {
                    const nestedField = withoutNestedTechnical(nestedFields).find(n => getNestedFieldElementId(n) === e.columnId);
                    const title = resolveByValue({
                        screenId,
                        propertyValue: nestedField?.properties.title,
                        skipHexFormat: true,
                        rowValue: recordValue ? splitValueToMergedValue(recordValue) : undefined,
                    }) || e.columnId;
                    return title ? `${title}: ${e.message}` : e.message;
                })
                    .join('\n');
                const props = {
                    error: message,
                };
                // CARBON-UPGRADE The context provider should be removed once carbon updates to React 17
                return (
                // <InputContext.Provider value={useInputBehaviour(true)}>
                React.createElement(ValidationIcon, { ...props })
                // </InputContext.Provider>
                );
            }
            if (warningMessage) {
                return (React.createElement(Icon, { key: "warning", className: "e-icon-warning-message", ariaLabel: warningMessage, fontSize: "small", color: tokens.colorsSemanticCaution500, type: "warning", role: "tooltip", tooltipMessage: warningMessage }));
            }
            if (infoMessage) {
                return (React.createElement(Icon, { key: "info", className: "e-icon-info-message", ariaLabel: infoMessage, fontSize: "small", color: tokens.colorsSemanticInfo500, type: "info", role: "tooltip", tooltipMessage: infoMessage }));
            }
            return null;
        };
        this.renderHeader = (baseClassName) => (React.createElement("div", { className: `e-${baseClassName}-header` },
            React.createElement("h3", { className: `e-${baseClassName}-title`, "data-testid": `e-${baseClassName}-title` },
                this.props.canSelect && (React.createElement("span", { className: `e-${baseClassName}-select`, "data-testid": `e-${baseClassName}-select` },
                    React.createElement(Checkbox, { checked: this.props.isSelected, value: "checked", disabled: this.props.isDisabled, onChange: (event) => {
                            if (this.props.onBlockSelectionChange) {
                                this.props.onBlockSelectionChange(event.target.checked);
                            }
                        } }))),
                this.renderTitle(),
                this.getStatusIcon(!!this.props.hideValidationSummary, this.props.validationErrors || [], this.props.nestedFields, this.props.screenId, this.props.recordValue, this.props.warning, this.props.info),
                this.props.headerLabel &&
                    this.props.recordValue &&
                    get(this.props.recordValue, getNestedFieldElementId(this.props.headerLabel)) && (React.createElement("div", { className: `e-${baseClassName}-header-label` },
                    React.createElement(ConnectedNestedFieldWrapper, { _id: this.props.recordValue._id, columnName: convertDeepBindToPathNotNull(this.props.headerLabel.properties.bind), columnProperties: this.props.headerLabel.properties, columnDefinition: this.props.headerLabel, value: get(this.props.recordValue, getNestedFieldElementId(this.props.headerLabel)), contextType: this.props.contextType, contextNode: this.props.contextNode, parentElementId: this.props.parentElementId, screenId: this.props.screenId, handlersArguments: { rowValue: splitValueToMergedValue(this.props.recordValue || {}) }, setFieldValue: this.onChange, nestedReadOnlyField: true }))),
                React.createElement("span", { className: "e-pod-header-spacer" }),
                this.props.actionPopoverItems && (React.createElement(XtremActionPopover, { items: this.props.actionPopoverItems, onOpen: this.props.onActionPopoverOpen, noIconSupport: true })),
                this.props.isCloseIconDisplayed && (React.createElement(IconButton, { disabled: this.props.isDisabled, onClick: this.props.onBlockRemoved, "data-testid": `e-${baseClassName}-close`, "aria-label": localize('@sage/xtrem-ui/pod-remove-item', 'Remove Item') },
                    React.createElement(Icon, { type: "cross" }))))));
    }
    renderTitle() {
        const title = this.props.title;
        const rowValue = this.props.recordValue;
        if (isNestedField(this.props.title) && rowValue) {
            const nestedFieldTitle = title;
            return (React.createElement(ConnectedNestedFieldWrapper, { _id: rowValue._id, columnName: convertDeepBindToPathNotNull(nestedFieldTitle.properties.bind), columnProperties: nestedFieldTitle.properties, columnDefinition: nestedFieldTitle, value: get(rowValue, getNestedFieldElementId(nestedFieldTitle)), contextType: this.props.contextType, contextNode: this.props.contextNode, parentElementId: this.props.parentElementId, screenId: this.props.screenId, handlersArguments: { rowValue: splitValueToMergedValue(rowValue) }, setFieldValue: this.onChange, nestedReadOnlyField: true }));
        }
        return React.createElement("span", { className: `e-${this.props.baseClassName}-title-text` }, this.resolveTitle());
    }
    render() {
        const { screenId, browser, nestedFields, isCloseIconDisplayed, onBlockClick, canSelect } = this.props;
        const baseClassName = this.props.baseClassName || 'block';
        const isTitleHidden = resolveByValue({
            screenId,
            propertyValue: this.props.isTitleHidden,
            skipHexFormat: true,
            rowValue: splitValueToMergedValue(this.props.recordValue || {}),
        });
        const isHidden = resolveByValue({
            screenId,
            propertyValue: this.props.isHidden,
            skipHexFormat: true,
            rowValue: splitValueToMergedValue(this.props.recordValue || {}),
        });
        const shouldRenderHeader = (this.props.title && !isTitleHidden) ||
            isCloseIconDisplayed ||
            canSelect ||
            (this.props.actionPopoverItems && this.props.actionPopoverItems.length > 0);
        const shouldRenderBody = !!this.props.recordValue;
        const computedWidth = browser
            ? calculateContainerWidth(browser.is, this.props.availableColumns)
            : this.props.availableColumns;
        const gridGutter = browser ? getGutterSize(browser.is) : 24;
        const gridVerticalMargin = this.props.contextType === ContextType.page && !this.props.noMargin ? 16 : 0;
        const classes = ['e-container-parent', `e-${baseClassName}-parent`];
        if (isElementHidden(this.props.item, this.props.browser) || isHidden) {
            classes.push('e-hidden');
        }
        if (this.props.contextType) {
            classes.push(`e-${baseClassName}-context-${this.props.contextType}`);
        }
        const testIds = [
            getDataTestIdAttribute(baseClassName, this.resolveTitle() || 'unknown', this.props.item.$containerId),
        ];
        if (this.props.additionalTestId) {
            testIds.push(this.props.additionalTestId);
        }
        return (React.createElement(GridColumn, { className: classes.join(' '), columnSpan: computedWidth },
            React.createElement("div", { className: `e-${baseClassName}`, onClick: onBlockClick, "data-testid": testIds.join(' ') },
                shouldRenderHeader && this.renderHeader(baseClassName),
                React.createElement("div", { className: `e-${baseClassName}-body` },
                    React.createElement(GridRow, { columns: computedWidth, gutter: gridGutter, margin: this.props.noMargin ? 0 : 16, verticalMargin: gridVerticalMargin }, shouldRenderBody && withoutNestedTechnical(nestedFields).map(this.renderField))))));
    }
}
const mapStateToProps = (state, props) => {
    const value = state.screenDefinitions[props.screenId].values[props.parentElementId];
    return {
        ...props,
        value,
        screenType: state.screenDefinitions[props.screenId].type,
        browser: state.browser,
        nodeTypes: state.nodeTypes,
        focusPosition: state.focusPosition,
    };
};
export const ConnectedNestedBlock = connect(mapStateToProps)(withCollectionValueItemSubscription(NestedBlock));
//# sourceMappingURL=nested-block.js.map