import { CalendarDate } from '@internationalized/date';
import { Datetime, formatDateToCurrentLocale, formatTimeToLocale, isValidIsoDate, } from '@sage/xtrem-date-time';
import { isNil } from 'lodash';
export function calendarDateToDatetime(calendarDate, timeZone) {
    const { year, month, day } = calendarDate;
    return Datetime.make(year, month, day, 0, 0, 0, 0, timeZone);
}
export function datetimeToCalendarDate(datetime, timeZone) {
    const valueWithTz = datetime.inTimeZone(timeZone);
    return new CalendarDate(valueWithTz.year, valueWithTz.month, valueWithTz.day);
}
export function datetimeToTime(datetime, timeZone) {
    const valueWithTz = datetime.inTimeZone(timeZone);
    return valueWithTz.time.toString();
}
function areDateTimesEqual(a, b) {
    if (isNil(a) || isNil(b)) {
        return (a ?? null) === (b ?? null);
    }
    return a.equals(b);
}
export function areDateRangesEqual(a, b) {
    return areDateTimesEqual(a?.start, b?.start) && areDateTimesEqual(a?.end, b?.end);
}
export function toDatetime(input, timeZone, locale = 'base') {
    if (Datetime.isDatetime(input)) {
        return input.inTimeZone(timeZone);
    }
    if (input instanceof Date) {
        return Datetime.fromJsDate(input, timeZone);
    }
    if (typeof input === 'string' && isValidIsoDate(input)) {
        return Datetime.parse(input, locale, undefined, timeZone);
    }
    throw new Error('Invalid date parameter');
}
export const formatDatetime = ({ date, locale = 'base', separator = ' - ', }) => {
    if (!date) {
        return '';
    }
    const formattedDate = formatDateToCurrentLocale(date, locale, 'FullDate');
    return [formattedDate, formatTimeToLocale(date.time, locale)].join(separator);
};
export const isDateInRange = (date, startDate, endDate) => {
    return date.compare(startDate) >= 0 && date.compare(endDate) <= 0;
};
export function isValidTimeZone(timeZone) {
    try {
        Intl.DateTimeFormat(undefined, { timeZone });
        return true;
    }
    catch (e) {
        return false;
    }
}
export const makeDatetime = (date, time, timeZone = 'UTC') => {
    return Datetime.make(date.year, date.month, date.day, time.hour, time.minute, time.second, undefined, timeZone);
};
//# sourceMappingURL=datetime-utils.js.map