import { Time } from '@sage/xtrem-date-time';
import ButtonMinor from 'carbon-react/esm/components/button-minor';
import { Checkbox } from 'carbon-react/esm/components/checkbox';
import Icon from 'carbon-react/esm/components/icon';
import Textbox from 'carbon-react/esm/components/textbox';
import React from 'react';
import { Calendar, CalendarGrid, I18nProvider } from 'react-aria-components';
import { localize } from '../../../service/i18n-service';
import { normalizeUnderscoreBind } from '../../../utils/abstract-fields-utils';
import { CalendarCellComponent } from './calendar-cell-component';
import { MonthYearHeaderComponent } from './month-year-header-component';
import { noop } from 'lodash';
import { useTime } from '../time/use-time';
import { formatDatetime, makeDatetime } from './datetime-utils';
import { TimeComponent as TimeComp } from '../time/time-component';
export function DatetimeInputComponent({ 'aria-label': ariaLabel, date, elementId, fieldId, nestedRowId, initialDate, inputRef, isDisabled, isPopoverOpen, isReadOnly, isTimeZoneHidden, locale, maxDate, minDate, onDateChange, onPopperOpenChange, onTimeChange, rangeStartDate, screenId, size = 'medium', time, timeZone = 'UTC', title, type, validationError, }) {
    const [isManuallyClosed, setIsManuallyClosed] = React.useState(false);
    const popoverDialogRef = React.useRef(null);
    const inputWrapperRef = React.useRef(null);
    const [isInfinite, setIsInfinite] = React.useState(!date); // If no date is selected, default to infinite
    const [isTimeZoneOpen, setIsTimeZoneOpen] = React.useState(false);
    const id = React.useMemo(() => `--${screenId}-bind-${normalizeUnderscoreBind(elementId || '')}-${type}${nestedRowId}`, [elementId, screenId, type, nestedRowId]);
    const handleTimeChange = React.useCallback((newTime) => {
        onTimeChange(newTime);
    }, [onTimeChange]);
    const { hasAmPmToggle, hoursRef, maxHours, minHours, minutesRef, onHoursBlur, onHoursChange, onKeyDown, onMinutesBlur, onBlurField, onMinutesChange, state, toggleChange, } = useTime({
        elementId: fieldId || `datetime-input-${Date.now()}`,
        locale,
        onChange: handleTimeChange,
        value: time?.toString(),
    });
    const handlePopoverOpenChange = React.useCallback((isOpen) => {
        if (!isOpen) {
            inputRef?.current?.blur();
            onBlurField();
            setIsManuallyClosed(true);
            setTimeout(() => setIsManuallyClosed(false), 0);
            popoverDialogRef.current?.hidePopover();
        }
        else {
            popoverDialogRef.current?.showPopover();
            requestAnimationFrame(() => adjustPopoverPosition());
        }
        onPopperOpenChange(isOpen, type);
    }, [inputRef, onBlurField, onPopperOpenChange, type]);
    const onPopoverToggle = React.useCallback((event) => {
        if (event.newState === 'open') {
            handlePopoverOpenChange(true);
        }
        else {
            handlePopoverOpenChange(false);
        }
    }, [handlePopoverOpenChange]);
    const handleToggle = React.useCallback((event) => {
        requestAnimationFrame(() => {
            adjustPopoverPosition();
            onPopoverToggle(event);
        });
    }, [onPopoverToggle]);
    const adjustPopoverPosition = () => {
        const pop = popoverDialogRef.current;
        const wrap = inputWrapperRef.current;
        if (!wrap || !pop)
            return;
        const wrapRect = wrap.getBoundingClientRect();
        const popRect = pop.getBoundingClientRect();
        const vw = window.innerWidth;
        const vh = window.innerHeight;
        let top = wrapRect.bottom;
        let left = wrapRect.left;
        if (top + popRect.height > vh) {
            top = wrapRect.top - popRect.height;
        }
        const margin = 8;
        const minLeft = margin;
        const maxLeft = Math.max(minLeft, vw - popRect.width - margin);
        left = Math.min(Math.max(left, minLeft), maxLeft);
        const minTop = margin;
        const maxTop = Math.max(minTop, vh - popRect.height - margin);
        top = Math.min(Math.max(top, minTop), maxTop);
        pop.style.top = `${top}px`;
        pop.style.left = `${left}px`;
    };
    React.useEffect(() => {
        const popoverDialogRefCopy = popoverDialogRef.current;
        const inputWrapperRefCopy = inputWrapperRef.current;
        if (!inputWrapperRefCopy || !popoverDialogRefCopy) {
            return noop;
        }
        popoverDialogRefCopy.addEventListener('toggle', handleToggle);
        return () => {
            popoverDialogRefCopy?.removeEventListener('toggle', handleToggle);
        };
    }, [id, handleToggle]);
    const handleDateChange = React.useCallback((newDate) => {
        onDateChange(newDate);
        if (isInfinite) {
            setIsInfinite(false);
        }
    }, [onDateChange, isInfinite]);
    React.useEffect(() => {
        if (date) {
            setIsInfinite(false);
        }
        else {
            setIsInfinite(true);
        }
    }, [date]);
    const isDateUnavailable = React.useCallback((startDate, minDate, maxDate) => (calendarDate) => {
        if (minDate && calendarDate.compare(minDate) < 0)
            return true;
        if (maxDate && calendarDate.compare(maxDate) > 0)
            return true;
        if (startDate && calendarDate.compare(startDate) < 0)
            return true;
        return false;
    }, []);
    const rangeStartCalendarDate = React.useMemo(() => rangeStartDate, [rangeStartDate]);
    const handleFocus = React.useCallback(() => {
        if (!isManuallyClosed) {
            handlePopoverOpenChange(true);
        }
    }, [handlePopoverOpenChange, isManuallyClosed]);
    const onToggleChange = React.useCallback((_, toggle) => {
        toggleChange(toggle);
    }, [toggleChange]);
    React.useEffect(() => {
        if (isPopoverOpen) {
            setTimeout(() => {
                popoverDialogRef.current
                    ?.querySelector('.react-aria-CalendarCell[tabindex="0"]')
                    ?.focus();
            }, 50);
        }
    }, [isPopoverOpen]);
    const onInputKeyDown = React.useCallback((ev) => {
        if (ev.key === 'Tab') {
            handlePopoverOpenChange(false);
        }
        if (ev.key === 'Backspace' || ev.key === 'Delete') {
            handleDateChange(null);
            handleTimeChange(null);
        }
        if (ev.key.startsWith('Key') || ev.key.startsWith('Digit') || ev.key.startsWith('Numpad')) {
            // Prevent the user from typing in the input field
            ev.preventDefault();
            ev.stopPropagation();
        }
    }, [handleDateChange, handlePopoverOpenChange, handleTimeChange]);
    const inputValue = date && time ? formatDatetime({ date: makeDatetime(date, Time.parse(time), timeZone), locale }) : '';
    const infiniteTooltipMessage = type === 'start'
        ? localize('@sage/xtrem-ui/infinite-indicator-start', 'No start date')
        : localize('@sage/xtrem-ui/infinite-indicator-end', 'No end date');
    return (React.createElement(React.Fragment, null,
        type === 'end' && (React.createElement(I18nProvider, { locale: locale },
            React.createElement("div", { className: "e-popover-dialog ag-custom-component-popup", popover: "auto", ref: popoverDialogRef, id: id, "data-type": type },
                React.createElement("div", { className: "e-calendar-container", "data-testid": `e-datetime-input-${type}-date-picker` },
                    React.createElement("div", { className: "e-calendar-infinite-container" },
                        React.createElement(Checkbox, { readOnly: isReadOnly, "aria-readonly": isReadOnly, disabled: isDisabled, "data-testid": `e-datetime-input-${type}-date-picker-checkbox`, label: infiniteTooltipMessage, checked: isInfinite, size: "small", className: "e-calendar-infinite-checkbox", onChange: event => {
                                if (isReadOnly || isDisabled)
                                    return;
                                setIsInfinite(event.target.checked);
                                // If is infinite set date as null
                                if (event.target.checked) {
                                    handleDateChange(null);
                                    handleTimeChange(null);
                                }
                            } })),
                    React.createElement(Calendar, { focusedValue: initialDate, maxValue: maxDate, minValue: minDate, isDisabled: isDisabled, onChange: handleDateChange, isReadOnly: isReadOnly, value: date, isDateUnavailable: isDateUnavailable(rangeStartCalendarDate, minDate, maxDate) },
                        React.createElement(MonthYearHeaderComponent, { isDisabled: isDisabled }),
                        React.createElement(CalendarGrid, { className: "e-calendar-grid-styled", weekdayStyle: "narrow" }, day => (React.createElement(CalendarCellComponent, { day: day, rangeStartCalendarDate: rangeStartCalendarDate, selectedDate: date || undefined, timeZone: timeZone }))))),
                React.createElement("div", { className: "e-time-component-container e-time-field" },
                    React.createElement("div", { className: "e-time-component-title" }, localize('@sage/xtrem-ui/date-time-range-time', 'Time')),
                    !isTimeZoneOpen && (React.createElement("div", { className: "e-time-component-wrapper" },
                        React.createElement(TimeComp, { screenId: screenId, elementId: elementId, dataTestId: `e-datetime-input-${type}-time-input`, fieldId: fieldId, isReadOnly: isReadOnly, hoursRef: hoursRef, minutesRef: minutesRef, isDisabled: isDisabled || isInfinite, maxHours: maxHours, minHours: minHours, onHoursBlur: onHoursBlur, onHoursChange: onHoursChange, onKeyDown: onKeyDown, onMinutesBlur: onMinutesBlur, onMinutesChange: onMinutesChange, state: state, hasAmPmToggle: hasAmPmToggle, localize: localize, onToggleChange: onToggleChange }),
                        !isTimeZoneHidden && (React.createElement(ButtonMinor, { "data-testid": `e-time-component-open-timezone-${type}`, iconType: "refresh_clock", disabled: isDisabled || isInfinite, onClick: () => setIsTimeZoneOpen(true) })))),
                    isTimeZoneOpen && (React.createElement("div", { className: "e-time-component-timezone-container" },
                        React.createElement(ButtonMinor, { "data-testid": `e-time-component-close-timezone-${type}`, iconType: "clock", onClick: () => setIsTimeZoneOpen(false) }),
                        React.createElement("div", { className: "e-time-component-timezone-input" },
                            React.createElement(Textbox, { label: localize('@sage/xtrem-ui/date-time-range-time-zone', 'Time zone:'), labelInline: true, id: `e-datetime-input-${type}-time-input-timezone`, "data-testid": `e-datetime-input-${type}-time-input-timezone`, value: timeZone, readOnly: true, inputWidth: 60, labelWidth: 100 })))))))),
        React.createElement("div", { className: "e-combined-input-wrapper", popovertarget: id, ref: inputWrapperRef },
            React.createElement(Textbox, { "data-testid": `e-datetime-input-${type}`, label: title, "aria-label": ariaLabel, size: size, inputIcon: "calendar", value: inputValue, onFocus: handleFocus, onClick: handleFocus, ref: inputRef, onKeyDown: onInputKeyDown, disabled: isDisabled, readOnly: isReadOnly, error: validationError }),
            isInfinite && type !== 'single' && (React.createElement(Icon, { tooltipMessage: infiniteTooltipMessage, className: "e-infinite-indicator", type: "arrows_left_right" }))),
        (type === 'start' || type === 'single') && (React.createElement(I18nProvider, { locale: locale },
            React.createElement("div", { className: "e-popover-dialog ag-custom-component-popup", popover: "auto", ref: popoverDialogRef, id: id, "data-type": type },
                React.createElement("div", { className: "e-calendar-container", "data-testid": `e-datetime-input-${type}-date-picker` },
                    type !== 'single' && (React.createElement("div", { className: "e-calendar-infinite-container" },
                        React.createElement(Checkbox, { readOnly: isReadOnly, "aria-readonly": isReadOnly, disabled: isDisabled, "data-testid": `e-datetime-input-${type}-date-picker-checkbox`, label: infiniteTooltipMessage, checked: isInfinite, size: "small", className: "e-calendar-infinite-checkbox", onChange: event => {
                                if (isReadOnly || isDisabled)
                                    return;
                                setIsInfinite(event.target.checked);
                                // If is infinite set date as null
                                if (event.target.checked) {
                                    handleDateChange(null);
                                    handleTimeChange(null);
                                }
                            } }))),
                    React.createElement(Calendar, { focusedValue: initialDate, maxValue: maxDate, minValue: minDate, isDisabled: isDisabled, onChange: handleDateChange, isReadOnly: isReadOnly, value: date, isDateUnavailable: isDateUnavailable(rangeStartCalendarDate, minDate, maxDate) },
                        React.createElement(MonthYearHeaderComponent, { isDisabled: isDisabled }),
                        React.createElement(CalendarGrid, { className: "e-calendar-grid-styled", weekdayStyle: "narrow" }, day => (React.createElement(CalendarCellComponent, { day: day, rangeStartCalendarDate: rangeStartCalendarDate, selectedDate: date || undefined, timeZone: timeZone }))))),
                React.createElement("div", { className: "e-time-component-container e-time-field" },
                    React.createElement("div", { className: "e-time-component-title" }, localize('@sage/xtrem-ui/date-time-range-time', 'Time')),
                    !isTimeZoneOpen && (React.createElement("div", { className: "e-time-component-wrapper" },
                        React.createElement(TimeComp, { screenId: screenId, elementId: elementId, dataTestId: `e-datetime-input-${type}-time-input`, fieldId: fieldId, isReadOnly: isReadOnly, hoursRef: hoursRef, minutesRef: minutesRef, isDisabled: isDisabled || isInfinite, maxHours: maxHours, minHours: minHours, onHoursBlur: onHoursBlur, onHoursChange: onHoursChange, onKeyDown: onKeyDown, onMinutesBlur: onMinutesBlur, onMinutesChange: onMinutesChange, state: state, hasAmPmToggle: hasAmPmToggle, localize: localize, onToggleChange: onToggleChange }),
                        !isTimeZoneHidden && (React.createElement(ButtonMinor, { "data-testid": `e-time-component-open-timezone-${type}`, iconType: "refresh_clock", disabled: isDisabled || isInfinite, onClick: () => setIsTimeZoneOpen(true) })))),
                    isTimeZoneOpen && (React.createElement("div", { className: "e-time-component-timezone-container" },
                        React.createElement(ButtonMinor, { "data-testid": `e-time-component-close-timezone-${type}`, iconType: "clock", onClick: () => setIsTimeZoneOpen(false) }),
                        React.createElement("div", { className: "e-time-component-timezone-input" },
                            React.createElement(Textbox, { label: localize('@sage/xtrem-ui/date-time-range-time-zone', 'Time zone:'), labelInline: true, id: `e-datetime-input-${type}-time-input-timezone`, "data-testid": `e-datetime-input-${type}-time-input-timezone`, value: timeZone, readOnly: true, inputWidth: 60, labelWidth: 100 }))))))))));
}
//# sourceMappingURL=datetime-input-component.js.map