/**
 * @packageDocumentation
 * @module root
 * */
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
import { get } from 'lodash';
import { ActionType } from '../redux/action-types';
import { getStore } from '../redux/store';
import { showToast } from '../service/toast-service';
import { getScreenElement } from '../service/screen-base-definition';
import { findNextField } from '../utils/layout-utils';
import { convertDeepBindToPathNotNull } from '../utils/nested-field-utils';
import { getPageDefinitionFromState } from '../utils/state-utils';
import { AbstractUiControlObject } from './abstract-ui-control-object';
import { ControlObjectProperty } from './property-decorators/control-object-property-decorator';
import { FieldControlObjectResolvedProperty } from './property-decorators/control-object-resolved-property-decorator';
/**
 * Any element holding a value than can be placed inside a page and can be interacted with (i.e. retrieving
 * and/or setting field's ui properties values). The element value CANNOT be modified
 */
export class ReadonlyFieldControlObject extends AbstractUiControlObject {
    static { this.defaultUiProperties = {
        ...AbstractUiControlObject.defaultUiProperties,
        isFullWidth: false,
        isHelperTextHidden: false,
        isHidden: false,
    }; }
    constructor(properties) {
        super(properties.screenId, properties.elementId, properties.getUiComponentProperties, properties.setUiComponentProperties, properties.componentKey);
        this._getValue = () => {
            return properties.getValue(this.screenId, this.elementId);
        };
        this._setValue = (newValue) => properties.setValue(this.screenId, this.elementId, newValue);
        this._refresh = ({ keepPageInfo, keepModifications = false }) => {
            return properties.refresh({
                screenId: this.screenId,
                elementId: this.elementId,
                keepPageInfo,
                keepModifications,
            });
        };
        this._focus = (row, nestedField) => {
            // Allow time for rendering to finish in case it is called from the onLoad function.
            setTimeout(() => {
                properties.focus(this.screenId, this.elementId, row, nestedField);
            }, 150);
        };
        this.layout = properties.layout;
        this.parent = properties.parent;
        this.insertBefore = properties.insertBefore;
        this.insertAfter = properties.insertAfter;
    }
    /** Refreshes the field's value from the server. */
    async refresh() {
        await this._refresh({ keepPageInfo: false }).catch(e => {
            showToast(e.message || e, { type: 'warning' });
        });
    }
    _redraw(columnBind) {
        return new Promise(resolve => setTimeout(() => {
            getStore().dispatch({
                type: ActionType.RedrawComponent,
                value: {
                    elementId: this.elementId,
                    screenId: this.screenId,
                    columnBind,
                },
            });
            resolve();
        }));
    }
    /** Field's value */
    get value() {
        // The following cast allows for field values different from what is exposed to the functional code.
        // In these cases custom getters & setters must be provided to marshal/unmarshal the value properly.
        // Refer to 'table-control-object' as an example.
        return this._getValue();
    }
    /** Field's value */
    set value(newValue) {
        // The following cast allows for field values different from what is exposed to the functional code.
        // In these cases custom getters & setters must be provided to marshal/unmarshal the value properly.
        // Refer to 'table-control-object' as an example.
        this._setValue(newValue);
    }
    /**
     * Fetch default values from the server even if the field is dirty.
     * If the `skipSet` argument is set to true, the values are requested from the server but not applied to the page.
     */
    async fetchDefault(skipSet = false) {
        const screenDefinition = getPageDefinitionFromState(this.screenId);
        if (screenDefinition.type !== 'page') {
            throw new Error('Default values can only be fetched in pages.');
        }
        const page = getScreenElement(screenDefinition);
        const result = await page.$.fetchDefaults([this.elementId], skipSet);
        return get(result, convertDeepBindToPathNotNull(this.properties.bind || this.elementId));
    }
    /** Moves the browser focus to this field */
    focus() {
        this._focus();
    }
    /** Returns the next field compared to the position of this field instance */
    getNextField(isFocusable = false) {
        return findNextField(this.screenId, this.elementId, isFocusable);
    }
}
__decorate([
    ControlObjectProperty()
    /** The helper text underneath the field */
], ReadonlyFieldControlObject.prototype, "helperText", void 0);
__decorate([
    ControlObjectProperty()
    /** Whether the field helper text is hidden or not. Defaults to false */
], ReadonlyFieldControlObject.prototype, "isHelperTextHidden", void 0);
__decorate([
    ControlObjectProperty()
    /** Whether the value is bound only to GraphQL mutations (isTransientInput = true) or not (isTransientInput = false). Defaults to false */
], ReadonlyFieldControlObject.prototype, "isTransientInput", void 0);
__decorate([
    FieldControlObjectResolvedProperty()
    /**
     * Whether the HTML element is editable (disabled = false) or not (disabled = true)
     *
     * The difference with readOnly is that disabled suggests that the field is not editable
     * for some validation reason (e.g. a button which can be clicked due to validation errors)
     */
], ReadonlyFieldControlObject.prototype, "isDisabled", void 0);
//# sourceMappingURL=readonly-field-control-object.js.map