export const LayoutDefaults = {
    node: { defaultWidth: 260, defaultHeight: 120 },
    group: { minWidth: 300, minHeight: 200, defaultWidth: 520, defaultHeight: 220, paddingX: 80, paddingY: 60 },
    grid: { gapX: 80, gapY: 60 },
    endLabel: { width: 120, height: 32 },
};
/**
 * Compute the horizontal offset to center a child inside a parent width.
 */
export function getCenteredChildX({ parentWidth, childWidth }) {
    return Math.round(parentWidth / 2) - Math.round(childWidth / 2);
}
/**
 * Given a repeat node absolute position and size, compute the group absolute position so the group
 * is centered under the repeat and overlaps 50% (group top = repeat center Y).
 */
export function computeGroupPositionUnderRepeat({ repeatPos, repeatSize, groupWidth, }) {
    return {
        x: Math.round((repeatPos.x ?? 0) + Math.round(repeatSize.width / 2) - Math.round(groupWidth / 2)),
        y: Math.round((repeatPos.y ?? 0) + Math.round(repeatSize.height / 2)),
    };
}
/**
 * Given a group absolute position and width, compute the repeat absolute position so the repeat is
 * centered horizontally and anchored 50% inside/50% outside above the group.
 */
export function computeAnchoredRepeatPosition({ groupPos, groupWidth, repeatSize, }) {
    return {
        x: Math.round((groupPos.x ?? 0) + getCenteredChildX({ parentWidth: groupWidth, childWidth: repeatSize.width })),
        y: Math.round((groupPos.y ?? 0) - Math.round(repeatSize.height / 2)),
    };
}
/**
 * Position the EndRepeat label at the bottom center of the group, 50% inside/50% outside.
 */
export function computeAnchoredEndRepeatPosition({ groupPos, groupSize, endSize, }) {
    return {
        x: Math.round((groupPos.x ?? 0) + getCenteredChildX({ parentWidth: groupSize.width, childWidth: endSize.width })),
        y: Math.round((groupPos.y ?? 0) + groupSize.height - Math.round(endSize.height / 2)),
    };
}
/**
 * When a repeat is collapsed, place the EndRepeat label overlapped just behind the repeat bottom edge.
 * The overlap moves the label slightly inside the group to reduce visual gap.
 */
export function computeCollapsedEndRepeatPosition({ repeatPos, repeatSize, endSize, overlap = 14, }) {
    return {
        x: Math.round((repeatPos.x ?? 0) + Math.round(repeatSize.width / 2) - Math.round(endSize.width / 2)),
        y: Math.round((repeatPos.y ?? 0) + repeatSize.height - endSize.height + overlap),
    };
}
/**
 * Calculate group size expansion to fit children with padding, returning shifts to apply to
 * children that are too close to the top/left so that the group position remains stable.
 */
export function autoResizeGroup({ children, currentStyle, paddingX = 20, paddingY = 20, minWidth = 300, minHeight = 200, }) {
    if (!children || children.length === 0) {
        return {
            width: currentStyle.width ?? minWidth,
            height: currentStyle.height ?? minHeight,
            shiftX: 0,
            shiftY: 0,
        };
    }
    const defaultNodeWidth = LayoutDefaults.node.defaultWidth;
    const defaultNodeHeight = LayoutDefaults.node.defaultHeight;
    const minX = Math.min(...children.map(c => c.position?.x ?? 0));
    const minY = Math.min(...children.map(c => c.position?.y ?? 0));
    const rightMost = Math.max(...children.map(c => (c.position?.x ?? 0) + (c.width ?? defaultNodeWidth)));
    const bottomMost = Math.max(...children.map(c => (c.position?.y ?? 0) + (c.height ?? defaultNodeHeight)));
    const shiftX = Math.max(0, paddingX - minX);
    const shiftY = Math.max(0, paddingY - minY);
    const width = Math.max(minWidth, rightMost + paddingX + shiftX);
    const height = Math.max(minHeight, bottomMost + paddingY + shiftY);
    return { width, height, shiftX, shiftY };
}
/**
 * After a group resize, re-center the anchored repeat horizontally while preserving its previous Y.
 * Respects a minimum left padding.
 */
export function recenterAnchoredRepeatHorizontally({ groupPos, groupWidth, repeatPrevY, repeatWidth, paddingX = 20, }) {
    const centeredX = Math.max(paddingX, getCenteredChildX({ parentWidth: groupWidth, childWidth: repeatWidth }));
    return { x: (groupPos.x ?? 0) + centeredX, y: repeatPrevY };
}
/**
 * Convenience: compute group position under repeat and both anchored positions (repeat top, end bottom)
 */
export function computeGroupAndAnchors({ repeatPos, repeatSize, groupSize, endSize, }) {
    const groupPos = computeGroupPositionUnderRepeat({ repeatPos, repeatSize, groupWidth: groupSize.width });
    const anchoredRepeatPos = computeAnchoredRepeatPosition({ groupPos, groupWidth: groupSize.width, repeatSize });
    const anchoredEndPos = computeAnchoredEndRepeatPosition({ groupPos, groupSize, endSize });
    return { groupPos, anchoredRepeatPos, anchoredEndPos };
}
/**
 * Compute absolute flow coordinates for a node whose position may be relative to a parent group.
 * Walks up the parent chain accumulating positions.
 */
export function computeAbsoluteFlowPosition(node, allNodes) {
    if (!node)
        return { x: 0, y: 0 };
    let x = node.position?.x ?? 0;
    let y = node.position?.y ?? 0;
    // Walk up the parent chain if present in node extras
    let current = node;
    while (true) {
        const parentId = current?.parentId;
        if (!parentId)
            break;
        const parent = allNodes.find(n => n.id === parentId);
        if (!parent)
            break;
        x += parent.position?.x ?? 0;
        y += parent.position?.y ?? 0;
        current = parent;
    }
    return { x, y };
}
/**
 * Shift siblings when a repeat expands: move nodes in the same scope that vertically overlap the expanded group
 * and those below them to avoid overlap. Save original positions to restore on collapse.
 */
export function resolveSiblingsOnRepeatExpand(nodes, repeatNodeId, getStyleWidth, getStyleHeight) {
    const newNodes = nodes.map(n => ({ ...n }));
    const groupNode = newNodes.find(n => n.type === 'group' && n.data?.anchoredRepeatId === repeatNodeId);
    if (!groupNode)
        return newNodes;
    // Collect structure ids: repeat + group + end + all descendants of group
    const structureIds = new Set();
    structureIds.add(repeatNodeId);
    structureIds.add(groupNode.id);
    const endRepeatId = groupNode.data?.anchoredEndRepeatId;
    if (endRepeatId)
        structureIds.add(endRepeatId);
    let changed = true;
    while (changed) {
        changed = false;
        for (const n of newNodes) {
            const parentId = n?.parentId;
            if (parentId && structureIds.has(parentId) && !structureIds.has(n.id)) {
                structureIds.add(n.id);
                changed = true;
            }
        }
    }
    // Expanded group absolute rect
    const groupAbs = computeAbsoluteFlowPosition(groupNode, newNodes);
    const groupWidth = getStyleWidth(groupNode, 410);
    const groupHeight = getStyleHeight(groupNode, 170);
    const groupLeft = groupAbs.x;
    const groupTop = groupAbs.y;
    const groupRight = groupLeft + groupWidth;
    const groupBottom = groupTop + groupHeight;
    // Scope: same parentId (including null)
    const scopeParentId = groupNode?.parentId ?? null;
    const getNodeAbsRect = (n) => {
        const abs = computeAbsoluteFlowPosition(n, newNodes);
        const width = typeof n.width === 'number' ? n.width : LayoutDefaults.node.defaultWidth;
        const height = typeof n.height === 'number' ? n.height : LayoutDefaults.node.defaultHeight;
        return { left: abs.x, top: abs.y, right: abs.x + width, bottom: abs.y + height };
    };
    const inSameScope = (n) => (n?.parentId ?? null) === scopeParentId;
    const overlapsHorizontally = (rect) => !(rect.right <= groupLeft || rect.left >= groupRight);
    const candidates = newNodes.filter(n => !structureIds.has(n.id) && inSameScope(n));
    const overlapping = candidates.filter(n => {
        const r = getNodeAbsRect(n);
        const verticalOverlap = r.top < groupBottom && r.bottom > groupTop;
        return verticalOverlap && overlapsHorizontally(r);
    });
    if (overlapping.length === 0)
        return newNodes;
    const minTop = Math.min(...overlapping.map(n => getNodeAbsRect(n).top));
    const desiredTop = groupBottom + LayoutDefaults.grid.gapY;
    const shiftDelta = Math.max(0, desiredTop - minTop);
    if (shiftDelta === 0)
        return newNodes;
    // Save original positions for restoration on collapse
    const savedPositions = {};
    for (let i = 0; i < newNodes.length; i += 1) {
        const n = newNodes[i];
        const isStructure = structureIds.has(n.id);
        const sameScope = inSameScope(n);
        if (!isStructure && sameScope) {
            const rect = getNodeAbsRect(n);
            if (rect.top >= minTop) {
                savedPositions[newNodes[i].id] = {
                    x: newNodes[i].position?.x ?? 0,
                    y: newNodes[i].position?.y ?? 0,
                };
                newNodes[i] = {
                    ...newNodes[i],
                    position: {
                        x: newNodes[i].position?.x ?? 0,
                        y: (newNodes[i].position?.y ?? 0) + shiftDelta,
                    },
                };
            }
        }
    }
    // Persist on group node data
    groupNode.data.__siblingRestorePositions = savedPositions;
    return newNodes;
}
export function resolveSiblingsOnRepeatCollapse(nodes, repeatNodeId) {
    const newNodes = nodes.map(n => ({ ...n }));
    const groupIndex = newNodes.findIndex(n => n.type === 'group' && n.data?.anchoredRepeatId === repeatNodeId);
    if (groupIndex < 0)
        return newNodes;
    const groupNode = newNodes[groupIndex];
    const saved = groupNode.data.__siblingRestorePositions;
    if (!saved || Object.keys(saved).length === 0)
        return newNodes;
    // Restore positions
    for (let i = 0; i < newNodes.length; i += 1) {
        const id = newNodes[i].id;
        const pos = saved[id];
        if (pos) {
            newNodes[i] = {
                ...newNodes[i],
                position: { x: pos.x, y: pos.y },
            };
        }
    }
    // Clear saved positions after restore
    const nextGroup = { ...groupNode, data: { ...groupNode.data } };
    delete nextGroup.data.__siblingRestorePositions;
    newNodes[groupIndex] = nextGroup;
    return newNodes;
}
//# sourceMappingURL=workflow-layout-utils.js.map