import type { Node, XYPosition } from 'reactflow';
import type { DefaultDataType } from './workflow-types';
export type Size = {
    width: number;
    height: number;
};
export declare const LayoutDefaults: {
    readonly node: {
        readonly defaultWidth: 260;
        readonly defaultHeight: 120;
    };
    readonly group: {
        readonly minWidth: 300;
        readonly minHeight: 200;
        readonly defaultWidth: 520;
        readonly defaultHeight: 220;
        readonly paddingX: 80;
        readonly paddingY: 60;
    };
    readonly grid: {
        readonly gapX: 80;
        readonly gapY: 60;
    };
    readonly endLabel: {
        readonly width: 120;
        readonly height: 32;
    };
};
/**
 * Compute the horizontal offset to center a child inside a parent width.
 */
export declare function getCenteredChildX({ parentWidth, childWidth }: {
    parentWidth: number;
    childWidth: number;
}): number;
/**
 * Given a repeat node absolute position and size, compute the group absolute position so the group
 * is centered under the repeat and overlaps 50% (group top = repeat center Y).
 */
export declare function computeGroupPositionUnderRepeat({ repeatPos, repeatSize, groupWidth, }: {
    repeatPos: XYPosition;
    repeatSize: Size;
    groupWidth: number;
}): XYPosition;
/**
 * Given a group absolute position and width, compute the repeat absolute position so the repeat is
 * centered horizontally and anchored 50% inside/50% outside above the group.
 */
export declare function computeAnchoredRepeatPosition({ groupPos, groupWidth, repeatSize, }: {
    groupPos: XYPosition;
    groupWidth: number;
    repeatSize: Size;
}): XYPosition;
/**
 * Position the EndRepeat label at the bottom center of the group, 50% inside/50% outside.
 */
export declare function computeAnchoredEndRepeatPosition({ groupPos, groupSize, endSize, }: {
    groupPos: XYPosition;
    groupSize: Size;
    endSize: Size;
}): XYPosition;
/**
 * When a repeat is collapsed, place the EndRepeat label overlapped just behind the repeat bottom edge.
 * The overlap moves the label slightly inside the group to reduce visual gap.
 */
export declare function computeCollapsedEndRepeatPosition({ repeatPos, repeatSize, endSize, overlap, }: {
    repeatPos: XYPosition;
    repeatSize: Size;
    endSize: Size;
    overlap?: number;
}): XYPosition;
/**
 * Calculate group size expansion to fit children with padding, returning shifts to apply to
 * children that are too close to the top/left so that the group position remains stable.
 */
export declare function autoResizeGroup({ children, currentStyle, paddingX, paddingY, minWidth, minHeight, }: {
    children: Array<{
        position: XYPosition;
        width?: number;
        height?: number;
    }>;
    currentStyle: {
        width?: number;
        height?: number;
    };
    paddingX?: number;
    paddingY?: number;
    minWidth?: number;
    minHeight?: number;
}): {
    width: number;
    height: number;
    shiftX: number;
    shiftY: number;
};
/**
 * After a group resize, re-center the anchored repeat horizontally while preserving its previous Y.
 * Respects a minimum left padding.
 */
export declare function recenterAnchoredRepeatHorizontally({ groupPos, groupWidth, repeatPrevY, repeatWidth, paddingX, }: {
    groupPos: XYPosition;
    groupWidth: number;
    repeatPrevY: number;
    repeatWidth: number;
    paddingX?: number;
}): XYPosition;
/**
 * Convenience: compute group position under repeat and both anchored positions (repeat top, end bottom)
 */
export declare function computeGroupAndAnchors({ repeatPos, repeatSize, groupSize, endSize, }: {
    repeatPos: XYPosition;
    repeatSize: Size;
    groupSize: Size;
    endSize: Size;
}): {
    groupPos: XYPosition;
    anchoredRepeatPos: XYPosition;
    anchoredEndPos: XYPosition;
};
/**
 * Compute absolute flow coordinates for a node whose position may be relative to a parent group.
 * Walks up the parent chain accumulating positions.
 */
export declare function computeAbsoluteFlowPosition(node: Node<DefaultDataType>, allNodes: Node<DefaultDataType>[]): {
    x: number;
    y: number;
};
/**
 * Shift siblings when a repeat expands: move nodes in the same scope that vertically overlap the expanded group
 * and those below them to avoid overlap. Save original positions to restore on collapse.
 */
export declare function resolveSiblingsOnRepeatExpand(nodes: Node<DefaultDataType>[], repeatNodeId: string, getStyleWidth: (node: {
    style?: React.CSSProperties;
    width?: number | null;
}, fallback: number) => number, getStyleHeight: (node: {
    style?: React.CSSProperties;
    height?: number | null;
}, fallback: number) => number): Node<DefaultDataType>[];
export declare function resolveSiblingsOnRepeatCollapse(nodes: Node<DefaultDataType>[], repeatNodeId: string): Node<DefaultDataType>[];
//# sourceMappingURL=workflow-layout-utils.d.ts.map