import * as React from 'react';
import { connect, useDispatch } from 'react-redux';
import { mapDispatchToProps, mapStateToProps } from '../field-base-component';
import ReactFlow, { useReactFlow, ReactFlowProvider, Background, useNodesState, useEdgesState, } from 'reactflow';
import { CustomNode } from './custom-node';
import { CarbonWrapper } from '../carbon-wrapper';
import { isEqual, isNil } from 'lodash';
import { handleChange } from '../../../utils/abstract-fields-utils';
import { useCompare } from '../../../utils/hooks/effects/use-compare';
import { changeEventHandler, createNewEdge, createNewNode, createRepeatNodeStructure, edgeStyle, isSingleEdgeRemoval, removeChildNodesAndEdgesFromStartPoint, removeTransientEdgeDataPropertiesFromEdges, removeTransientNodeDataPropertiesFromNodes, roundToNearestTwenty, } from './workflow-component-utils';
import { AddWorkflowNodeDialog } from './add-workflow-node-dialog';
import { loadWorkflowNodes } from '../../../redux/actions/workflow-nodes-actions';
import { CustomEdge } from './custom-edge';
import Typography from 'carbon-react/esm/components/typography';
import { isFieldDisabled, isFieldReadOnly } from '../carbon-helpers';
import { WorkflowContext } from './workflow-context-provider';
import { localize } from '../../../service/i18n-service';
import { findAncestorDatasetProperty } from '../../../utils/dom';
import { confirmationDialog } from '../../../service/dialog-service';
import Button from 'carbon-react/esm/components/button';
import { openFieldInDialog } from '../../../redux/actions';
import { WorkflowControlButtons } from './control-buttons';
import { useDeepEqualSelector } from '../../../utils/hooks/use-deep-equal-selector';
import { SubflowNode } from './nodes/subflow-node';
import { computeAnchoredEndRepeatPosition, computeAnchoredRepeatPosition, computeCollapsedEndRepeatPosition, computeGroupPositionUnderRepeat, computeAbsoluteFlowPosition, LayoutDefaults, resolveSiblingsOnRepeatExpand, resolveSiblingsOnRepeatCollapse, } from './workflow-layout-utils';
import { useHistoryState } from './use-history-state';
import { useGroupAutoResize } from './use-group-auto-resize';
import { useAnchoredRepeatLayout } from './use-anchored-repeat-layout';
const edgeTypes = {
    default: CustomEdge,
};
export function WorkflowComponent(props) {
    return (React.createElement(ReactFlowProvider, null,
        React.createElement(WorkflowInnerComponent, { ...props })));
}
export function WorkflowInnerComponent(props) {
    const { elementId, fieldProperties, isParentDisabled, screenId, setFieldValue, validate, value } = props;
    const isDragging = React.useRef(false);
    const componentRef = React.useRef(null);
    const emptyStateElementRef = React.useRef(null);
    const dispatch = useDispatch();
    const workflowNodes = useDeepEqualSelector(s => s.workflowNodes);
    const expansionDialogControl = useDeepEqualSelector(s => {
        return (Object.values(s.activeDialogs).find(d => d.screenId === screenId && d.content instanceof Array && d.content[0]?.id === elementId)?.dialogControl ?? null);
    });
    const hasExternalValueChanged = useCompare(value);
    const connectingStartNodeId = React.useRef(null);
    const edgeSplittingId = React.useRef(null);
    const nextNodePosition = React.useRef({});
    const [addDialogFilters, setAddDialogFilters] = React.useState(null);
    const { screenToFlowPosition, flowToScreenPosition, zoomIn, zoomOut, fitView } = useReactFlow();
    const [nodes, setNodes, onNodesChange] = useNodesState([]);
    const [edges, setEdges, onEdgesChange] = useEdgesState([]);
    const [errors, setErrors] = React.useState([]);
    const onTelemetryEvent = useDeepEqualSelector((s) => s.applicationContext?.onTelemetryEvent);
    const [eventLogs, setEventLogs] = React.useState([]);
    const hasFitView = React.useRef(false);
    const selectedRecordId = useDeepEqualSelector((state) => {
        const screenDef = state.screenDefinitions?.[screenId];
        return screenDef?.selectedRecordId;
    });
    // Helpers to compute numeric width/height from style or node metrics
    const getStyleWidth = React.useCallback((node, fallback) => {
        const w = node?.style?.width;
        if (typeof w === 'number')
            return w;
        if (typeof node?.width === 'number')
            return node.width;
        return fallback;
    }, []);
    const getStyleHeight = React.useCallback((node, fallback) => {
        const h = node?.style?.height;
        if (typeof h === 'number')
            return h;
        if (typeof node?.height === 'number')
            return node.height;
        return fallback;
    }, []);
    // Use custom history hook instead of inline state management
    const historyState = useHistoryState(nodes, edges, setNodes, setEdges);
    const { takeHistorySnapshot, resetHistory, isWorkflowDirty, onUndo, onRedo, isUndoDisabled, isRedoDisabled } = historyState;
    // Use custom group auto-resize hook
    const groupAutoResize = useGroupAutoResize(nodes, setNodes, isDragging, getStyleWidth, getStyleHeight);
    const { scheduleGroupResize } = groupAutoResize;
    // Use custom anchored repeat layout hook
    useAnchoredRepeatLayout(nodes, setNodes);
    const onExpand = React.useCallback(() => {
        dispatch(openFieldInDialog(screenId, elementId));
    }, [dispatch, elementId, screenId]);
    const onMinimize = React.useCallback(() => {
        expansionDialogControl?.cancel();
    }, [expansionDialogControl]);
    const isDisabled = React.useMemo(() => {
        return Boolean(isParentDisabled) || isFieldDisabled(screenId, fieldProperties, value, {});
    }, [fieldProperties, isParentDisabled, screenId, value]);
    const isReadOnly = React.useMemo(() => {
        return isFieldReadOnly(screenId, fieldProperties, value, {});
    }, [fieldProperties, screenId, value]);
    // From workflow nodes we set all to CustomNode type for ReactFlow
    const nodeTypes = React.useMemo(() => {
        const nodeTypes = workflowNodes?.reduce((types, node) => {
            types[node.key] = CustomNode;
            return types;
        }, {}) || {};
        return { ...nodeTypes, group: SubflowNode, repeat: CustomNode, endRepeat: CustomNode };
    }, [workflowNodes]);
    // Compute derived visibility when a repeat is collapsed: hide its group, all descendants, and anchored endRepeat
    const hiddenIds = React.useMemo(() => {
        const ids = new Set();
        // Find collapsed repeat nodes
        nodes.forEach(n => {
            const baseType = n.type?.split('/')?.[0];
            const isCollapsedRepeat = baseType === 'repeat' && Boolean(n.data?.isCollapsed);
            if (!isCollapsedRepeat) {
                return;
            }
            const candidateGroupId = n.data?.groupId ||
                nodes.find(g => g.type === 'group' && g.data?.anchoredRepeatId === n.id)?.id;
            if (!candidateGroupId) {
                return;
            }
            const queue = [candidateGroupId];
            while (queue.length > 0) {
                const current = queue.shift();
                if (!ids.has(current)) {
                    ids.add(current);
                    // Keep End repeat node visible even when group is collapsed
                    nodes.forEach(child => {
                        if (child.parentId === current) {
                            queue.push(child.id);
                        }
                    });
                }
            }
        });
        return ids;
    }, [nodes]);
    const renderNodes = React.useMemo(() => {
        if (hiddenIds.size === 0)
            return nodes;
        return nodes.filter(n => !hiddenIds.has(n.id));
    }, [nodes, hiddenIds]);
    const renderNodeIdSet = React.useMemo(() => new Set(renderNodes.map(n => n.id)), [renderNodes]);
    const renderEdges = React.useMemo(() => {
        if (hiddenIds.size === 0)
            return edges;
        return edges.filter(e => renderNodeIdSet.has(e.source) && renderNodeIdSet.has(e.target));
    }, [edges, hiddenIds, renderNodeIdSet]);
    // When a repeat is collapsed, reposition its End repeat to sit just under the repeat box
    const adjustedNodes = React.useMemo(() => {
        if (hiddenIds.size === 0)
            return renderNodes;
        const updated = renderNodes.map(n => ({ ...n }));
        nodes.forEach(n => {
            const baseType = n.type?.split('/')?.[0];
            if (baseType !== 'repeat' || !n.data?.isCollapsed)
                return;
            const groupId = n.data?.groupId || nodes.find(g => g.type === 'group' && g.data?.anchoredRepeatId === n.id)?.id;
            const groupNode = groupId ? nodes.find(no => no.id === groupId) : undefined;
            const endRepeatId = groupNode?.data?.anchoredEndRepeatId;
            if (!endRepeatId)
                return;
            const endRepeatIdx = updated.findIndex(no => no.id === endRepeatId);
            if (endRepeatIdx < 0)
                return;
            const repeatWidth = n?.width ?? 260;
            const repeatHeight = n?.height ?? 120;
            const endRepeatWidth = updated[endRepeatIdx]?.width ?? 120;
            const endRepeatHeight = updated[endRepeatIdx]?.height ?? 32;
            const pos = computeCollapsedEndRepeatPosition({
                repeatPos: { x: n.position?.x ?? 0, y: n.position?.y ?? 0 },
                repeatSize: { width: repeatWidth, height: repeatHeight },
                endSize: { width: endRepeatWidth, height: endRepeatHeight },
                overlap: 14,
            });
            updated[endRepeatIdx].position = pos;
        });
        return updated;
    }, [renderNodes, hiddenIds, nodes]);
    const onAddNodeDialogClose = React.useCallback(() => {
        connectingStartNodeId.current = null;
        edgeSplittingId.current = null;
        setAddDialogFilters(null);
    }, []);
    const onChange = React.useCallback((newValue = { edges, nodes }) => {
        if (hasExternalValueChanged) {
            return;
        }
        const cleanNodes = removeTransientNodeDataPropertiesFromNodes(newValue.nodes);
        const cleanEdges = removeTransientEdgeDataPropertiesFromEdges(newValue.edges);
        const needToValidate = () => {
            if (value?.nodes.length !== newValue.nodes.length) {
                return true;
            }
            // If some of the nodes data has been changed we need to call the validate
            return cleanNodes.some((node, index) => {
                const oldNode = value?.nodes[index];
                if (oldNode && node.data && oldNode.data) {
                    return !isEqual(node.data, oldNode.data);
                }
                return false;
            });
        };
        if (isWorkflowDirty &&
            ((!newValue.nodes.find(n => n.dragging) && !isEqual(cleanNodes, value?.nodes ?? [])) ||
                !isEqual(cleanEdges, value?.edges ?? []))) {
            handleChange(elementId, { nodes: cleanNodes, edges: cleanEdges }, setFieldValue, needToValidate() ? validate : undefined, // We only call validate function if data in nodes has changed or if the number of nodes has changed
            changeEventHandler(screenId, elementId));
        }
    }, [
        edges,
        nodes,
        hasExternalValueChanged,
        isWorkflowDirty,
        value?.nodes,
        value?.edges,
        elementId,
        setFieldValue,
        validate,
        screenId,
    ]);
    const onNodeDataChange = React.useCallback((nodeId, data) => {
        takeHistorySnapshot();
        const newNodes = [...nodes];
        const index = newNodes.findIndex(n => n.id === nodeId);
        const prevNode = newNodes[index];
        const wasCollapsed = Boolean(prevNode?.data?.isCollapsed);
        const willBeCollapsed = Boolean(data?.isCollapsed);
        newNodes[index] = {
            ...newNodes[index],
            data,
        };
        // When a repeat unfolds (collapsed -> expanded), ensure non-group siblings below are pushed down
        try {
            const baseType = prevNode?.type?.split('/')?.[0];
            if (baseType === 'repeat' && wasCollapsed && !willBeCollapsed) {
                const resolved = resolveSiblingsOnRepeatExpand(newNodes, nodeId, getStyleWidth, getStyleHeight);
                for (let i = 0; i < newNodes.length; i += 1)
                    newNodes[i] = resolved[i];
            }
            // When a repeat folds (expanded -> collapsed), restore saved positions
            if (baseType === 'repeat' && !wasCollapsed && willBeCollapsed) {
                const restored = resolveSiblingsOnRepeatCollapse(newNodes, nodeId);
                for (let i = 0; i < newNodes.length; i += 1)
                    newNodes[i] = restored[i];
            }
        }
        catch {
            // intentionally silent
        }
        setNodes(newNodes);
    }, [nodes, setNodes, takeHistorySnapshot, getStyleWidth, getStyleHeight]);
    const onNodeInsertToEdge = React.useCallback((edgeId) => {
        edgeSplittingId.current = edgeId;
        const edge = edges.find(e => e.id === edgeId);
        const sourceNode = edge?.source;
        const sourceHandle = edge?.sourceHandle;
        if (sourceNode && sourceHandle) {
            connectingStartNodeId.current = { nodeId: sourceNode, handleId: sourceHandle };
            setAddDialogFilters(['condition', 'action']);
        }
    }, [edges]);
    const onNodeBelowNode = React.useCallback((nodeId, type, handleId) => {
        const targetNode = nodes.find(n => n.id === nodeId);
        if (!targetNode) {
            return;
        }
        // Compute absolute flow coordinates for the target node (children positions are relative to parent)
        const computeAbsolutePosition = (node) => computeAbsoluteFlowPosition(node, nodes);
        const abs = computeAbsolutePosition(targetNode);
        const { x, y } = flowToScreenPosition(abs);
        const verticalDelta = LayoutDefaults.node.defaultHeight + LayoutDefaults.grid.gapY;
        const horizontalDelta = LayoutDefaults.node.defaultWidth + LayoutDefaults.grid.gapX;
        if (type === 'repeat-open-dialog') {
            connectingStartNodeId.current = { nodeId, handleId: handleId || 'out' };
            if (handleId === 'out-true') {
                nextNodePosition.current = { clientX: x - horizontalDelta, clientY: y + verticalDelta };
            }
            else if (handleId === 'out-false') {
                nextNodePosition.current = { clientX: x + horizontalDelta, clientY: y + verticalDelta };
            }
            else if (handleId === 'out') {
                nextNodePosition.current = { clientX: x + horizontalDelta, clientY: y + verticalDelta };
            }
            else {
                nextNodePosition.current = {
                    clientX: x + Math.round(horizontalDelta / 2),
                    clientY: y + verticalDelta,
                };
            }
            setAddDialogFilters(['repeat']);
            return;
        }
        connectingStartNodeId.current = { nodeId, handleId: handleId || 'out' };
        if (handleId === 'out-true') {
            nextNodePosition.current = {
                clientX: x - horizontalDelta,
                clientY: y + verticalDelta,
            };
        }
        else if (handleId === 'out-false') {
            nextNodePosition.current = {
                clientX: x + horizontalDelta,
                clientY: y + verticalDelta,
            };
        }
        else if (handleId === 'out') {
            nextNodePosition.current = {
                clientX: x + horizontalDelta,
                clientY: y + verticalDelta,
            };
        }
        else {
            nextNodePosition.current = {
                clientX: x + Math.round(horizontalDelta / 2),
                clientY: y + verticalDelta,
            };
        }
        setAddDialogFilters(type ? [type] : ['condition', 'action']);
    }, [flowToScreenPosition, nodes]);
    const onClick = React.useCallback(() => {
        if (!componentRef.current || componentRef.current.contains(document.activeElement)) {
            return;
        }
        onChange();
    }, [onChange]);
    // Effects (grouped)
    /** Reset fitView flag when record changes */
    React.useEffect(() => {
        hasFitView.current = false;
    }, [selectedRecordId]);
    /** Auto fit view on initial load with dynamic padding based on node count */
    React.useEffect(() => {
        const nodesCount = value?.nodes?.length ?? 0;
        if (!hasFitView.current && nodesCount > 0) {
            setTimeout(() => {
                const dynamicPadding = nodesCount >= 3 ? 0.4 : 2;
                fitView({ padding: dynamicPadding, maxZoom: 1 });
                hasFitView.current = true;
            }, 100);
        }
    }, [value?.nodes, fitView]);
    /** Load validation errors into local state */
    React.useEffect(() => {
        const validationErrors = props.validationErrors;
        if (validationErrors && validationErrors.length > 0) {
            setErrors(validationErrors.map(e => ({ stepId: e.recordId || '', message: e.message })));
        }
        else {
            setErrors([]);
        }
    }, [props.validationErrors, setNodes, value?.nodes, setErrors]);
    /** Load event logs into local state */
    React.useEffect(() => {
        if (!isNil(props.fieldProperties.eventLogs) && !isEqual(eventLogs, props.fieldProperties.eventLogs)) {
            setEventLogs(props.fieldProperties.eventLogs || []);
        }
    }, [props.fieldProperties.eventLogs, eventLogs, setEventLogs]);
    /** Inject errors and logs into node data so CustomNode can render them */
    React.useEffect(() => {
        setNodes(prevNodes => prevNodes.map(node => {
            const error = errors.find(e => e.stepId === node.id);
            const lastLog = eventLogs.filter(l => l.stepId === node.id).pop();
            return {
                ...node,
                data: {
                    ...node.data,
                    message: error?.message,
                    eventLog: lastLog,
                },
            };
        }));
    }, [errors, eventLogs, setNodes]);
    /** Ensure workflow node definitions are loaded */
    React.useEffect(() => {
        if (!workflowNodes) {
            dispatch(loadWorkflowNodes());
        }
    }, [dispatch, workflowNodes]);
    /** Sync external value into internal state when it changes */
    React.useEffect(() => {
        const areNodesDifferent = !isEqual(value?.nodes, removeTransientNodeDataPropertiesFromNodes(nodes));
        if (hasExternalValueChanged && areNodesDifferent) {
            setNodes((value?.nodes || []).map(n => ({
                ...n,
                // Enforce non-draggable for group and endRepeat nodes
                draggable: n.type === 'group' || n.type === 'endRepeat' ? false : (n.draggable ?? true),
                selectable: n.type === 'group' ? false : (n.selectable ?? true),
                data: {
                    ...n.data,
                    type: n.type,
                },
            })));
        }
        const areEdgesDifferent = !isEqual(value?.edges, removeTransientEdgeDataPropertiesFromEdges(edges));
        if (hasExternalValueChanged && areEdgesDifferent) {
            setEdges((value?.edges || []).map(e => ({
                ...e,
                ...edgeStyle,
                data: { screenId, elementId },
            })));
        }
        if (hasExternalValueChanged && (areEdgesDifferent || areNodesDifferent)) {
            resetHistory();
        }
    }, [
        value,
        setNodes,
        setEdges,
        elementId,
        screenId,
        nodes,
        edges,
        hasExternalValueChanged,
        isReadOnly,
        isDisabled,
        resetHistory,
    ]);
    /** Persist changes when nodes/edges change */
    React.useEffect(() => {
        onChange();
    }, [onChange, nodes, edges]);
    /** Global click handler to trigger change when clicking outside */
    React.useEffect(() => {
        window.addEventListener('click', onClick);
        return () => {
            window.removeEventListener('click', onClick);
        };
    }, [onClick]);
    const isValidConnection = React.useCallback((connection) => {
        // Connection must have both source and target
        if (!connection.source || !connection.target) {
            return false;
        }
        // Find source and target nodes
        const sourceNode = nodes.find(n => n.id === connection.source);
        const targetNode = nodes.find(n => n.id === connection.target);
        if (!sourceNode || !targetNode) {
            return false;
        }
        // Get parent IDs (group membership) for both nodes
        const sourceParentId = sourceNode.parentId;
        const targetParentId = targetNode.parentId;
        // Check if source is a repeat node (starts with 'repeat/')
        const sourceIsRepeat = sourceNode.type?.split('/')?.[0] === 'repeat';
        // Check if target is a repeat node (starts with 'repeat/')
        const targetIsRepeat = targetNode.type?.split('/')?.[0] === 'repeat';
        // Special case: if source is a repeat node with 'out' handle
        // The target must belong to the repeat's group (target.parentId === source.data.groupId)
        if (sourceIsRepeat) {
            const sourceGroupId = sourceNode.data?.groupId;
            // If repeat has a group, target must be inside that group
            if (sourceGroupId) {
                return targetParentId === sourceGroupId;
            }
        }
        // Special case: if target is a repeat node
        // The source must be in the same context as the repeat node (not inside the repeat's own group)
        if (targetIsRepeat) {
            const targetGroupId = targetNode.data?.groupId;
            // Source should NOT be inside the repeat's group
            if (targetGroupId && sourceParentId === targetGroupId) {
                return false;
            }
        }
        // Both nodes must be in the same group context:
        // - Both have no parent (outside any repeat group)
        // - Both have the same parent (inside the same repeat group)
        return sourceParentId === targetParentId;
    }, [nodes]);
    const onConnectStart = React.useCallback((_, { nodeId, handleId }) => {
        if (nodeId && handleId) {
            connectingStartNodeId.current = { nodeId, handleId };
        }
    }, []);
    const onConnectEnd = React.useCallback((event) => {
        if (!connectingStartNodeId.current)
            return;
        const target = event.target;
        const targetIsPane = target?.classList?.contains('react-flow__pane');
        if (targetIsPane) {
            nextNodePosition.current = {
                clientX: event.clientX,
                clientY: event.clientY,
            };
            setAddDialogFilters(['condition', 'action']);
            return;
        }
        const nodeId = findAncestorDatasetProperty(target, 'nodeid');
        if (nodeId && connectingStartNodeId.current.nodeId !== nodeId) {
            // Validate the connection before creating the edge
            const isValid = isValidConnection({
                source: connectingStartNodeId.current.nodeId,
                target: nodeId,
            });
            if (!isValid) {
                // Connection is not valid, do not create the edge
                return;
            }
            takeHistorySnapshot();
            setEdges([
                ...edges,
                createNewEdge({
                    edges,
                    targetNodeId: nodeId,
                    sourceHandleId: connectingStartNodeId.current.handleId,
                    sourceNodeId: connectingStartNodeId.current.nodeId,
                    data: { screenId, elementId },
                }),
            ]);
        }
    }, [edges, elementId, screenId, setEdges, takeHistorySnapshot, isValidConnection]);
    const onEdgesChangeInternal = React.useCallback(change => {
        if (isDisabled || isReadOnly) {
            return;
        }
        if (isSingleEdgeRemoval(change, edges)) {
            return;
        }
        onEdgesChange(change);
    }, [edges, isDisabled, isReadOnly, onEdgesChange]);
    const handleRemovalChange = React.useCallback(async (change) => {
        const connectingEdges = edges.filter(e => e.source === change.id);
        if (connectingEdges.length > 0) {
            try {
                await confirmationDialog(props.screenId, 'info', localize('@sage/xtrem-ui/workflow-delete-node-chain-title', 'Delete step flow'), localize('@sage/xtrem-ui/workflow-delete-node-chain-message', 'If you remove this step, any subsequent steps with no other links are also removed.'));
            }
            catch (e) {
                return;
            }
        }
        takeHistorySnapshot();
        let newEdges = [...edges];
        const newNodes = [...nodes];
        const removedNode = nodes.find(n => n.id === change.id);
        if (!removedNode) {
            setNodes(newNodes);
            setEdges(newEdges);
            return;
        }
        removeChildNodesAndEdgesFromStartPoint(change.id, newEdges, newNodes);
        // If removing an anchored repeat, also remove its target group and all children recursively
        if (removedNode?.type?.split('/')?.[0] === 'repeat') {
            // Prefer explicit groupId, else derive from a group that anchors this repeat
            const groupId = removedNode?.data?.groupId ||
                nodes.find(n => n.type === 'group' && n.data?.anchoredRepeatId === removedNode.id)?.id;
            if (!groupId) {
                const remainingIdsNoGroup = newNodes.map(n => n.id);
                newEdges = newEdges.filter(e => remainingIdsNoGroup.includes(e.source) && remainingIdsNoGroup.includes(e.target));
                setNodes(newNodes);
                setEdges(newEdges);
                return;
            }
            const idsToRemove = new Set([groupId]);
            let changed = true;
            while (changed) {
                changed = false;
                for (const n of newNodes) {
                    const parentId = n.parentId;
                    if (parentId && idsToRemove.has(parentId) && !idsToRemove.has(n.id)) {
                        idsToRemove.add(n.id);
                        changed = true;
                    }
                }
            }
            // Remove nodes and edges connected to them
            for (const id of idsToRemove) {
                const idx = newNodes.findIndex(n => n.id === id);
                if (idx >= 0) {
                    newNodes.splice(idx, 1);
                }
            }
            newEdges = newEdges.filter(e => !idsToRemove.has(e.source) && !idsToRemove.has(e.target));
        }
        const remainingIds = newNodes.map(n => n.id);
        newEdges = newEdges.filter(e => remainingIds.includes(e.source) && remainingIds.includes(e.target));
        setNodes(newNodes);
        setEdges(newEdges);
    }, [edges, nodes, props.screenId, setEdges, setNodes, takeHistorySnapshot]);
    const handlePositionChange = React.useCallback((change) => {
        const targetNode = nodes.find(n => n.id === change.id);
        // If a group moves, keep its anchored repeat attached to the top edge, centered
        if (targetNode?.type === 'group') {
            const groupNode = targetNode;
            const repeatId = groupNode?.data?.anchoredRepeatId;
            if (repeatId && change?.position) {
                const idx = nodes.findIndex(n => n.id === repeatId);
                if (idx >= 0) {
                    const groupWidth = getStyleWidth(groupNode, 410);
                    const anchoredPos = computeAnchoredRepeatPosition({
                        groupPos: { x: groupNode.position?.x ?? 0, y: groupNode.position?.y ?? 0 },
                        groupWidth,
                        repeatSize: { width: nodes[idx]?.width ?? 260, height: nodes[idx]?.height ?? 120 },
                    });
                    const newNodes = nodes.map(n => ({ ...n }));
                    newNodes[idx].position = anchoredPos;
                    setNodes(newNodes);
                }
            }
        }
        // If a repeat moves, reposition its target group so the group stays just below and centered
        if (targetNode && targetNode.type?.split('/')?.[0] === 'repeat') {
            const groupId = targetNode.data?.groupId ||
                nodes.find(n => n.type === 'group' && n.data?.anchoredRepeatId === targetNode.id)?.id;
            if (groupId) {
                const groupIndex = nodes.findIndex(n => n.id === groupId);
                if (groupIndex >= 0 && targetNode) {
                    const repeatWidth = targetNode?.width ?? 260;
                    const repeatHeight = targetNode?.height ?? 120;
                    const groupWidth = getStyleWidth(nodes[groupIndex], 410);
                    const repeatPos = change?.position || targetNode.position || { x: 0, y: 0 };
                    const groupPos = computeGroupPositionUnderRepeat({
                        repeatPos: { x: repeatPos.x ?? 0, y: repeatPos.y ?? 0 },
                        repeatSize: { width: repeatWidth, height: repeatHeight },
                        groupWidth,
                    });
                    const newNodes = nodes.map(n => ({ ...n }));
                    newNodes[groupIndex].position = groupPos;
                    // Keep End repeat centered at group boundary as repeat moves (50% inside, 50% outside)
                    const endRepeatId = newNodes[groupIndex].data?.anchoredEndRepeatId;
                    if (endRepeatId) {
                        const endRepeatIndex = newNodes.findIndex(n => n.id === endRepeatId);
                        if (endRepeatIndex >= 0) {
                            const endRepeatWidth = newNodes[endRepeatIndex]?.width ?? 120;
                            const endRepeatHeight = newNodes[endRepeatIndex]?.height ?? 32;
                            const endPos = computeAnchoredEndRepeatPosition({
                                groupPos: { x: groupPos.x, y: groupPos.y },
                                groupSize: {
                                    width: getStyleWidth(nodes[groupIndex], 410),
                                    height: getStyleHeight(nodes[groupIndex], 170),
                                },
                                endSize: { width: endRepeatWidth, height: endRepeatHeight },
                            });
                            newNodes[endRepeatIndex].position = endPos;
                        }
                    }
                    setNodes(newNodes);
                }
            }
        }
        // While dragging a child inside a group, schedule live auto-resize for that group
        if (change.dragging && targetNode?.parentId) {
            scheduleGroupResize(targetNode.parentId);
        }
        // We need to take a history snapshot only when the user starts dragging a node
        if (!isDragging.current && change.dragging) {
            isDragging.current = true;
            takeHistorySnapshot();
        }
        else if (isDragging.current && !change.dragging) {
            isDragging.current = false;
            // When a child node inside a group stops dragging, schedule a resize for that group
            const parentGroupId = targetNode?.parentId || undefined;
            if (parentGroupId) {
                scheduleGroupResize(parentGroupId);
            }
        }
        onNodesChange([change]);
    }, [nodes, onNodesChange, setNodes, takeHistorySnapshot, getStyleWidth, getStyleHeight, scheduleGroupResize]);
    const onNodesChangeInternal = React.useCallback(async (change) => {
        if (isDisabled || isReadOnly) {
            return;
        }
        const removalChange = change.find(c => c.type === 'remove');
        if (removalChange) {
            // Prevent deleting group directly; only deletable via repeat cascade
            const nodeToRemove = nodes.find(n => n.id === removalChange.id);
            if (nodeToRemove?.type === 'group') {
                return;
            }
            await handleRemovalChange(removalChange);
            return;
        }
        const positionChange = change.find(c => c.type === 'position');
        if (positionChange) {
            await handlePositionChange(positionChange);
            return;
        }
        onNodesChange(change);
    }, [handlePositionChange, handleRemovalChange, isDisabled, isReadOnly, onNodesChange, nodes]);
    const onAddFirstElement = React.useCallback(() => {
        setAddDialogFilters(['event']);
    }, []);
    const addNewNodeByNextPosition = React.useCallback((selectedNodeType, values) => {
        const position = nextNodePosition.current.clientX && nextNodePosition.current.clientY
            ? screenToFlowPosition({
                x: (nextNodePosition.current.clientX ?? 170) - 125,
                y: nextNodePosition.current.clientY ?? 20,
            })
            : screenToFlowPosition({
                x: roundToNearestTwenty((emptyStateElementRef.current?.offsetWidth || 40) / 2 - 125),
                y: 40,
            });
        const newNode = createNewNode({ selectedNodeType, values, nodes, position });
        // If the user selected a Repeat node from the dialog, we must create the Group and EndRepeat node
        const isRepeatLike = (selectedNodeType?.split('/')?.[0] || selectedNodeType) === 'repeat';
        if (isRepeatLike) {
            const { nodes: resultNodes, edges: resultEdges } = createRepeatNodeStructure({
                newNode,
                nodes,
                edges,
                position,
                connectingStartNodeId: connectingStartNodeId.current,
                screenId,
                elementId,
                getStyleWidth,
                scheduleGroupResize,
            });
            setNodes(resultNodes);
            setEdges(resultEdges);
            return;
        }
        // Group auto-resize is handled by the groupAutoResize hook
        const newEdges = [...edges];
        if (connectingStartNodeId.current) {
            newEdges.push(createNewEdge({
                edges,
                targetNodeId: newNode.id,
                sourceHandleId: connectingStartNodeId.current.handleId,
                sourceNodeId: connectingStartNodeId.current.nodeId,
                data: { screenId, elementId },
            }));
        }
        // Prepare a working copy of nodes including the new node so we can apply branch flags and grouping below
        const newNodes = [...nodes, newNode];
        // If the new node comes from a condition node where the true/false branches were not enabled, we need to set the ifTrueBranch/ifFalseBranch in condition data
        if (connectingStartNodeId.current &&
            ['out-true', 'out-false'].includes(connectingStartNodeId.current.handleId)) {
            const branchKey = connectingStartNodeId.current.handleId === 'out-true' ? 'ifTrueBranch' : 'ifFalseBranch';
            const condIndex = newNodes.findIndex(n => n.id === connectingStartNodeId.current?.nodeId);
            if (condIndex >= 0) {
                newNodes[condIndex] = {
                    ...newNodes[condIndex],
                    data: { ...newNodes[condIndex].data, [branchKey]: true },
                };
            }
        }
        // Ensure new nodes created after repeat (or any node inside a group) remain inside the same group.
        // Exception: when connecting from repeat's "out" handle, follow normal flow (outside group).
        if (connectingStartNodeId.current) {
            const sourceNode = nodes.find(n => n.id === connectingStartNodeId.current?.nodeId);
            const isRepeatOutOutside = sourceNode?.type?.split('/')?.[0] === 'repeat' &&
                connectingStartNodeId.current.handleId === 'out';
            // If the source is the anchored repeat, route children to its target group (only for 'out' handle)
            const parentGroupId = !isRepeatOutOutside &&
                (sourceNode?.data?.groupId ||
                    sourceNode?.parentId ||
                    nodes.find(n => n.type === 'group' && n?.data?.anchoredRepeatId === sourceNode?.id)?.id);
            if (parentGroupId) {
                const groupIndex = newNodes.findIndex((n) => n.id === parentGroupId);
                const parentGroup = groupIndex >= 0 ? newNodes[groupIndex] : nodes.find(n => n.id === parentGroupId);
                if (parentGroup) {
                    // Compute relative desired position beneath the source node inside the group
                    // Use a mutable relative copy to allow pre-resize calculations
                    // Use parent's ABSOLUTE position for nested groups
                    const parentAbsPos = computeAbsoluteFlowPosition(parentGroup, nodes);
                    const relative = {
                        x: position.x - parentAbsPos.x,
                        y: position.y - parentAbsPos.y,
                    };
                    // Apply relative position and containment
                    const idx = newNodes.findIndex((n) => n.id === newNode.id);
                    newNodes[idx] = {
                        ...newNodes[idx],
                        position: relative,
                        parentId: parentGroupId,
                        extent: 'parent',
                    };
                    scheduleGroupResize(parentGroupId);
                }
            }
        }
        setNodes(newNodes);
        setEdges(newEdges);
    }, [
        edges,
        elementId,
        nodes,
        screenId,
        screenToFlowPosition,
        setEdges,
        setNodes,
        emptyStateElementRef,
        scheduleGroupResize,
        getStyleWidth,
    ]);
    const addNewNodeByEdgeSplitting = React.useCallback((selectedNodeType, values) => {
        const targetEdgeIndex = edges.findIndex(e => e.id === edgeSplittingId.current);
        const newEdges = [...edges];
        const splitEdge = newEdges.splice(targetEdgeIndex, 1)[0];
        const previousNode = nodes.find(n => n.id === splitEdge.source);
        const nextNode = nodes.find(n => n.id === splitEdge.target);
        const currentNodes = [...nodes];
        if (!previousNode || !nextNode) {
            return;
        }
        // Compute absolute midpoint between previous and next (handle children relative positions)
        const computeAbsolutePosition = (node) => computeAbsoluteFlowPosition(node, nodes);
        const prevAbs = computeAbsolutePosition(previousNode);
        const nextAbs = computeAbsolutePosition(nextNode);
        const newNode = createNewNode({
            selectedNodeType,
            values,
            nodes,
            position: {
                x: (prevAbs.x + nextAbs.x) / 2,
                y: (prevAbs.y + nextAbs.y) / 2,
            },
        });
        // Keep new node inside the same group as the source node if present; pre-resize group to fit
        const parentGroupId = previousNode?.parentId;
        const groupedNewNode = newNode;
        const newPreviousEdge = createNewEdge({
            edges,
            targetNodeId: groupedNewNode.id,
            sourceHandleId: splitEdge.sourceHandle,
            sourceNodeId: previousNode.id,
            data: { screenId, elementId },
        });
        const newNextEdge = createNewEdge({
            edges,
            targetNodeId: nextNode.id,
            sourceHandleId: selectedNodeType === 'condition' ? 'out-true' : 'out',
            sourceNodeId: groupedNewNode.id,
            data: { screenId, elementId },
        });
        scheduleGroupResize(parentGroupId);
        setNodes([...currentNodes, groupedNewNode]);
        setEdges([...newEdges, newPreviousEdge, newNextEdge]);
    }, [edges, elementId, nodes, screenId, setEdges, setNodes, scheduleGroupResize]);
    const onNewNodeAdded = React.useCallback(({ selectedNodeType, values }) => {
        if (nodes.length === 0 || (nextNodePosition.current.clientX && nextNodePosition.current.clientY)) {
            addNewNodeByNextPosition(selectedNodeType, values);
        }
        if (edgeSplittingId.current) {
            addNewNodeByEdgeSplitting(selectedNodeType, values);
        }
        setAddDialogFilters(null);
        connectingStartNodeId.current = null;
        edgeSplittingId.current = null;
        nextNodePosition.current = {};
        takeHistorySnapshot();
        if (nodes.length === 0) {
            onTelemetryEvent?.('workflowStartingPointAdded', {
                nodeType: selectedNodeType,
                entityName: values.entityName,
                topic: values.topic,
            });
        }
        else {
            onTelemetryEvent?.('workflowNodeAdded', {
                nodeType: selectedNodeType,
            });
        }
    }, [addNewNodeByEdgeSplitting, addNewNodeByNextPosition, nodes, takeHistorySnapshot, onTelemetryEvent]);
    const workflowContext = React.useMemo(() => ({
        screenId,
        elementId,
        onNodeDataChange,
        onNodeInsertToEdge,
        onNodeBelowNode,
        isReadOnly: isReadOnly || isDisabled,
    }), [screenId, elementId, onNodeDataChange, onNodeInsertToEdge, onNodeBelowNode, isReadOnly, isDisabled]);
    return (React.createElement(React.Fragment, null,
        React.createElement(CarbonWrapper, { ...props, className: "e-workflow-field", componentName: "workflow", noReadOnlySupport: true, value: !!value || false },
            nodes.length === 0 && (React.createElement("div", { className: "e-workflow-empty", ref: emptyStateElementRef },
                React.createElement("div", { className: "e-workflow-empty-content" },
                    React.createElement(Typography, { lineHeight: "30px", variant: "h3" }, localize('@sage/xtrem-ui/workflow-empty', 'This workflow is currently empty')),
                    !isReadOnly && !isDisabled && (React.createElement(Button, { "data-pendoid": "workflow-empty-add-trigger-event", mt: 1, iconType: "add", onClick: onAddFirstElement, "data-testid": "add-item-button" }, localize('@sage/xtrem-ui/workflow-add-trigger-event', 'Add a trigger event')))))),
            nodes.length !== 0 && (React.createElement("div", { className: "e-workflow-wrapper", ref: componentRef },
                React.createElement(WorkflowContext.Provider, { value: workflowContext },
                    React.createElement(ReactFlow, { nodesDraggable: !isReadOnly && !isDisabled, nodesConnectable: !isReadOnly && !isDisabled, edgesFocusable: !isReadOnly && !isDisabled, nodesFocusable: !isReadOnly && !isDisabled, nodes: adjustedNodes, edges: renderEdges, snapToGrid: true, snapGrid: [20, 20], onConnectStart: onConnectStart, onConnectEnd: onConnectEnd, onNodesChange: onNodesChangeInternal, onEdgesChange: onEdgesChangeInternal, nodeTypes: nodeTypes, edgeTypes: edgeTypes, deleteKeyCode: ['Backspace', 'Delete'], isValidConnection: isValidConnection },
                        React.createElement(Background, { color: "#004455" }),
                        React.createElement(WorkflowControlButtons, { zoomIn: zoomIn, zoomOut: zoomOut, fitView: fitView, onExpand: onExpand, onMinimize: onMinimize, onUndo: onUndo, onRedo: onRedo, isUndoDisabled: isUndoDisabled, isRedoDisabled: isRedoDisabled, expansionDialogControl: expansionDialogControl, localize: localize })))))),
        React.createElement(AddWorkflowNodeDialog, { isOpen: !!addDialogFilters, onClose: onAddNodeDialogClose, onConfirm: onNewNodeAdded, elementId: props.elementId, screenId: props.screenId, previousNodeId: connectingStartNodeId.current?.nodeId, filterType: addDialogFilters || [] })));
}
export const ConnectedFormDesignerComponent = connect(mapStateToProps(), mapDispatchToProps())(WorkflowComponent);
export default ConnectedFormDesignerComponent;
//# sourceMappingURL=workflow-component.js.map