import { remove, uniq, uniqBy } from 'lodash';
import { MarkerType } from 'reactflow';
import { localize } from '../../../service/i18n-service';
import { triggerFieldEvent } from '../../../utils/events';
import { useFieldValue } from '../../../utils/hooks/effects/use-set-field-value';
import { computeAbsoluteFlowPosition, LayoutDefaults } from './workflow-layout-utils';
export const edgeStyle = {
    style: {
        strokeWidth: 1,
        stroke: '#A6A6A6',
    },
    markerEnd: {
        type: MarkerType.Arrow,
        width: 12,
        height: 12,
        color: '#A6A6A6',
    },
};
export const removeTransientNodeDataProperties = (data) => {
    const cleanData = { ...data };
    delete cleanData.type;
    return cleanData;
};
export const removeTransientNodeDataPropertiesFromNodes = (nodes) => nodes.map(n => {
    const base = {
        data: removeTransientNodeDataProperties(n.data),
        height: n.height,
        id: n.id,
        position: n.position,
        type: n.type,
        width: n.width,
        style: n.style,
        // Preserve hierarchical structure properties for subflow nodes
        ...(n.parentId && { parentId: n.parentId }),
        ...(n.extent && { extent: n.extent }),
    };
    return base;
});
export const removeTransientEdgeDataPropertiesFromEdges = (nodes) => nodes.map(e => {
    const edgeCopy = { ...e };
    delete edgeCopy.markerEnd;
    delete edgeCopy.data;
    delete edgeCopy.style;
    delete edgeCopy.selected;
    return edgeCopy;
});
export const changeEventHandler = (screenId, elementId) => () => triggerFieldEvent(screenId, elementId, 'onChange');
/**
 * Retrieve all previous nodes in the workflow, following all paths back to the start node.
 */
export const usePreviousWorkflowNodes = (nodeId, screenId, elementId, includeCurrent = false) => {
    const fieldValue = useFieldValue(screenId, elementId);
    if (!fieldValue || !nodeId) {
        return [];
    }
    // Helpers to avoid function declarations inside the loop (no-loop-func)
    const findEdgesByTarget = (targetId) => fieldValue.edges.filter(e => e.target === targetId);
    const findNodeById = (id) => fieldValue.nodes.find(n => n.id === id);
    const node = findNodeById(nodeId);
    const previousNodes = [];
    const visitedNodeIds = {};
    if (includeCurrent && node) {
        previousNodes.push(node);
        visitedNodeIds[node.id] = true;
    }
    let watchDog = 0;
    const nodeIdsToInspect = findEdgesByTarget(nodeId).map(e => e.source);
    while (nodeIdsToInspect.length > 0) {
        watchDog += 1;
        if (watchDog > 1000) {
            // Safety exit to avoid infinite loops
            // Should never happen in a valid workflow
            // eslint-disable-next-line no-console
            console.warn('usePreviousWorkflowNodes: safety exit to avoid infinite loop');
            break;
        }
        const currentNodeId = nodeIdsToInspect.shift();
        if (!currentNodeId)
            break;
        if (visitedNodeIds[currentNodeId]) {
            // This node was already processed (cycle or diamond topology)
            // eslint-disable-next-line no-continue
            continue;
        }
        const currentNode = findNodeById(currentNodeId);
        if (!currentNode) {
            // Should never happen but ...
            // eslint-disable-next-line no-continue
            continue;
        }
        previousNodes.push(currentNode);
        visitedNodeIds[currentNodeId] = true;
        // Add all its predecessors to the list of nodes to inspect
        nodeIdsToInspect.push(...findEdgesByTarget(currentNodeId)
            .map(e => e.source)
            .filter(id => !visitedNodeIds[id]));
    }
    return previousNodes.reverse();
};
export const useSubsequentWorkflowNodes = (nodeId, screenId, elementId, includeCurrent = false) => {
    const fieldValue = useFieldValue(screenId, elementId);
    if (!fieldValue || !nodeId) {
        return [];
    }
    const currentNode = fieldValue.nodes.find(n => n.id === nodeId);
    const subsequentNodes = [];
    let currentNodeId = nodeId;
    if (includeCurrent && currentNode) {
        subsequentNodes.push(currentNode);
    }
    // Helper to avoid function declarations inside the loop (no-loop-func)
    const findEdgeBySource = (sourceId) => fieldValue.edges.find(e => e.source === sourceId);
    const findNodeById = (id) => fieldValue.nodes.find(n => n.id === id);
    while (true) {
        const nextEdge = findEdgeBySource(currentNodeId);
        if (!nextEdge) {
            break;
        }
        const nextNodeId = nextEdge.target;
        if (!nextNodeId) {
            break;
        }
        const node = findNodeById(nextNodeId);
        if (!node) {
            break;
        }
        // If the node is already in the list, we have a circular reference
        if (subsequentNodes.some(n => n.id === node.id)) {
            break;
        }
        subsequentNodes.push(node);
        currentNodeId = nextNodeId;
    }
    return subsequentNodes.reverse();
};
export const useSourceNode = (edgeId, screenId, elementId) => {
    const fieldValue = useFieldValue(screenId, elementId);
    if (!fieldValue) {
        return null;
    }
    const edge = fieldValue?.edges.find(e => e.id === edgeId);
    if (!edge) {
        return null;
    }
    return fieldValue.nodes.find(n => n.id === edge.source) || null;
};
export const hasEdgeConnectedToNode = (nodeId, edges, sourceHandle) => {
    const connectedEdges = edges.filter(e => e.source === nodeId || e.target === nodeId);
    if (sourceHandle) {
        return connectedEdges.some(e => e.sourceHandle === sourceHandle);
    }
    return connectedEdges.length > 0;
};
export const useWorkflowNodeVariables = (nodeId, screenId, elementId, includeCurrent = false) => {
    const previousNodes = usePreviousWorkflowNodes(nodeId, screenId, elementId, includeCurrent);
    const collected = { inputVariables: [], oldRootPaths: [] };
    const outerContexts = [];
    removeTransientNodeDataPropertiesFromNodes(previousNodes).forEach((node) => {
        if (node.type === 'endRepeat') {
            // End of a repeat: stop adding its innerVariables to following nodes
            outerContexts.pop();
        }
        const stepVariablesToAdd = node.data.stepVariables == null ? [] : [...node.data.stepVariables];
        if (node.type?.startsWith('repeat/')) {
            // The repeat action contains 2 kinds of stepVariables:
            // - those selected by the user (for the source as instance): can be used in any steps (even steps outside the repeat)
            // - those generated from its outputVariableName: cannot only be used in inner steps
            const pathsExcludedForInnerSteps = stepVariablesToAdd
                .filter(v => v.path.startsWith(`${node.data.outputVariableName}.`))
                .map(v => v.path);
            outerContexts.push({
                innerVariables: node.data.innerVariables ?? [],
                pathsExcludedForInnerSteps,
            });
        }
        collected.inputVariables.push(...stepVariablesToAdd);
        const outputVariables = node.data.outputVariables;
        if (outputVariables && outputVariables.length > 0) {
            // Compatibility code: formerly the step could have an array of outputVariables
            collected.inputVariables.push(...outputVariables);
        }
        delete node.data.outputVariables;
        if (node.data.oldRootPaths) {
            collected.oldRootPaths.push(...node.data.oldRootPaths);
        }
    });
    if (outerContexts.length > 0) {
        // We are inside (at least) one 'repeat' action
        // We have to remove all the excluded paths from the remaining outer contexts
        // outerContexts.length is the number of nested 'repeat' actions we are in
        outerContexts.forEach(innerVariables => {
            collected.inputVariables.push(...innerVariables.innerVariables);
        });
        // Remove all the excluded paths from the remaining outer contexts
        const pathsExcludedForInnerSteps = uniq(outerContexts.flatMap(c => c.pathsExcludedForInnerSteps));
        collected.inputVariables = collected.inputVariables.filter(v => !pathsExcludedForInnerSteps.includes(v.path));
    }
    return {
        inputVariables: uniqBy(collected.inputVariables, 'path'),
        oldRootPaths: uniq(collected.oldRootPaths),
    };
};
export function removeChildNodesAndEdgesFromStartPoint(nodeId, tempEdges, tempNodes) {
    // Create a snapshot of the original nodes for lookup
    const originalNodes = [...tempNodes];
    const originalEdges = [...tempEdges];
    const nodeToRemove = originalNodes.find(n => n.id === nodeId);
    if (!nodeToRemove) {
        return;
    }
    // Collect all IDs to remove in a set for efficient lookup
    const idsToRemove = new Set([nodeId]);
    // Iteratively collect nodes to remove until no more changes
    let changed = true;
    while (changed) {
        changed = false;
        for (const currentId of Array.from(idsToRemove)) {
            const currentNode = originalNodes.find(n => n.id === currentId);
            if (currentNode) {
                // If this is a repeat node, also collect its group and endRepeat
                const nodeType = currentNode.type?.split('/')?.[0];
                if (nodeType === 'repeat') {
                    const groupId = currentNode.data?.groupId ||
                        originalNodes.find(g => g.type === 'group' && g.data?.anchoredRepeatId === currentId)?.id;
                    if (groupId && !idsToRemove.has(groupId)) {
                        idsToRemove.add(groupId);
                        changed = true;
                    }
                }
                // If this is a group, also collect its anchored repeat and endRepeat
                if (currentNode.type === 'group') {
                    const anchoredRepeatId = currentNode.data?.anchoredRepeatId;
                    const anchoredEndRepeatId = currentNode.data?.anchoredEndRepeatId;
                    if (anchoredRepeatId && !idsToRemove.has(anchoredRepeatId)) {
                        idsToRemove.add(anchoredRepeatId);
                        changed = true;
                    }
                    if (anchoredEndRepeatId && !idsToRemove.has(anchoredEndRepeatId)) {
                        idsToRemove.add(anchoredEndRepeatId);
                        changed = true;
                    }
                }
                // Collect all child nodes (nodes with parentId pointing to this node)
                const childNodes = originalNodes.filter(n => n.parentId === currentId && !idsToRemove.has(n.id));
                for (const child of childNodes) {
                    idsToRemove.add(child.id);
                    changed = true;
                }
            }
        }
        // Collect target nodes from edges where source nodes are being removed
        // Only if the target has no incoming edges from nodes that will remain
        for (const node of originalNodes) {
            if (!idsToRemove.has(node.id)) {
                const incomingEdges = originalEdges.filter(e => e.target === node.id);
                if (incomingEdges.length > 0) {
                    const incomingEdgesFromRemainingNodes = incomingEdges.filter(e => !idsToRemove.has(e.source));
                    if (incomingEdgesFromRemainingNodes.length === 0) {
                        idsToRemove.add(node.id);
                        changed = true;
                    }
                }
            }
        }
    }
    // Remove all collected nodes
    remove(tempNodes, n => idsToRemove.has(n.id));
    // Remove all edges connected to removed nodes
    remove(tempEdges, e => idsToRemove.has(e.source) || idsToRemove.has(e.target));
}
export function createNewNode({ selectedNodeType, values, nodes, position = { x: 20, y: 20 }, }) {
    return {
        id: allocateId(selectedNodeType, nodes),
        position,
        type: selectedNodeType,
        data: {
            type: selectedNodeType,
            ...values,
        },
    };
}
export function createNewEdge({ sourceNodeId, sourceHandleId, edges, targetNodeId, data, }) {
    return {
        id: allocateId(`${sourceNodeId}--${sourceHandleId}`, edges),
        source: sourceNodeId,
        sourceHandle: sourceHandleId,
        target: targetNodeId,
        data,
        ...edgeStyle,
    };
}
export function allocateId(prefix, items) {
    for (let i = 1; i < 1000; i += 1) {
        const id = `${prefix}-${i}`;
        if (!items.find(n => n.id === id))
            return id;
    }
    throw new Error('Could not find a unique id');
}
export function isSingleEdgeRemoval(change, edges) {
    const removalChange = change.find(c => c.type === 'remove');
    if (!removalChange) {
        return false;
    }
    const edgeToRemove = edges.find(e => e.id === removalChange.id);
    const edgesTargetingNode = edgeToRemove ? edges.filter(e => e.target === edgeToRemove.target) : [];
    return edgesTargetingNode.length < 2;
}
export function roundToNearestTwenty(num) {
    return Math.round(num / 20) * 20;
}
/**
 * Creates the group, anchored repeat, and end repeat nodes for a repeat node.
 * Also links the group with the anchored repeat and the anchored repeat with the end repeat.
 * Also sets the branch flags for the source node if it is a condition node.
 * Also schedules a group resize for the parent group if this is a nested group.
 */
export function createRepeatNodeStructure({ newNode, nodes, edges, position, connectingStartNodeId, screenId, elementId, getStyleWidth, scheduleGroupResize, }) {
    // Calculate group position so that the repeat sits half-overlapped above and centered
    const groupWidthDefault = LayoutDefaults.group.defaultWidth;
    const groupHeightDefault = LayoutDefaults.group.defaultHeight;
    const repeatWidthDefault = LayoutDefaults.node.defaultWidth;
    const repeatHeightDefault = LayoutDefaults.node.defaultHeight;
    // If we started a connection from a node that is inside a group, nest the new repeat group
    const sourceNode = connectingStartNodeId ? nodes.find(n => n.id === connectingStartNodeId.nodeId) || null : null;
    const parentGroupId = sourceNode ? sourceNode.data?.groupId || sourceNode.parentId : undefined;
    let groupFlowPos;
    if (!parentGroupId) {
        // When adding below a repeat, center using the anchored group's width if available
        let refLeftAbs = position.x;
        let refWidth = repeatWidthDefault;
        const isSourceRepeat = (sourceNode?.type?.split('/')?.[0] || '') === 'repeat';
        if (isSourceRepeat) {
            const anchoredGroupId = sourceNode?.data?.groupId;
            const anchoredGroup = anchoredGroupId ? nodes.find(n => n.id === anchoredGroupId) : null;
            if (anchoredGroup) {
                const abs = computeAbsoluteFlowPosition(anchoredGroup, nodes);
                refLeftAbs = abs.x;
                refWidth = getStyleWidth(anchoredGroup, anchoredGroup.width ?? refWidth);
            }
        }
        groupFlowPos = {
            x: Math.round(refLeftAbs + Math.round(refWidth / 2) - Math.round(groupWidthDefault / 2)),
            // Anchor group so that repeat sits 50% inside; add vertical padding as top margin
            y: Math.round(position.y + Math.round(repeatHeightDefault / 2)),
        };
    }
    else {
        const parentGroup = nodes.find(n => n.id === parentGroupId);
        // position is absolute; convert to relative to parent group (use parent's ABSOLUTE position for nested groups)
        const parentAbs = computeAbsoluteFlowPosition(parentGroup, nodes);
        // Also, if the source is a repeat anchored to this same parent group, center by the parent's width
        let relativeX;
        const isSourceRepeat = (sourceNode?.type?.split('/')?.[0] || '') === 'repeat';
        if (isSourceRepeat) {
            const parentWidth = getStyleWidth(parentGroup, parentGroup.width ?? groupWidthDefault);
            // Center the new group within the parent group
            relativeX = Math.round(Math.round(parentWidth / 2) - Math.round(groupWidthDefault / 2));
        }
        else {
            relativeX = Math.round(position.x - parentAbs.x - Math.round((groupWidthDefault - repeatWidthDefault) / 2));
        }
        groupFlowPos = {
            x: relativeX,
            // Same vertical logic for nested groups
            y: Math.round(position.y - parentAbs.y + Math.round(repeatHeightDefault / 2)),
        };
    }
    const groupNodeId = allocateId('group', nodes);
    const groupNode = {
        id: groupNodeId,
        position: groupFlowPos,
        type: 'group',
        selectable: false,
        data: { type: 'repeat' },
        style: { width: groupWidthDefault, height: groupHeightDefault },
        draggable: false,
        ...(parentGroupId ? { parentId: parentGroupId, extent: 'parent' } : {}),
    };
    // Anchor the repeat to the top edge of the group (50% inside, 50% outside)
    const anchoredRepeatPos = {
        x: groupFlowPos.x + Math.round((groupWidthDefault - repeatWidthDefault) / 2),
        y: groupFlowPos.y - Math.round(repeatHeightDefault / 2),
    };
    const anchoredRepeatNode = {
        ...newNode,
        // Keep original node.type and data; only augment with groupId and positioning
        position: anchoredRepeatPos,
        data: {
            ...newNode.data,
            groupId: groupNodeId,
        },
        style: { ...(newNode.style || {}), zIndex: 1000 },
        ...(parentGroupId ? { parentId: parentGroupId, extent: 'parent' } : {}),
    };
    // Link group with anchored repeat so we can keep it attached on group move
    groupNode.data = {
        ...groupNode.data,
        anchoredRepeatId: anchoredRepeatNode.id,
    };
    // End repeat node centered at group bottom, 50% inside/50% outside
    const endRepeatNodeId = allocateId('endRepeat', nodes);
    const endRepeatNode = {
        id: endRepeatNodeId,
        type: 'endRepeat',
        draggable: false,
        position: {
            x: groupFlowPos.x + Math.round((groupWidthDefault - 120) / 2),
            y: groupFlowPos.y + groupHeightDefault - Math.round(32 / 2),
        },
        data: {
            type: 'endRepeat',
            title: localize('@sage/xtrem-ui/workflow-end-repeat-title', 'End repeat'),
        },
    };
    if (parentGroupId) {
        endRepeatNode.parentId = parentGroupId;
        endRepeatNode.extent = 'parent';
    }
    // Link group with anchored End repeat
    groupNode.data = {
        ...groupNode.data,
        anchoredEndRepeatId: endRepeatNodeId,
    };
    // Prepare edges and branch flags
    const newEdges = [...edges];
    const updatedNodes = [...nodes];
    if (connectingStartNodeId && ['out-true', 'out-false'].includes(connectingStartNodeId.handleId)) {
        const branchKey = connectingStartNodeId.handleId === 'out-true' ? 'ifTrueBranch' : 'ifFalseBranch';
        const index = updatedNodes.findIndex(n => n.id === connectingStartNodeId.nodeId);
        if (index >= 0) {
            updatedNodes[index] = {
                ...updatedNodes[index],
                data: { ...updatedNodes[index].data, [branchKey]: true },
            };
        }
    }
    // Link source to the anchored repeat for any handle if there is a connection start
    if (connectingStartNodeId) {
        newEdges.push(createNewEdge({
            edges,
            targetNodeId: anchoredRepeatNode.id,
            sourceHandleId: connectingStartNodeId.handleId,
            sourceNodeId: connectingStartNodeId.nodeId,
            data: { screenId, elementId },
        }));
    }
    // Always link repeat's regular 'out' to the End repeat (invisible edge)
    newEdges.push(createNewEdge({
        edges,
        sourceNodeId: anchoredRepeatNode.id,
        sourceHandleId: 'out',
        targetNodeId: endRepeatNodeId,
        data: { screenId, elementId },
    }));
    // Schedule resize for parent group if this is a nested group
    if (parentGroupId) {
        scheduleGroupResize(parentGroupId);
    }
    const resultNodes = [...updatedNodes, groupNode, anchoredRepeatNode, endRepeatNode];
    return {
        nodes: resultNodes,
        edges: newEdges,
        updatedNodes,
    };
}
//# sourceMappingURL=workflow-component-utils.js.map